// uTaskerCombine.cpp : Defines the entry point for the console application.
//
/*
Version V1.2 20-09-2012
=======================

"uTaskerCombine -v" to get the version number.

Utility used to combine two binary files into a single file - often used to combine a boot loader with application
code to result in a single downloadable file.

Use "uTaskerCombine file1.bin file2.bin 0x1000 file3.bin"

Adds the content of file 2 at an offset of 0x1000 to the content of file2 (filling and space between the two with 0xff)
and saves the result to file3.bin.

Option = "uTaskerCombine file1.bin file2.bin 0x1000 file3.bin file4.ihex"

Adding a further file causes the resulting binary to be also saved in file4.ihex in Intel hex format. This is mainly
used for NXP projects where the program FlashMagic is used to program the combined binary file, but requires the code
to be in this format.

Option = "uTaskerCombine file1.bin file2.bin 0x1000 file3.bin file4.srec"

If the optional hex file has an extension .srec or .out it is considered as requiring Motorola S-REC instead.

Option = "uTaskerCombine file1.bin file2.bin 0x1000 file3.bin file4.srec 0x80000000"

An address following the hex file output is interpreted as an offset to be added to the content of the hex file - for example
the AVR32 has its code in internal flash starting at 0x80000000 and so requires thsi offset to load to the correct address.

If the exact format of the srec file is to be controlled it can be done so by specifying S1, S2 or S3 after the file name
(before any optional offset) eg. "uTaskerCombine file1.bin file2.bin 0x1000 file3.bin file4.srec S3 0x80000000"


Version V1.3 21-01-2016
=======================

Allow concatenating two binary files by defining 0 offset

Version V1.4 19-04-2016
=======================

Correct optional offset when building with VS2010, respecting that the input is now in unicode

Version V1.5 31-07-2017
=======================

Terminate iHex line with CR + LF (rather than just LF)

Version V1.6 16-03-2020
=======================

Correctly recognise srec/ihex files in unicode

Version V1.7 03-07-2020
=======================

Add option to combine with file object headers

Version V1.8 13-07-2020
=======================

Remove complete file path when embedding file objects
Ensure iHex extended record is terminated with CR and LF

*/

#include "stdafx.h"
#include "stdio.h"
#include <string.h>
#include "conio.h"
#include "Fcntl.h"
#include "io.h"
#include <sys/stat.h>
#include <sys/>
#include <sys/stat.h>
#include <share.h>
  

// Local functions
//
static unsigned long fnGetAddress(char *, unsigned long *);
static void fnIntelHex(int File, unsigned char *content, unsigned long ulLength, unsigned long ulOffset);
static void fnMotorolaHex(int File, unsigned char *content, unsigned long ulLength, unsigned long ulOffset, char cType);
static int fnCheckType(char *ptrFileName);
static int fnAddFileObject(unsigned char *ptrBuffer, unsigned long ulFileObjectLength, const wchar_t *file_name, const wchar_t *disk_name);


#define MAX_MEMORY (2*1024*1024)                                         // max 2M memory
static unsigned char ucApplication[MAX_MEMORY];                                 

#define MOTOROLA_SREC   0
#define INTEL_HEX       1


/*/////////////////////////////////////////
#include <stdio.h>
#include <math.h>
#define PI 3.14159265
////////////////////////////////////////*/

extern int _tmain(int argc, char* argv[])
{
	int theFile;
	int refFile;
    int addFile;
    int iRtn = 0;
    unsigned long ulFileLength;
    unsigned long ulCheck;
    unsigned short ucCheckSum = 0;
    int iExtraCodeLength = 0; 
    unsigned long ulMergeTo;
    unsigned long ulOptionalFileObjectLength = 0;

    
    /*///////////////////////////////////////////////////////////////////// program temporarily mis-used to generate a sine table
            int HexFile = _open("sine.c",  (_O_TRUNC  | _O_CREAT | _O_WRONLY), _S_IWRITE);
            #define NUMBER_OF_SAMPLES 200
            int iSize = 0;
            signed short value;
            double degrees = 0;
            double sine;
            unsigned char temp;
            char svalue[8];
            svalue[0] = '0';
            svalue[1] = 'x';
            svalue[6] = ',';
            svalue[7] = 0;
            for (iSize = 0; iSize < NUMBER_OF_SAMPLES; iSize++) {
                sine = sin (degrees*PI/180);
                degrees += (double)((double)360/(double)NUMBER_OF_SAMPLES);
                value = (signed short)(sine * 1023);
                temp = (unsigned char)((value >> 12) & 0x0f);
                temp += '0';
                if (temp > '9') {
                    temp += ('A' - '9' - 1);
                }
                svalue[2] = temp;

                temp = (unsigned char)((value >> 8) & 0x0f);
                temp += '0';
                if (temp > '9') {
                    temp += ('A' - '9' - 1);
                }
                svalue[3] = temp;

                temp = (unsigned char)((value >> 4) & 0x0f);
                temp += '0';
                if (temp > '9') {
                    temp += ('A' - '9' - 1);
                }
                svalue[4] = temp;

                temp = (unsigned char)((value) & 0x0f);
                temp += '0';
                if (temp > '9') {
                    temp += ('A' - '9' - 1);
                }
                svalue[5] = temp;
                _write(HexFile, "(signed short)", 14);
                _write(HexFile, svalue, 7);
            }
            _close(HexFile);
    /////////////////////////////////////////////////////////////////////*/

    printf("uTaskerCombine V1.8\n\n");

	if (argv[1] == 0) {
		return 0;
	}
    if ((argv[1][0] == '-') && ((argv[1][2] == 'v') || ((argv[1][2] == 'V')))) { // just display version number
        return 0;
    }

	if ((iRtn = _wsopen_s(&refFile, (const wchar_t *)argv[1], (_O_BINARY | _O_RDONLY), _SH_DENYRD, _S_IREAD)) != 0) { // the boot code
        printf("Input file 1 could not be found. Terminating\n\n");
        return (1);
    }
	if ((iRtn = _wsopen_s(&addFile, (const wchar_t *)argv[2], (_O_BINARY | _O_RDONLY), _SH_DENYRD, _S_IREAD)) != 0) { // the first application code
        printf("Input file 2 could not be found. Terminating\n\n");
        return (1);
    }


    ulMergeTo = fnGetAddress(argv[3], &ulOptionalFileObjectLength);      // optional offset address of the second file

	if ((iRtn = _wsopen_s(&theFile, (const wchar_t *)argv[4], (_O_BINARY | _O_RDWR | _O_CREAT | _O_TRUNC), _SH_DENYRD, _S_IWRITE)) != 0) { // the output file
        printf("Output file could not be created. Terminating\n\n");
        return (1);
    }

    ulFileLength = _read(refFile, ucApplication, MAX_MEMORY);            // read the first file content to memory
    if (ulMergeTo == 0) {
        ulMergeTo = ulFileLength;
    }
    if (ulFileLength > (ulMergeTo + ulOptionalFileObjectLength)) {
        printf("Boot code too long (> %x). Terminating\n\n", ulMergeTo);
        return (1);
    }

    memset(&ucApplication[ulFileLength], 0xff, (ulMergeTo - ulFileLength)); // pad with 0xff up to application start

    if (ulOptionalFileObjectLength != 0) {
        fnAddFileObject(&ucApplication[ulMergeTo], ulOptionalFileObjectLength, (const wchar_t *)argv[2], (const wchar_t *)argv[3]); // insert a file object
        ulMergeTo += ulOptionalFileObjectLength;
    }

    ulFileLength = _read(addFile, &ucApplication[ulMergeTo], (MAX_MEMORY - ulMergeTo)); // add the application

    ulCheck = _write(theFile, ucApplication, (ulFileLength + ulMergeTo));
    if (ulCheck != (ulFileLength + ulMergeTo)) {
	    iRtn = 1;
    }
    if (argv[5] != 0) {                                                  // either intel hex file or srec
        int HexFile;
	    if ((iRtn = _wsopen_s(&HexFile, (const wchar_t *)argv[5], (_O_BINARY | _O_RDWR | _O_CREAT | _O_TRUNC), _SH_DENYRD, _S_IWRITE)) != 0) { // the first application code
            printf("Hex file could not be generated!!\n\n");
            return (1);
        }
        else {
            unsigned long ulOffset = 0;
            char cType = '1';                                            // allow S1 as smallest type
            if (argv[6] != 0) {                                          // offset
                if ((*argv[6] == 'S') || (*argv[6] == 's')) {            // specifying a particular srec type
                    cType = *(argv[6] + 1);                              // the srec type
                    if (argv[7] != 0) {
                        ulOffset = fnGetAddress(argv[7], 0);
                    }
                }
                else {
                    ulOffset = fnGetAddress(argv[6], 0);
                }
            }
            if (fnCheckType(argv[5]) == MOTOROLA_SREC) {
                fnMotorolaHex(HexFile, ucApplication, (ulFileLength + ulMergeTo), ulOffset, cType);
            }
            else {                                                       // default is intel hex
                fnIntelHex(HexFile, ucApplication, (ulFileLength + ulMergeTo), ulOffset);
            }
            _close(HexFile);
        }
    }
    _close(addFile);
    _close(refFile);
    _close(theFile);
	return (iRtn);
}

typedef struct stDIR_ENTRY_STRUCTURE_FAT32
{
    unsigned char DIR_Name[11];                                          // directory short name. If the first byte is 0xe5 the directory entry is free. If it is 0x00 this and all following are free. If it is 0x05 it means that the actual file name begins with 0xe5 (makes Japanese character set possible). May not start with ' ' or lower (apart from special case for 0x05) and lower case characters are not allowed. The following characters are not allowed: "	0x22, 0x2A, 0x2B, 0x2C, 0x2E, 0x2F, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x5B, 0x5C, 0x5D, and 0x7C
    unsigned char DIR_Attr;                                              // file attributes
    unsigned char DIR_NTRes;                                             // reserved for Windows NT - should be 0
    unsigned char DIR_CrtTimeTenth;                                      // millisecond stamp at file creation time. Actually contains a count of tenths of a second 0..199
    unsigned char DIR_CrtTime[2];                                        // time file was created
    unsigned char DIR_CrtDate[2];                                        // data file was created
    unsigned char DIR_LstAccDate[2];                                     // last access date (read or write), set to same as DIR_WrtDate on write
    unsigned char DIR_FstClusHI[2];                                      // high word of this entry's first cluster number (always 0 for a FAT12 or FAT16 volume)
    unsigned char DIR_WrtTime[2];                                        // time of last write, whereby a file creation is considered as a write
    unsigned char DIR_WrtDate[2];                                        // date of last write, whereby a file creation is considered as a write
    unsigned char DIR_FstClusLO[2];                                      // low word of this entry's first cluster number
    unsigned char DIR_FileSize[4];                                       // file's size in bytes
} DIR_ENTRY_STRUCTURE_FAT32;

typedef struct stFILE_OBJECT_INFO
{
    unsigned long  file_length;
    unsigned short usCreationDate;
    unsigned short usCreationTime;
    unsigned short usModifiedDate;
    unsigned short usModifiedTime;
    unsigned short usAccessedDate;
    const char     *ptrShortFileName;
} FILE_OBJECT_INFO;

typedef struct stLFN_ENTRY_STRUCTURE_FAT32
{
    unsigned char LFN_EntryNumber;                                       // entry number starting from last - 0x40 is always set in the first entry and the value decrements until 1
    unsigned char LFN_Name_0;                                            // first letter
    unsigned char LFN_Name_0_extension;                                  // first letter extension - is always 0 in English character set
    unsigned char LFN_Name_1;                                            // second letter
    unsigned char LFN_Name_1_extension;                                  // second letter extension - is always 0 in English character set
    unsigned char LFN_Name_2;                                            // third letter
    unsigned char LFN_Name_2_extension;                                  // third letter extension - is always 0 in English character set
    unsigned char LFN_Name_3;                                            // fourth letter
    unsigned char LFN_Name_3_extension;                                  // fourth letter extension - is always 0 in English character set
    unsigned char LFN_Name_4;                                            // fifth letter
    unsigned char LFN_Name_4_extension;                                  // fifth letter extension - is always 0 in English character set
    unsigned char LFN_Attribute;                                         // always 0x0f
    unsigned char LFN_Zero0;                                             // always zero
    unsigned char LFN_Checksum;                                          // check sum
    unsigned char LFN_Name_5;                                            // sixth letter
    unsigned char LFN_Name_5_extension;                                  // sixth letter extension - is always 0 in English character set
    unsigned char LFN_Name_6;                                            // seventh letter
    unsigned char LFN_Name_6_extension;                                  // seventh letter extension - is always 0 in English character set
    unsigned char LFN_Name_7;                                            // eighth letter
    unsigned char LFN_Name_7_extension;                                  // eighth letter extension - is always 0 in English character set
    unsigned char LFN_Name_8;                                            // ninth letter
    unsigned char LFN_Name_8_extension;                                  // ninth letter extension - is always 0 in English character set
    unsigned char LFN_Name_9;                                            // tenth letter
    unsigned char LFN_Name_9_extension;                                  // tenth letter extension - is always 0 in English character set
    unsigned char LFN_Name_10;                                           // eleventh letter
    unsigned char LFN_Name_10_extension;                                 // eleventh letter extension - is always 0 in English character set
    unsigned char LFN_Zero1;                                             // always zero
    unsigned char LFN_Zero2;                                             // always zero
    unsigned char LFN_Name_11;                                           // twelfth letter
    unsigned char LFN_Name_11_extension;                                 // twelfth letter extension - is always 0 in English character set
    unsigned char LFN_Name_12;                                           // thirteenth letter
    unsigned char LFN_Name_12_extension;                                 // thirteenth letter extension - is always 0 in English character set
} LFN_ENTRY_STRUCTURE_FAT32;

#define DIR_ATTR_VOLUME_ID               0x08
#define DIR_ATTR_ARCHIVE                 0x20

#include <stdlib.h>
#include <time.h>

static void fnSetObjectDetails(DIR_ENTRY_STRUCTURE_FAT32 *ptrEntry, FILE_OBJECT_INFO *ptrFileObjectInfo)
{
    unsigned long ulFileLength = (unsigned long)(ptrFileObjectInfo->file_length); // the length of file that has been saved
    ptrEntry->DIR_CrtTime[0] = (unsigned char)(ptrFileObjectInfo->usCreationTime);
    ptrEntry->DIR_CrtTime[1] = (unsigned char)(ptrFileObjectInfo->usCreationTime >> 8);
    ptrEntry->DIR_CrtDate[0] = (unsigned char)(ptrFileObjectInfo->usCreationDate);
    ptrEntry->DIR_CrtDate[1] = (unsigned char)(ptrFileObjectInfo->usCreationDate >> 8);
    ptrEntry->DIR_LstAccDate[0] = (unsigned char)(ptrFileObjectInfo->usAccessedDate);
    ptrEntry->DIR_LstAccDate[1] = (unsigned char)(ptrFileObjectInfo->usAccessedDate >> 8);
    ptrEntry->DIR_WrtDate[0] = (unsigned char)(ptrFileObjectInfo->usModifiedDate);
    ptrEntry->DIR_WrtDate[1] = (unsigned char)(ptrFileObjectInfo->usModifiedDate >> 8);
    ptrEntry->DIR_WrtTime[0] = (unsigned char)(ptrFileObjectInfo->usModifiedTime);
    ptrEntry->DIR_WrtTime[1] = (unsigned char)(ptrFileObjectInfo->usModifiedTime >> 8);
    ptrEntry->DIR_FileSize[0] = (unsigned char)(ulFileLength);           // enter the file size
    ptrEntry->DIR_FileSize[1] = (unsigned char)(ulFileLength >> 8);
    ptrEntry->DIR_FileSize[2] = (unsigned char)(ulFileLength >> 16);
    ptrEntry->DIR_FileSize[3] = (unsigned char)(ulFileLength >> 24);
}

// Create a short file name alisas that does not represent a valid file name and minimises the risk
// of causing problems with older systems supporting long file names and causing collision in a directory
// This implementation is based on the original algorithm used by VFAT which can be found at https://lkml.org/lkml/2009/6/26/313.
// All credits go to the original author, whos comments are retained
//
static void fnCreateInvalidSFN_alias(char cInvalidSFN_alias[12])          // the length includes the NT byte
{
    unsigned long ulRandom;
    memset(cInvalidSFN_alias, 0, 12);                                    // set content to zero including NT byte

   /* we start with a space followed by nul as spaces at the
    * start of an entry are trimmed in FAT, which means that
    * starting the 11 bytes with 0x20 0x00 gives us a value which
    * cannot be used to access the file. It also means that the
    * value as seen from all Windows and Linux APIs is a constant
    */
    cInvalidSFN_alias[0] = ' ';

    /* we use / and 2 nul bytes for the extension. These are
     * invalid in FAT and mean that utilities that show the
     * directory show no extension, but still work via the long
     * name for old Linux kernels
     */
    cInvalidSFN_alias[8] = '/';

    /*
     * fill the remaining 6 bytes with random invalid values
     * This gives us a low collision rate, which means a low
     * chance of problems with chkdsk.exe and WindowsXP
     */
    do {
        ulRandom = ((unsigned short)rand() | ((unsigned short)rand() << 16));
        ulRandom &= 0x1f1f1f1f;
    } while (ulRandom == 0);                                             // avoid 0 as explained below
    memcpy(&cInvalidSFN_alias[2], &ulRandom, 4);                         // fill out [2..5] with random values
    ulRandom = (rand() & 0x1f1f);
    memcpy(&cInvalidSFN_alias[6], &ulRandom, 2);                         // fill out [6..7] with random values

   /* a value of zero would leave us with only nul and spaces,
    * which would not work with older linux systems
    */
}

// Calculate the checkum of a short file name alias
//
static unsigned char fnLFN_checksum(char *cPtrSFN_alias)
{
    int i;
    unsigned char ucCheckSum = 0;

    for (i = 11; i; i--) {
        ucCheckSum = ((ucCheckSum & 1) << 7) + (ucCheckSum >> 1) + *cPtrSFN_alias++;
    }
    return ucCheckSum;
}


static int fnInsertLFN_name(DIR_ENTRY_STRUCTURE_FAT32 **ptrFileEntry, char fileName[256], char cShortFileName[8 + 3 + 1])
{
    DIR_ENTRY_STRUCTURE_FAT32 *ptrDirectoryEntry = *ptrFileEntry;
    LFN_ENTRY_STRUCTURE_FAT32 *ptrLongFileEntry = 0;
    const char *ptrLongFileName = fileName;
    int i;
    size_t iNameLength = strlen(fileName);                               // length of long file name that must be saved
    unsigned char ucEntryLength = 1;                                     // smallest size that can be required by a LFN (SFN will be required after it too)
    const char *ptrReverseLFN = (ptrLongFileName + iNameLength);         // set the LFN name pointer to the end of the string (terminator) since it is going to be copied in reverse order
    unsigned char ucNextCharacter = 0xff;                                // default is pad character
    unsigned char ucNextCharacterExtension = 0xff;
    unsigned char ucSFN_alias_checksum;
    fnCreateInvalidSFN_alias(cShortFileName);                            // create invalid SFN entry that avoids FAT32 LFN patent issues, doesn't allow compatibility with non-LFN systems and has lowest probability to cause Windows XP or chkdsk problems
    ucSFN_alias_checksum = fnLFN_checksum(cShortFileName);               // calculate the short file name alias's checksum

    iNameLength++;                                                       // include the string terminator in the length
    while (iNameLength > 13) {                                           // calculate the number of entries required to store the long file name in
        ucEntryLength++;
        iNameLength -= 13;                                               // subtract the part of the name that can fit in each single entry
    }
    i = ucEntryLength;                                                   // we check whether there is a known hole in the directory for reuse

    ucEntryLength |= 0x40;                                               // mark first entry
    i = -1;
    while (1) {                                                          // for each character in the LFN
        if (iNameLength >= 13) {                                         // pad end when the LFN doesn't fill an entry
            ucNextCharacter = *ptrReverseLFN;
            ucNextCharacterExtension = 0;                                // only english character set supported
        }
        switch (i--) {
        case 0:                                                          // move to next entry
        {
            ptrDirectoryEntry++;
            if (ptrReverseLFN < ptrLongFileName) {                       // the complete LFN has been added
                *ptrFileEntry = ptrDirectoryEntry;
                return 0;
            }
        }
        // Fall through intentional
        //
        case -1:
            ptrLongFileEntry = (LFN_ENTRY_STRUCTURE_FAT32 *)ptrDirectoryEntry;
            memset(ptrLongFileEntry, 0x00, sizeof(LFN_ENTRY_STRUCTURE_FAT32)); // ensure initially zeroed
            ptrLongFileEntry->LFN_EntryNumber = ucEntryLength;           // entry number
            ucEntryLength &= ~0x40;
            ucEntryLength--;
            ptrLongFileEntry->LFN_Attribute = 0x0f;                      // mark as hidden system file so that SFN systems will ignore the entry
            ptrLongFileEntry->LFN_Checksum = ucSFN_alias_checksum;       // the SFN alias checksum is inserted in each entry
            i = 31;
            continue;
        case 1:
            ptrLongFileEntry->LFN_Name_0 = ucNextCharacter;
            break;
        case 2:
            ptrLongFileEntry->LFN_Name_0_extension = ucNextCharacterExtension;
            continue;
        case 3:
            ptrLongFileEntry->LFN_Name_1 = ucNextCharacter;
            break;
        case 4:
            ptrLongFileEntry->LFN_Name_1_extension = ucNextCharacterExtension;
            continue;
        case 5:
            ptrLongFileEntry->LFN_Name_2 = ucNextCharacter;
            break;
        case 6:
            ptrLongFileEntry->LFN_Name_2_extension = ucNextCharacterExtension;
            continue;
        case 7:
            ptrLongFileEntry->LFN_Name_3 = ucNextCharacter;
            break;
        case 8:
            ptrLongFileEntry->LFN_Name_3_extension = ucNextCharacterExtension;
            continue;
        case 9:
            ptrLongFileEntry->LFN_Name_4 = ucNextCharacter;
            break;
        case 10:
            ptrLongFileEntry->LFN_Name_4_extension = ucNextCharacterExtension;
            continue;
            //case 11:                                                   // attribute (added during initialisation)
            //case 12:                                                   // zero (zeroed during initialisation)
            //case 13:                                                   // checksum (added during initialisation)
           //     continue;
        case 14:
            ptrLongFileEntry->LFN_Name_5 = ucNextCharacter;
            i -= 3;                                                      // skip 11, 12, and 13
            break;
        case 15:
            ptrLongFileEntry->LFN_Name_5_extension = ucNextCharacterExtension;
            continue;
        case 16:
            ptrLongFileEntry->LFN_Name_6 = ucNextCharacter;
            break;
        case 17:
            ptrLongFileEntry->LFN_Name_6_extension = ucNextCharacterExtension;
            continue;
        case 18:
            ptrLongFileEntry->LFN_Name_7 = ucNextCharacter;
            break;
        case 19:
            ptrLongFileEntry->LFN_Name_7_extension = ucNextCharacterExtension;
            continue;
        case 20:
            ptrLongFileEntry->LFN_Name_8 = ucNextCharacter;
            break;
        case 21:
            ptrLongFileEntry->LFN_Name_8_extension = ucNextCharacterExtension;
            continue;
        case 22:
            ptrLongFileEntry->LFN_Name_9 = ucNextCharacter;
            break;
        case 23:
            ptrLongFileEntry->LFN_Name_9_extension = ucNextCharacterExtension;
            continue;
        case 24:
            ptrLongFileEntry->LFN_Name_10 = ucNextCharacter;
            break;
        case 25:
            ptrLongFileEntry->LFN_Name_10_extension = ucNextCharacterExtension;
            continue;
            //case 26:                                                         // zeros (zeroed during initialisation)
            //case 27:
            //    continue;
        case 28:
            ptrLongFileEntry->LFN_Name_11 = ucNextCharacter;
            i -= 2;                                                      // skip 26 and 27
            break;
        case 29:
            ptrLongFileEntry->LFN_Name_11_extension = ucNextCharacterExtension;
            continue;
        case 30:
            ptrLongFileEntry->LFN_Name_12 = ucNextCharacter;
            break;
        case 31:
            ptrLongFileEntry->LFN_Name_12_extension = ucNextCharacterExtension;
            continue;
        }
        if (iNameLength >= 13) {                                         // name character was added
            ptrReverseLFN--;
        }
        else {
            iNameLength++;                                               // pad was added
        }
    }
    return 0;
}

static unsigned char fnGetDec(char *ptrBuf)
{
    unsigned char ucValue = (*ptrBuf++ - '0');
    ucValue *= 10;
    ucValue += (*ptrBuf - '0');
    return ucValue;
}

static unsigned short fnGetTime(__time64_t time)
{
    char timebuf[26];
    unsigned char ucSeconds;
    unsigned char ucMinutes;
    unsigned char ucHours;
    char *ptrBuf = timebuf;
    ctime_s(timebuf, 26, &time);
    while (*ptrBuf != ':') {
        ptrBuf++;
    }
    ptrBuf -= 2;
    ucHours = fnGetDec(ptrBuf);
    ptrBuf += 3;
    ucMinutes = fnGetDec(ptrBuf);
    ptrBuf += 3;
    ucSeconds = fnGetDec(ptrBuf);
    return (ucSeconds | (ucMinutes << 5) | (ucHours << 11));
}

static const char *months[12] = {
    "jan",
    "feb",
    "mar",
    "apr",
    "may",
    "jun",
    "jul",
    "aug",
    "sep",
    "oct",
    "nov",
    "dec",
};

static char *fnGetMonth(char *ptrBuf, unsigned char *ptrMonth)
{
    unsigned char ucMonth = 0;
    char *ptrThisMonth;
    int iMatch;
    int i;
    while (*ptrBuf != 0) {
        i = 0;
        while (i < 12) {                                                 // for each possible month
            ptrThisMonth = ptrBuf;
            iMatch = 0;
            while ((*ptrThisMonth == months[i][iMatch]) || (*ptrThisMonth == (months[i][iMatch] + ('A' - 'a')))) {
                ptrThisMonth++;
                iMatch++;
                if (iMatch == 3) {                                       // month found
                    *ptrMonth = (unsigned char)(i + 1);
                    return ptrThisMonth;
                }
            }
            i++;
        }
        ptrBuf++;
    }
    return 0;
}

unsigned char fnGetDay(char *ptrBuf)
{
    unsigned char ucValue;
    while (*ptrBuf == ' ') {
        ptrBuf++;
    }
    ucValue = (*ptrBuf++ - '0');
    if (*ptrBuf != ' ') {
        ucValue *= 10;
        ucValue += (*ptrBuf - '0');
    }
    return ucValue;
}

static unsigned long fnGetYear(char *ptrBuf)
{
    unsigned long ulValue;
    while (*ptrBuf != '\n') {
        ptrBuf++;
    }
    ptrBuf -= 4;
    ulValue = (*ptrBuf++ - '0');
    ulValue *= 10;
    ulValue += (*ptrBuf++ - '0');
    ulValue *= 10;
    ulValue += (*ptrBuf++ - '0');
    ulValue *= 10;
    ulValue += (*ptrBuf - '0');
    return ulValue;
}

static unsigned short fnGetDate(__time64_t time)
{
    char timebuf[26];
    unsigned char ucDay;
    unsigned char ucMonth;
    unsigned long ulYear;
    char *ptrBuf;
    ctime_s(timebuf, 26, &time);
    ptrBuf = fnGetMonth(timebuf, &ucMonth);
    ucDay = fnGetDay(ptrBuf);
    ulYear = fnGetYear(ptrBuf);
    return (ucDay | (ucMonth << 5) | ((unsigned char)(ulYear - 1980) << 9));
}

static int fnAddFileObject(unsigned char *ptrBuffer, unsigned long ulFileObjectLength, const wchar_t *file_name, const wchar_t *disk_name)
{
    unsigned long ulFileLength;
    char fileName[256];
    char diskName[256];
    int i = 0;
    int j = 0;
    struct _stat buf;
    while (1) {
        fileName[i] = (char)file_name[i];                                // change from unicode name to character name
        if (fileName[i] == 0) {
            break;
        }
        i++;
    }
    _stat((const char *)fileName, &buf);
    j = 0;
    i = 0;
    while (1) {                                                          // remove path to leave just file name
        fileName[j] = (char)file_name[i];
        if (fileName[j] == 0) {
            break;
        }
        else if ((fileName[j] == '\\') || (fileName[j] == '/')) {
            j = 0;
        }
        else if (fileName[j] != '"') {
            j++;
        }
        i++;
    }
    memset(ptrBuffer, 0, ulFileObjectLength);
    ulFileLength = buf.st_size;
    j = 0;
    i = 0;
    while (1) {
        diskName[j] = (char)disk_name[i];
        if (diskName[j] == 0) {
            break;
        }
        i++;
        if (diskName[j] == '-') {
            while (1) {
                diskName[j] = (char)disk_name[i];
                if ((diskName[j] == 0) || (diskName[j] == ']')) {
                    char cShortFileName[8 + 3 + 1];
                    DIR_ENTRY_STRUCTURE_FAT32 *ptrFileEntry;
                    FILE_OBJECT_INFO volumeEntry;
                    ptrFileEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptrBuffer;
                    ptrFileEntry->DIR_Attr = DIR_ATTR_VOLUME_ID;         // volume
                    memcpy(ptrFileEntry->DIR_Name, diskName, 11);        // volume name
                    volumeEntry.file_length = 0;
                    volumeEntry.usCreationTime = fnGetTime(buf.st_ctime);
                    volumeEntry.usCreationDate = fnGetDate(buf.st_ctime);
                    volumeEntry.usModifiedTime = fnGetTime(buf.st_mtime);
                    volumeEntry.usModifiedDate = fnGetDate(buf.st_mtime);
                    volumeEntry.usAccessedDate = fnGetDate(buf.st_atime);
                    fnSetObjectDetails(ptrFileEntry, &volumeEntry);      // add volume's creation date/time details
                    ptrFileEntry++;
                    ptrFileEntry->DIR_NTRes = 0x18;
                    ptrFileEntry->DIR_Attr = DIR_ATTR_ARCHIVE;           // file
                    ptrFileEntry->DIR_FstClusLO[0] = 3;                  // first cluster used for file content
                    volumeEntry.file_length = ulFileLength;
                    fnInsertLFN_name(&ptrFileEntry, fileName, cShortFileName);
                    ptrFileEntry->DIR_Attr = DIR_ATTR_ARCHIVE;
                    memcpy(ptrFileEntry->DIR_Name, cShortFileName, 11);  // add the short file name
                    fnSetObjectDetails(ptrFileEntry, &volumeEntry);      // add volume's creation date/time details
                    return 0;
                }
                i++;
                j++;
            }
        }
    }
    return -1;
}

static int fnCheckType(char *ptrFileName)
{ 
    char *ptrTestName = ptrFileName;
    char *ptrExtension = 0;
    while ((*ptrTestName != 0) || (*(ptrTestName + 1) != 0)) {
        if (*ptrTestName == '.') {
            ptrExtension = ptrTestName;
        }
        ptrTestName++;
    }
    if (ptrExtension == 0) {
        return INTEL_HEX;                                                // no file extension found so default to intel hex format
    }
    do {
        ptrExtension++;
    } while (*ptrExtension == 0x00);
    if ((*ptrExtension != 'S') && (*ptrExtension != 's')) {              // the extension is expected to be .srec or .out
        do {
            ptrExtension++;
        } while (*ptrExtension == 0x00);
        if ((*ptrExtension != 'O') && (*ptrExtension != 'o')) {          // the extension is expected to be .out
            return INTEL_HEX;
        }
        do {
            ptrExtension++;
        } while (*ptrExtension == 0x00);
        if ((*ptrExtension != 'U') && (*ptrExtension != 'u')) {          // the extension is expected to be .out
            return INTEL_HEX;
        }
        do {
            ptrExtension++;
        } while (*ptrExtension == 0x00);
        if ((*ptrExtension != 'T') && (*ptrExtension != 't')) {          // the extension is expected to be .out
            return INTEL_HEX;
        }
        return MOTOROLA_SREC;                                            // *.out accepted as motorola srec format required
    }
    do {
        ptrExtension++;
    } while (*ptrExtension == 0x00);
    if ((*ptrExtension != 'R') && (*ptrExtension != 'r')) {              // the extension is expected to be .srec
        return INTEL_HEX;
    }
    do {
        ptrExtension++;
    } while (*ptrExtension == 0x00);
    if ((*ptrExtension != 'E') && (*ptrExtension != 'e')) {              // the extension is expected to be .srec
        return INTEL_HEX;
    }
    do {
        ptrExtension++;
    } while (*ptrExtension == 0x00);
    if ((*ptrExtension != 'C') && (*ptrExtension != 'c')) {              // the extension is expected to be .srec
        return INTEL_HEX;
    }
    return MOTOROLA_SREC;                                                // *.srec accepted as motorola srec format required
}

static unsigned long fnGetAddress(char *ptrAddStr, unsigned long *ptrFileObjectLength)
{
    unsigned long ulValue = 0;
    unsigned char ucNewByte;
    if ((*(ptrAddStr + 2) == 'x') || (*(ptrAddStr + 2) == 'X')) {        // hex input
        ptrAddStr += 4;
        while ((ucNewByte = *ptrAddStr) != 0) {
            ptrAddStr += 2;
            if (ucNewByte == '[') {                                      // adding an optional file object length)
                if ((*(ptrAddStr + 2) == 'x') || (*(ptrAddStr + 2) == 'X')) { // hex input
                    unsigned long ulFileObjectLength = 0;
                    ptrAddStr += 4;
                    while ((ucNewByte = *ptrAddStr) != '-') {
                        if ((ucNewByte == 0) || (ucNewByte == ']')) {
                            break;
                        }
                        ptrAddStr += 2;
                        ulFileObjectLength <<= 4;
                        ucNewByte -= '0';
                        if (ucNewByte > 9) {
                            ucNewByte -= ('A' - '9' - 1);
                            if (ucNewByte > 0x0f) {
                                ucNewByte -= ('a' - 'A');
                            }
                        }
                        ulFileObjectLength |= ucNewByte;
                    }
                    if (ptrFileObjectLength != 0) {
                        *ptrFileObjectLength = ulFileObjectLength;
                    }
                }
                return ulValue;
            }
            ulValue <<= 4;
            ucNewByte -= '0';
            if (ucNewByte > 9) {
                ucNewByte -= ('A' - '9' - 1);
                if (ucNewByte > 0x0f) {
                    ucNewByte -= ('a' - 'A');
                }
            }
            ulValue |= ucNewByte;
        }
    }
    else {                                                               // decimal input
        while ((ucNewByte = *ptrAddStr) != 0) {
            ptrAddStr += 2;
            ulValue *= 10;
            ucNewByte -= '0';
            ulValue += ucNewByte;
        }
    }
    return ulValue;
}

static void fnAddByte(char *ptrStr, unsigned char ucValue)
{
    char cTemp = (ucValue >> 4);
    cTemp += '0';
    if (cTemp > '9') {
        cTemp += ('A' - '9' - 1);
    }
    *ptrStr++ = cTemp;
    cTemp = (ucValue & 0x0f);
    cTemp += '0';
    if (cTemp > '9') {
        cTemp += ('A' - '9' - 1);
    }
    *ptrStr = cTemp;
}

static void fnAddWord(char *ptrStr, unsigned short usValue)
{
    fnAddByte(ptrStr, (unsigned char)(usValue >> 8));
    fnAddByte((ptrStr + 2), (unsigned char)(usValue));
}

static void fnAdd24Word(char *ptrStr, unsigned long ulValue)
{
    fnAddByte(ptrStr, (unsigned char)(ulValue >> 16));
    fnAddByte((ptrStr + 2), (unsigned char)(ulValue >> 8));
    fnAddByte((ptrStr + 4), (unsigned char)(ulValue));
}

static void fnAddLong(char *ptrStr, unsigned long ulValue)
{
    fnAddByte(ptrStr, (unsigned char)(ulValue >> 24));
    fnAddByte((ptrStr + 2), (unsigned char)(ulValue >> 16));
    fnAddByte((ptrStr + 4), (unsigned char)(ulValue >> 8));
    fnAddByte((ptrStr + 6), (unsigned char)(ulValue));
}

static void fnIntelHex(int File, unsigned char *content, unsigned long ulLength, unsigned long ulOffset)
{
    #define LINE_LENGTH 16
    static const char cFinal[] = ":00000001FF\r\n";
    unsigned short usExtended = 0;
    char cLine[(LINE_LENGTH * 2) + 14];
    int iLineLength = sizeof(cLine);
    unsigned char ucLineLength = LINE_LENGTH;
    unsigned char ucCheckSum;
    int i;
    cLine[0] = ':';
    while (ulLength != 0) {
        if (usExtended != (ulOffset >> 16)) {
            usExtended = (unsigned short)(ulOffset >> 16);
            fnAddByte(&cLine[1], 2);
            fnAddWord(&cLine[3], 0x0000);
            if (ulOffset > 0xffffff) {
                ucCheckSum = 6;
                fnAddByte(&cLine[7], 4);                                 // extended linear address record type
                fnAddWord(&cLine[9], usExtended);
                ucCheckSum += (unsigned char)((usExtended) >> 8);
                ucCheckSum += (unsigned char)(usExtended);
            }
            else {
                ucCheckSum = 4;
                fnAddByte(&cLine[7], 2);                                 // extended address record type
                fnAddWord(&cLine[9], (usExtended << 12));
                ucCheckSum += (unsigned char)((usExtended << 12) >> 8);
                ucCheckSum += (unsigned char)(usExtended << 12);
            }
            fnAddByte(&cLine[13], ~(ucCheckSum - 1));
            cLine[15] = '\r';
            cLine[16] = '\n';
            _write(File, cLine, 17);
            continue;
        }
        if (ulLength < LINE_LENGTH) {
            ucLineLength = (unsigned char)ulLength;
        }
        ucCheckSum = ucLineLength;
        fnAddByte(&cLine[1], ucLineLength);
        ucCheckSum += (unsigned char)(ulOffset >> 8);
        ucCheckSum += (unsigned char)(ulOffset);
        fnAddWord(&cLine[3], (unsigned short)ulOffset);
        fnAddByte(&cLine[7], 0);                                         // data line type
        i = 0;
        while (i < ucLineLength) {
            ucCheckSum += *content;
            fnAddByte(&cLine[9 + i*2], *content++);
            i++;
        }
        fnAddByte(&cLine[9 + (ucLineLength*2)], ~(ucCheckSum-1));
        cLine[11 + (ucLineLength * 2)] = '\r';
        cLine[12 + (ucLineLength * 2)] = '\n';
        _write(File, cLine, 13 + (ucLineLength*2));
        ulLength -= ucLineLength;
        ulOffset += ucLineLength;
    }
    //:1000000018F09FE51CF09FE518F09FE514F09FE5C0

    // Add last line
    _write(File, cFinal, (sizeof(cFinal) - 1));
}

static int fnAddAddress(char *ptrLine, int iOffset, unsigned long ulAddress, unsigned char ucAddLen, unsigned char *ptrChecksum)
{
    switch (ucAddLen) {
    case 2:
        *ptrChecksum += (unsigned char)(ulAddress >> 8);
        *ptrChecksum += (unsigned char)(ulAddress);
        fnAddWord(&ptrLine[iOffset], (unsigned short)ulAddress);
        return (iOffset + 4);
    case 3:
        *ptrChecksum += (unsigned char)(ulAddress >> 16);
        *ptrChecksum += (unsigned char)(ulAddress >> 8);
        *ptrChecksum += (unsigned char)(ulAddress);
        fnAdd24Word(&ptrLine[iOffset], ulAddress);
        return (iOffset + 6);
    case 4:
        *ptrChecksum += (unsigned char)(ulAddress >> 24);
        *ptrChecksum += (unsigned char)(ulAddress >> 16);
        *ptrChecksum += (unsigned char)(ulAddress >> 8);
        *ptrChecksum += (unsigned char)(ulAddress);
        fnAddLong(&ptrLine[iOffset], ulAddress);
        return (iOffset + 8);
    default:
        return 0;
    }
}


static void fnMotorolaHex(int File, unsigned char *content, unsigned long ulLength, unsigned long ulOffset, char cType)
{
    unsigned char ucCheckSum = 0;
    int iIndex;
    char cLine[(LINE_LENGTH * 2) + 14];
    unsigned char ucAddLen;
    unsigned char ucLineLength;
    if ((ulLength + ulOffset) > 0xffff) {                                // S2 required
        if ((ulLength + ulOffset) > 0xffffff) {                          // S3 required
            cType = '3';
        }
        else {
            if (cType < '2') {
                cType = '2';
            }
        }
    }
    if (cType == '1') {
        ucAddLen = 2;
    }
    else if (cType == '2') {
        ucAddLen = 3;
    }
    else {
        ucAddLen = 4;
    }
    cLine[0] = 'S';
    cLine[1] = '0';                                                      // record type
    ucCheckSum = (ucAddLen + 1);
    fnAddByte(&cLine[2], ucCheckSum);                                    // byte count
    iIndex = fnAddAddress(cLine, 4, 0, ucAddLen, &ucCheckSum);
    fnAddByte(&cLine[iIndex], (~ucCheckSum));
    iIndex += 2;
    cLine[iIndex++] = '\n';
    _write(File, cLine, iIndex);

    cLine[1] = cType;                                                    // record type used for data
    while (ulLength != 0) {
        cLine[1] = cType;
        ucLineLength = LINE_LENGTH;
        if (ulLength < LINE_LENGTH) {
            ucLineLength = (unsigned char)ulLength;
        }
        ucCheckSum = (ucLineLength + ucAddLen + 1);
        fnAddByte(&cLine[2], ucCheckSum);                                // byte count
        iIndex = fnAddAddress(cLine, 4, ulOffset, ucAddLen, &ucCheckSum);
        ulOffset += ucLineLength;
        while (ucLineLength--) {
            ucCheckSum += *content;
            fnAddByte(&cLine[iIndex], *content++);
            iIndex += 2;
            ulLength--;
        }
        fnAddByte(&cLine[iIndex], (~ucCheckSum));
        iIndex += 2;
        cLine[iIndex++] = '\n';
        _write(File, cLine, iIndex);
    }

    cLine[1] = '9';                                                      // final record type
    ucCheckSum = (ucAddLen + 1);
    fnAddByte(&cLine[2], ucCheckSum);                                    // byte count
    iIndex = fnAddAddress(cLine, 4, 0, ucAddLen, &ucCheckSum);
    fnAddByte(&cLine[iIndex], (~ucCheckSum));
    iIndex += 2;
    cLine[iIndex++] = '\n';
    _write(File, cLine, iIndex);
}