/************************************************************************
    Mark Butcher    Bsc (Hons) MPhil MIET

    M.J.Butcher Consulting
    Birchstrasse 20f,    CH-5406, Rtihof
    Switzerland

    www.uTasker.com    Skype: M_J_Butcher

    ---------------------------------------------------------------------
    File:        mass_storage.c
    Project:     uTasker project
    ---------------------------------------------------------------------
    Copyright (C) M.J.Butcher Consulting 2004..2012
    *********************************************************************

    05.01.2010 Modify free cluster value written to info sector during formatting {1}
    19.01.2010 Reset file length when creating files to avoid taking values from previously deleted entres {2}
    20.01.2010 Add new time when writing to a file - add creation parameter to function fnSetTimeDate() {3}
    21.01.2010 Correction in fnAllocateCluster() which could cause FAT corruption on cluster growth {4}
    26.01.2010 Add SD controller interface                               {5}
    27.01.2010 Correct creating, renaming and deleting files and directories in referenced sub-directories with help of flag UTDIR_SET_START {6}
    31.01.2010 Correct root directory referenced paths                   {7}
    31.01.2010 Extend utOpenFile() mode parameter from unsigned char to unsigned short {8}
    02.02.2010 Don't allow a file to be renamed unless open for rename or write {9}
    02.02.2010 Don't allow a file to be written or deleted if it is read-only {10}
    08.02.2010 Ensure that the mass storage polling is terminated after each command that needed to wait {12}
    12.02.2010 Correct utReadFile() sector increment when end of file meets sector boundard {13}
    16.02.2010 Correct cluster counting when using info block as reference (move {4} to be generally used) {14}
    16.02.2010 Correct original FAT sector when allocating new clusters  {15}
    18.02.2010 Set directory back to lowest directory when the directory cluster end is exhausted {16}
    18.02.2010 Correctly extend directory clusters                       {17}
    19.03.2010 Add force of sector load rather than a transfer to a buffer {18}
    29.03.2010 Add fnGetLocalFileTime() to allow file data/time stamps to be returned by systems with corresponding support {19}
    30.03.2010 Block reformat when the disk is write protected and clear disk state when starting {20} utFATV1.1
    09.04.2010 First version of MANAGED_FILES support including delete control {21}
    14.06.2010 Add fnWriteSector()                                       {22}
    06.07.2010 utReadDirectory() made extern                             {23}
    30.07.2010 Optimised local routines to avoid reading partial data when not necessary - simplifies and improves compatibility with SD controller operation
    04.08.2010 Move various defines from here to mass_storage.h
    06.08.2010 Introduce SDCARD_MALLOC() macro to better control the malloc() area used {24}
    16.09.2010 Ensure that root-referenced paths work with correct location {25}
    16.09.2010 Ensure that the public disk location entry can not be used uninitialised {26} utFATV1.6
    24.09.2010 Change ucDiskFlags to usDiskFlags and add DISK_NOT_PRESENT when no disk detected and DISK_TYPE_NOT_SUPPORTED when non-supported type {27}
    25.09.2010 Add uMatchFileExtension()                                 {28} utFAT1.7
    17.10.2010 Add NAND Flash support                                    {29} utFAT1.8
    17.02.2011 Copy volume label when a disk is mounted                  {30}
    21.02.2011 Cast DIR_NAME_FREE comparison                             {31}
    24.02.2011 Add UTFAT16 support                                       {32}
    24.02.2011 Add support for full formatting (including data section) and formatting flags {33}
    25.02.2011 Block any write actions when the disk is write protected and allow use to select how the WP switch is read {34} utFAT1.9
    20.08.2011 Correct cluster allocation over sector boundaries         {35}
    20.08.2011 Correct big-endian cluster deletion                       {36}
    24.08.2011 Recognise empty file names and reject them                {37}
    24.08.2011 FAT16 cluster corrections                                 {38} utFAT1.10
    17.09.2011 Allow UTFAT_APPEND attribute to set the file pointer straight to the end of the file when opening for write {39}
    18.01.2012 Allow operation without UTMANAGED_FILE_COUNT and/or without UTFAT_WRITE {40}
    18.01.2012 Enable debug output to be disabled with UTFAT_DISABLE_DEBUG_OUT {41}
    18.01.2012 Allow fnRunThroughClusters() to return error value and limit forward seek to end of file {42} utFAT1.11
    20.01.2012 Remove some unused variables when no UTFAT_WRITE/UTMANAGED_FILE_COUNT operation {43}
    28.01.2012 Add LFN length protection                                 {44}
    28.02.2012 Add LFN delete support                                    {45}
    06.03.2012 Don't delete file content cluster chains when they don't exist {46}
    21.03.2012 Allow directory paths aaa\bbb\ terminated with \          {47}
    02.04.2012 Add utServer()                                            {48}
    02.04.2012 Add SD card simulation display interface                  {49}
    02.04.2012 Add card removal monitoring                               {50}
    27.07.2012 Improve return result from ut_read_disk() to include errors {51}
    27.07.2012 Add card detection input polling and interrupt modes      {52}
    05.08.2012 Card removal monitoring based on checking card modified to read last section {53} utFAT1.12

*/


/* =================================================================== */
/*                           include files                             */
/* =================================================================== */

#include "config.h"


/* =================================================================== */
/*                          local definitions                          */
/* =================================================================== */

#if !defined UTFAT_DISABLE_DEBUG_OUT                                     // {41} allow all debug messages to be disabled
    #define fnMemoryDebugMsg(x) fnDebugMsg(x)                            // enable debug output
    #define fnMemoryDebugHex(x, y) fnDebugHex((x), (y))                  // enable debug output
    #define fnMemoryDebugDec(x, y) fnDebugDec((x), (y))                  // enable debug output
#else
    #define fnMemoryDebugMsg(x)                                          // disable debug output
    #define fnMemoryDebugHex(x, y)                                       // disable debug output
    #define fnMemoryDebugDec(x, y)                                       // disable debug output
#endif

#ifndef SDCARD_MALLOC
    #define SDCARD_MALLOC(x) uMalloc(x)
#endif

#define OWN_TASK                         TASK_MASS_STORAGE

#if defined SDCARD_SUPPORT

#define T_POWER_STABILISE                (DELAY_LIMIT)(SEC * 0.05)
#define T_NEXT_CHECK                     (DELAY_LIMIT)(SEC * SD_CARD_RETRY_INTERVAL)
#define T_CLEAN_SPARE                    (DELAY_LIMIT)(SEC * 0.10)

// Timer events
//
#define E_POWER_STABILISED               1
#define E_POLL_SD_CARD                   2
#define E_CLEAN_SPARE                    3
#define E_CHECK_CARD_REMOVAL             4                               // used when trying to detect removal by reading register

// Interrpt events
//
#define E_SDCARD_DETECTION_CHANGE        1

#define SD_STATE_STARTING                0
#define SD_STATE_WAIT_CARD               1
#define SD_STATE_STABILISING             2
#define SD_STATE_GO_IDLE                 3
#define SD_STATE_IF_COND                 4
#define SD_STATE_APP_CMD55_CMD41         5
#define SD_STATE_APP_CMD55_CMD41_2       6
#define SD_STATE_OCR                     7
#define SD_STATE_GET_INFO                8
#define SD_STATE_SET_ADDRESS             9
#define SD_STATE_GET_SECTORS             10
#define SD_SET_INTERFACE_PREPARE         11
#define SD_SET_INTERFACE                 12
#define SD_SELECT_CARD                   13
#define SD_SET_BLOCK_LENGTH              14
#define SD_MOUNTING_1                    15
#define SD_MOUNTING_2                    16
#define SD_MOUNTING_3                    17
#define SD_MOUNTING_4                    18
#define SD_STATE_READY                   19

#define SD_STATE_FORMATTING_DISK_1       20
#define SD_STATE_FORMATTING_DISK_2       21
#define SD_STATE_FORMATTING_DISK_3       22
#define SD_STATE_FORMATTING_DISK_4       23
#define SD_STATE_FORMATTING_DISK_5       24
#define SD_STATE_FORMATTING_DISK_6       25
#define SD_STATE_FORMATTING_DISK_7       26
#define SD_STATE_FORMATTING_DISK_8       31
#define SD_STATE_FORMATTING_DISK_8A      32
#define SD_STATE_FORMATTING_DISK_9       33
#define SD_STATE_FORMATTING_DISK_10      34
#define SD_STATE_FORMATTING_DISK_11      35
#define SD_STATE_CHECKING_DISK_REMOVAL   36

#define SD_NOT_FORMATTED                 50


#define _IDLE_MEMORY                     0x00
#define _INITIALISED_MEMORY              0x01
#define _READING_MEMORY                  0x02
#define _WRITING_MEMORY                  0x04
#define _FORMATTING_DISK                 0x08
#define _COUNTING_CLUSTERS               0x10


#define END_DIRECTORY_ENTRIES            -2
#define MATCH_FALSE                      -1
#define MATCH_CONTINUE                   0
#define MATCH_SUCCESSFUL                 1


// File search results
//
#define FULLY_QUALIFIED_LONG_NAME        3
#define LONG_NAME_PARAGRAPH              2
#define FULLY_QUALIFIED_SHORT_NAME       1
#define SHORT_NAME_PARAGRAPH             0
#define INVALID_PARAGRAPH                -1


// FAT32 BPB_ExtFlags flags
//
#define BPB_ExtFlags_0_ZERO_BASED_NUMBER 0x0f                            // only valid if mirroring is disabled
#define BPB_ExtFlags_0_MIRRORED_FAT      0x00
#define BPB_ExtFlags_0_ONE_FAT           0x80                            // mirroring disabled

#define DIR_NAME_FREE                    0xe5

#define DIR_ATTR_READ_ONLY               0x01
#define DIR_ATTR_HIDDEN                  0x02
#define DIR_ATTR_SYSTEM                  0x04
#define DIR_ATTR_VOLUME_ID               0x08
#define DIR_ATTR_DIRECTORY               0x10
#define DIR_ATTR_ARCHIVE                 0x20
#define DIR_ATTR_LONG_NAME               (DIR_ATTR_READ_ONLY | DIR_ATTR_HIDDEN | DIR_ATTR_SYSTEM | DIR_ATTR_VOLUME_ID)
#define DIR_ATTR_MASK                    (DIR_ATTR_LONG_NAME | DIR_ATTR_DIRECTORY | DIR_ATTR_ARCHIVE)

#define BACK_SLASH                       0x5c
#define FORWARD_SLASH                    0x2f

#define NT_FLAG                          0x18

#define NEW_ABSOLUTE_CLUSTER             0x00
#define INITIALISE_DIR_CLUSTER           0x01
#define INITIALISE_DIR_EXTENSION         0x02
#define UPDATE_FAT_END                   0x04
#define UPDATE_FAT_END_IN_DIFF_SECT      0x08
#define NEW_RELATIVE_CLUSTER             (UPDATE_FAT_END | UPDATE_FAT_END_IN_DIFF_SECT)


#define _CHAR_REJECT                     0x01
#define _CHAR_TERMINATE                  0x02
#define _CHAR_CAPITAL                    0x04
#define _CHAR_SMALL                      0x08
#define _CHAR_NUMBER                     0x10
#define _CHAR_REJECT_NON_JAP             0x20

    #ifdef SD_CONTROLLER_AVAILABLE
        #define SD_CONTROLLER_SHIFT      2
    #else
        #define SD_CONTROLLER_SHIFT      0
    #endif
#endif

/* =================================================================== */
/*                      local structure definitions                    */
/* =================================================================== */


/* =================================================================== */
/*                 local function prototype declarations               */
/* =================================================================== */

#if defined SDCARD_SUPPORT
    #if defined SD_CONTROLLER_AVAILABLE                                  // routines supplied by HW specific module {5}
        extern void fnInitSDCardInterface(void);
        extern int  fnSendSD_command(const unsigned char ucCommand[6], unsigned char *ucResult, unsigned char *ptrReturnData);
        extern int  fnGetSector(unsigned char *ptrBuf);       
        extern int  fnReadPartialSector(unsigned char *ptrBuf, unsigned short usStart, unsigned short usStop);
        extern int  fnPutSector(unsigned char *ptrBuf);
    #elif defined NAND_FLASH_FAT
        #include "NAND_driver.h"                                         // include NAND driver code
    #else
        static int  fnWaitSD_ready(int iMaxWait);
        static int  fnSendSD_command(const unsigned char ucCommand[6], unsigned char *ucResult, unsigned char *ptrReturnData);
        static int  fnGetSector(unsigned char *ptrBuf);
        static int  fnReadPartialSector(unsigned char *ptrBuf, unsigned short usStart, unsigned short usStop);
    #endif
    static int  utReadDisk(UTDISK *ptr_utDisk, unsigned long ulSectorNumber);
    static int  utReadDiskSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf);
    static int  utReadPartialDiskData(UTDISK *ptr_utDisk, unsigned long ulSector, void *ptrBuf, unsigned short usOffset, unsigned short usLength);
    static int  fnLoadSector(UTDISK *ptr_utDisk, unsigned long ulSector);
    #if !defined NAND_FLASH_FAT
        static const unsigned char *fnCreateCommand(unsigned char ucCommand, unsigned long ulValue);
    #endif
    static int  ut_read_disk(UTDISK *ptr_utDisk);
    static void fnCardNotFormatted(void);
    static void fnInitialisationError(int);
    #ifdef UTFAT_WRITE
        static int  utCommitSector(UTDISK *ptr_utDisk, unsigned long ulSector);
        static int  utCommitSectorData(UTDISK *ptr_utDisk, void *ptrBuffer, unsigned long ulSectorNumber);
        static int  utDeleteSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber);
        static unsigned long fnAllocateCluster(UTDISK *ptr_utDisk, unsigned long ulPresentCluster, unsigned char ucClusterType);
        static int  fnDeleteFileContent(UTFILE *ptr_utFile, UTDISK *ptr_utDisk, int iDestroyClusters);
        static int  fnDeleteClusterChain(unsigned long ulClusterStart, unsigned char ucDrive, int iDestroyClusters);
        static void fnAddInfoSect(INFO_SECTOR_FAT32 *ptrInfoSector, unsigned long ulFreeCount, unsigned long ulNextFree);
    #endif
    #if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0         // {43}
        static int fnFileLocked(UTFILE *ptr_utFile);
    #endif
    #if defined SDCARD_DETECT_INPUT_INTERRUPT 
        static void fnPrepareDetectInterrupt(void);                      // {52}
    #endif
#endif

/* =================================================================== */
/*                             constants                               */
/* =================================================================== */

#if defined SDCARD_SUPPORT

#if !defined NAND_FLASH_FAT
    static const unsigned char ucGO_IDLE_STATE_CMD0[6]      = {GO_IDLE_STATE_CMD0, 0x00, 0x00, 0x00, 0x00, CS_GO_IDLE_STATE_CMD0};
    static const unsigned char ucIF_COND_CMD8[6]            = {SEND_IF_COND_CMD8, 0x00, 0x00, VOLTAGE_2_7__3_6, CHECK_PATTERN, CS_SEND_IF_COND_CMD8};
    #ifdef SD_CONTROLLER_AVAILABLE                                       // no CRC is set to buffer since the SD controller appends this automatically {5}
        static const unsigned char ucSEND_OP_COND_ACMD_CMD41[5] = {SEND_OP_COND_ACMD_CMD41, HIGH_CAPACITY_SD_CARD_MEMORY, 0xff, 0x80, 0x00};
        static const unsigned char ucSET_REL_ADD_CMD3[5]    = {SET_REL_ADD_CMD3, 0x00, 0x00, 0x00, 0x00};
        static const unsigned char ucSEND_CID_CMD2[5]       = {SEND_CID_CMD2, 0x00, 0x00, 0x00, 0x00};
        static const unsigned char ucSET_BLOCK_LENGTH_CMD16[5]  = {SET_BLOCKLEN_CMD16, 0x00, 0x02, 0x00, 0x02}; // 512 byte block length
        static unsigned char ucSELECT_CARD_CMD7[5]          = {SELECT_CARD_CMD7, 0x00, 0x00, 0x00, 0x00};
        static unsigned char ucSEND_CSD_CMD9[5]             = {SEND_CSD_CMD9, 0x00, 0x00, 0x00, 0x00};
        static unsigned char ucAPP_CMD_CMD55[5]             = {APP_CMD_CMD55, 0x00, 0x00, 0x00, 0x00};
        static unsigned char ucSET_BUS_WIDTH_CMD6[5]        = {SET_BUS_WIDTH_CMD6, 0x00, 0x00, 0x00, 0x02};
    #else
        static const unsigned char ucSEND_OP_COND_ACMD_CMD41[6] = {SEND_OP_COND_ACMD_CMD41, HIGH_CAPACITY_SD_CARD_MEMORY, 0x00, 0x00, 0x00, CS_SEND_OP_COND_ACMD_CMD41};
        static const unsigned char ucSEND_CSD_CMD9[6]       = {SEND_CSD_CMD9, 0x00, 0x00, 0x00, 0x00, CS_SEND_CSD_CMD9};
        static const unsigned char ucAPP_CMD_CMD55[6]       = {APP_CMD_CMD55, 0x00, 0x00, 0x00, 0x00, CS_APP_CMD_CMD55};
        static const unsigned char ucREAD_OCR_CMD58[6]      = {READ_OCR_CMD58, 0x00, 0x00, 0x00, 0x00, CS_READ_OCR_CMD58};
    #endif
#endif

#if defined UTFAT_WRITE && defined UTFAT_FORMATTING                      // {43}
static unsigned char ucEmptyFAT32[12] = {
    LITTLE_LONG_WORD_BYTES(MEDIA_VALUE_FIXED),
    LITTLE_LONG_WORD_BYTES(0xffffffff),
    LITTLE_LONG_WORD_BYTES(CLUSTER_MASK)
};

    #ifdef UTFAT16
static unsigned char ucEmptyFAT16[4] = {
    LITTLE_LONG_WORD_BYTES(0xfffffff8)
};
    #endif
#endif

static const unsigned char ucCharacterTable[] = {
    (0),                                                                 // !
    (_CHAR_REJECT),                                                      // "
    (0),                                                                 // #
    (0),                                                                 // $
    (0),                                                                 // %
    (0),                                                                 // &
    (0),                                                                 // 
    (0),                                                                 // (
    (0),                                                                 // )
    (_CHAR_REJECT),                                                      // *
    (_CHAR_REJECT),                                                      // +
    (0),                                                                 // ,
    (0),                                                                 // -
    (0),                                                                 // .
    (_CHAR_TERMINATE),                                                   // /
    (_CHAR_NUMBER),                                                      // 0
    (_CHAR_NUMBER),                                                      // 1
    (_CHAR_NUMBER),                                                      // 2
    (_CHAR_NUMBER),                                                      // 3
    (_CHAR_NUMBER),                                                      // 4
    (_CHAR_NUMBER),                                                      // 5
    (_CHAR_NUMBER),                                                      // 6
    (_CHAR_NUMBER),                                                      // 7
    (_CHAR_NUMBER),                                                      // 8
    (_CHAR_NUMBER),                                                      // 9
    (_CHAR_REJECT),                                                      // :
    (_CHAR_REJECT),                                                      // ;
    (_CHAR_REJECT),                                                      // <
    (_CHAR_REJECT),                                                      // =
    (_CHAR_REJECT),                                                      // >
    (0),                                                                 // ?
    (0),                                                                 // @
    (_CHAR_CAPITAL),                                                     // A
    (_CHAR_CAPITAL),                                                     // B
    (_CHAR_CAPITAL),                                                     // C
    (_CHAR_CAPITAL),                                                     // D
    (_CHAR_CAPITAL),                                                     // E
    (_CHAR_CAPITAL),                                                     // F
    (_CHAR_CAPITAL),                                                     // G
    (_CHAR_CAPITAL),                                                     // H
    (_CHAR_CAPITAL),                                                     // I
    (_CHAR_CAPITAL),                                                     // J
    (_CHAR_CAPITAL),                                                     // K
    (_CHAR_CAPITAL),                                                     // L
    (_CHAR_CAPITAL),                                                     // M
    (_CHAR_CAPITAL),                                                     // N
    (_CHAR_CAPITAL),                                                     // O
    (_CHAR_CAPITAL),                                                     // P
    (_CHAR_CAPITAL),                                                     // Q
    (_CHAR_CAPITAL),                                                     // R
    (_CHAR_CAPITAL),                                                     // S
    (_CHAR_CAPITAL),                                                     // T
    (_CHAR_CAPITAL),                                                     // U
    (_CHAR_CAPITAL),                                                     // V
    (_CHAR_CAPITAL),                                                     // W
    (_CHAR_CAPITAL),                                                     // X
    (_CHAR_CAPITAL),                                                     // Y
    (_CHAR_CAPITAL),                                                     // Z
    (_CHAR_REJECT_NON_JAP),                                              // [
    (_CHAR_TERMINATE),                                                   // back slash
    (_CHAR_REJECT_NON_JAP),                                              // ]
    (0),                                                                 // ^
    (0),                                                                 // _
    (0),                                                                 // '
    (_CHAR_SMALL),                                                       // a
    (_CHAR_SMALL),                                                       // b
    (_CHAR_SMALL),                                                       // c
    (_CHAR_SMALL),                                                       // d
    (_CHAR_SMALL),                                                       // e
    (_CHAR_SMALL),                                                       // f
    (_CHAR_SMALL),                                                       // g
    (_CHAR_SMALL),                                                       // h
    (_CHAR_SMALL),                                                       // i
    (_CHAR_SMALL),                                                       // j
    (_CHAR_SMALL),                                                       // k
    (_CHAR_SMALL),                                                       // l
    (_CHAR_SMALL),                                                       // m
    (_CHAR_SMALL),                                                       // n
    (_CHAR_SMALL),                                                       // o
    (_CHAR_SMALL),                                                       // p
    (_CHAR_SMALL),                                                       // q
    (_CHAR_SMALL),                                                       // r
    (_CHAR_SMALL),                                                       // s
    (_CHAR_SMALL),                                                       // t
    (_CHAR_SMALL),                                                       // u
    (_CHAR_SMALL),                                                       // v
    (_CHAR_SMALL),                                                       // w
    (_CHAR_SMALL),                                                       // x
    (_CHAR_SMALL),                                                       // y
    (_CHAR_SMALL),                                                       // z
    (0),                                                                 // {
    (_CHAR_REJECT_NON_JAP),                                              // |
    (0),                                                                 // }
    (0),                                                                 // ~
};
#endif

/* =================================================================== */
/*                     global variable definitions                     */
/* =================================================================== */


/* =================================================================== */
/*                      local variable definitions                     */
/* =================================================================== */

#if defined SDCARD_SUPPORT
static UTDISK utDisks[SDCARDS] = {{0}};

    #if !defined SD_CONTROLLER_AVAILABLE && !defined NAND_FLASH_FAT
        #if defined _WINDOWS
            static int CommandTimeout = 10;
        #else
            static int CommandTimeout = 5000;                            // change depending on SPI speed
        #endif
    #endif

static int iMemoryOperation = _IDLE_MEMORY;
static int iMemoryState = SD_STATE_STARTING;

static unsigned long  ulClusterSectorCheck = 0;
static unsigned long  ulActiveFreeClusterCount = 0;
static UTASK_TASK     cluster_task = 0;
    #if defined HTTP_ROOT || defined FTP_ROOT
        static unsigned short usServerStates = 0;                        // {48}
        static unsigned short usServerResets = 0;
    #endif


    #if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0
typedef struct stUTMANAGED_FILE
{
    unsigned char   managed_mode;
    unsigned char   managed_owner;
    UTFILE         *utManagedFile;
}   UTMANAGED_FILE;

static UTMANAGED_FILE utManagedFiles[UTMANAGED_FILE_COUNT] = {{0}};

    #endif
#endif


#ifdef MANAGED_FILES                                                     // {21}

    #ifdef SUB_FILE_SIZE
        #define SUBFILE_WRITE  , ucSubFileInProgress
        #define SUB_FILE_ON    ,SUB_FILE_TYPE
    #else
        #define SUBFILE_WRITE
        #define SUB_FILE_ON
    #endif

static int iManagedMedia = 0;

static MANAGED_FILE managed_files[MANAGED_FILE_COUNT] = {{0}};


extern int uFileManagedDelete(int fileHandle)
{
	int iEraseStatus = fnEraseFlashSector(managed_files[fileHandle].managed_write, 0); // start single page erase
	if (iEraseStatus != MEDIA_BUSY) {
        managed_files[fileHandle].managed_write += iEraseStatus;         // erase of block started so increment the erase pointer
		if (managed_files[fileHandle].managed_size <= (unsigned int)iEraseStatus) {
			managed_files[fileHandle].managed_mode = 0;
            managed_files[fileHandle].managed_mode &= ~WAITING_DELETE;
			if (managed_files[fileHandle].ucParameters & AUTO_CLOSE) {
				managed_files[fileHandle].managed_owner = 0;
			}
			if (managed_files[fileHandle].fileOperationCallback) {
				managed_files[fileHandle].fileOperationCallback(fileHandle, 0);
			}
			return 0;                                                    // complete
		}
		else {
			managed_files[fileHandle].managed_size -= iEraseStatus;
		}
#ifdef TIME_SLICED_FILE_OPERATION
		if (managed_files[fileHandle].period != 0) {                     // if a page delete period rate is defined wait until this expires before continuing
			uTaskerGlobalMonoTimer(OWN_TASK, managed_files[fileHandle].period, (unsigned char)(fileHandle | _DELAYED_DELETE));
            return 1;
		}
#endif
	}
	managed_files[fileHandle].managed_mode |= WAITING_DELETE;            // either waiting to delete a busy page or waiting for a page to complete deletion
    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);                      // set to run again
    iManagedMedia = 1;                                                   // mark that this media type needs monitoring
    return 1;                                                            // delete not yet complete
}

extern int uOpenManagedFile(void *ptrFileName, UTASK_TASK owner_task, unsigned char ucMode)
{
    int i = 0;
    int iFree = 0;
    unsigned char *ptrFile;
	MAX_FILE_LENGTH file_length;
	if (ucMode & MANAGED_MEMORY_AREA) {
		MANAGED_MEMORY_AREA_BLOCK *ptrMemoryArea = (MANAGED_MEMORY_AREA_BLOCK *)ptrFileName;
		ptrFile = ptrMemoryArea->ptrStart;
		file_length = ptrMemoryArea->size;                               // size of the area
	}
	else {
        ptrFile = uOpenFile((CHAR *)ptrFileName);                        // get the file to be opened from the uFileSystem
		file_length = uGetFileLength(ptrFile);                           // file length of existing file
	}
    while (i < MANAGED_FILE_COUNT) {                                     // check whether this file is presently being protected
        if (managed_files[i].managed_owner != 0) {
            if (managed_files[i].managed_mode & (MANAGED_WRITE | MANAGED_LOCK | MANAGED_DELETE)) {
                if ((ptrFile >= managed_files[i].managed_start) && (ptrFile < (managed_files[i].managed_start + managed_files[i].managed_size))) {
                    return MANAGED_FILE_NO_ACCESS;                       // this file can presently not be accessed
                }
            }
        }
        else {
            iFree = (i + 1);                                             // free space found
        }
        i++;
    }
    if (iFree-- == 0) {
        return MANAGED_FILE_NO_FILE_HANDLE;                              // all managed file spaces are presently occupied
    }
    managed_files[iFree].managed_owner = owner_task;
    managed_files[iFree].managed_write = managed_files[iFree].managed_start = ptrFile;
    managed_files[iFree].managed_mode = ucMode;
    managed_files[iFree].managed_size = file_length;                     // file length of existing file
    if (ucMode & MANAGED_MEMORY_AREA) {
		MANAGED_MEMORY_AREA_BLOCK *ptrMemoryArea = (MANAGED_MEMORY_AREA_BLOCK *)ptrFileName;
		managed_files[iFree].ucParameters = ptrMemoryArea->ucParameters;
		managed_files[iFree].fileOperationCallback = ptrMemoryArea->fileOperationCallback;
		managed_files[iFree].period = ptrMemoryArea->period;
		if (ptrMemoryArea->ucParameters & AUTO_DELETE) {                 // if a delete is to be performed immediately call the function
			uFileManagedDelete(iFree);                                   // start deletion
		}
	}
    return iFree;                                                        // return file handle
}

static void fnManagedMediaCheck(void)
{
    int i = MANAGED_FILE_COUNT;
    while (i > 0) {                                                      // check all managed files
        i--;
        if (managed_files[i].managed_owner != 0) {                       // for each one that is owned
            if (managed_files[i].managed_mode & WAITING_DELETE) {        // waiting to start of continue delete operation
                uFileManagedDelete(i);
			}
        }
    }
}

#endif



#if defined SDCARD_SUPPORT || defined MANAGED_FILES

// Mass storage task
//
extern void fnMassStorage(TTASKTABLE *ptrTaskTable)
{
    #ifdef _WINDOWS
    static int iFormatCount;
    #endif
    unsigned char ucInputMessage[HEADER_LENGTH];                         // reserve space for receiving messages
    #if defined SDCARD_SUPPORT
    int iActionResult = 0;
        #if !defined NAND_FLASH_FAT                                      // {29}
    unsigned char ucData[18];
    unsigned char ucResult = 0;
        #endif
    #endif
    #if defined MANAGED_FILES
    if (iManagedMedia != 0) {
        iManagedMedia = 0;
        fnManagedMediaCheck();
    }
    #endif
    #if defined SDCARD_SUPPORT
    if (iMemoryOperation & _READING_MEMORY) {                            // reading
        if ((iActionResult = utReadDisk(&utDisks[0], 0)) == CARD_BUSY_WAIT) {
            return;                                                      // still reading so keep waiting
        }
    }
    if (!(iMemoryOperation & _INITIALISED_MEMORY)) {                     // if initialisation in progress
        #if defined UTFAT_WRITE && defined UTFAT_FORMATTING              // {43}
        static unsigned long ulFatSectors;
        static unsigned long ulFAT32size;
        static unsigned char ucFatCopies;
        #endif
        switch (iMemoryState) {                                          // perform the initialisation state-event process
        case SD_STATE_STARTING:
        case SD_STATE_WAIT_CARD:
        #if defined NAND_FLASH_FAT                                       // {29}
            if (utDisks[0].ptrSectorData == 0) {
                fnInitNAND();                                            // initialise the NAND interface
                utDisks[0].ptrSectorData = SDCARD_MALLOC(512);           // allocate a buffer which will contain data read from the present sector (this should be long word aligned and possibly in a specific mememroy are in case DMA operation is to be used)
                utDisks[0].ulSD_sectors = (USER_AREA_BLOCKS * NAND_PAGES_IN_BLOCK);
                uTaskerStateChange(OWN_TASK, UTASKER_GO);                // switch to polling mode of operation
            }
            if (fnGetBlankBlocks() != 0) {                               // start the process of reading all blocks to get their present state
                return;                                                  // repeat until the complete NAND flash has been checked for blank blocks
            }
            uTaskerStateChange(OWN_TASK, UTASKER_STOP);
        #else
            #ifdef SD_CONTROLLER_AVAILABLE                               // {5}
            fnInitSDCardInterface();                                     // HW interface initialisation
            ucSEND_CSD_CMD9[1] = ucSELECT_CARD_CMD7[1] = ucAPP_CMD_CMD55[1] = ucSET_BUS_WIDTH_CMD6[1] = 0;  // start with zeroed RCA address
            ucSEND_CSD_CMD9[2] = ucSELECT_CARD_CMD7[2] = ucAPP_CMD_CMD55[2] = ucSET_BUS_WIDTH_CMD6[2] = 0;
            iMemoryState = SD_STATE_STABILISING;                         // move to stabilisation delay state
            uTaskerMonoTimer(OWN_TASK, T_POWER_STABILISE, E_POWER_STABILISED); // wait until SD card power stabilised
            #else
            INITIALISE_SPI_SD_INTERFACE();                               // initialise the SPI interface to the card
                #if defined SDCARD_DETECT_INPUT_POLL || defined SDCARD_DETECT_INPUT_INTERRUPT // {52}
                    #if defined SDCARD_DETECT_INPUT_INTERRUPT 
            fnPrepareDetectInterrupt();                                  // prepare interrupt detection of SD card presence
                    #endif
            if (!(SDCARD_DETECTION())) {                                 // if card is not detected immediately abort mounting process
                fnInitialisationError(0);                                // try to remount the card
                break;
            }
                    #if defined _WINDOWS
            else {
                SD_card_state(SDCARD_INSERTED, SDCARD_REMOVED);
            }
                    #endif
                #endif
            SET_SD_DI_CS_HIGH();                                         // prepare chip select and DI ready for the initialisation
            POWER_UP_SD_CARD();                                          // power up if necessary
            iMemoryState = SD_STATE_STABILISING;                         // move to stabilisation delay state
            ENABLE_SPI_SD_OPERATION();                                   // dummy to solve an AVR32 GCC optimising problem
            uTaskerMonoTimer(OWN_TASK, T_POWER_STABILISE, E_POWER_STABILISED); // wait until SD card power stabilised
            #endif
            break;

        case SD_STATE_STABILISING:                                       // delay after applying power to the SD-card
            #ifdef SD_CONTROLLER_AVAILABLE                               // {5}
            POWER_UP_SD_CARD();                                          // apply power
            iMemoryState = SD_STATE_GO_IDLE;                             // prepare to communicate
            uTaskerMonoTimer(OWN_TASK, T_POWER_STABILISE, E_POWER_STABILISED); // wait until SD card power stabilised
            break;
            #else
            {                                                            // send at least 74 clocks to the SD-card
                int i = 10;
                while (i--) {                                            // set the SD card to native command mode by sending 80 clocks
                    WRITE_SPI_CMD(0xff);                                 // write dummy tx
                    WAIT_TRANSMISSON_END();                              // wait until transmission complete
                    READ_SPI_DATA();                                     // read 10 dummy bytes from the interface in order to generate 80 clock pulses on the interface (at least 74 needed)
                }
                SET_SD_CS_LOW();                                         // assert the chip select line to the SD-card ready to start the initialisation sequence
                iMemoryState = SD_STATE_GO_IDLE;                         // move to next state
                SET_SD_CARD_MODE();                                      // allow final mode in case the DIN line had to be pulled up during native mode sequence
            }                                                            // fall through
            #endif
        case SD_STATE_GO_IDLE:
            if ((iActionResult = fnSendSD_command(ucGO_IDLE_STATE_CMD0, &ucResult, 0)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                ucResult = 0;                                            // set error since the result is not as expected
            }
            if (R1_IN_IDLE_STATE != ucResult) {                          // no valid card detected so disable power and try again after a delay
                fnInitialisationError(0);                                // the card is no present or is behaving incorrectly - stop and try again later
                break;
            }                                                            // SD card must return the idle state to be able to continue
                                                                         // in the idle state the card accepts only commands 0, 1, ACMD41 and 58
            iMemoryState = SD_STATE_IF_COND;                             // fall through
        case SD_STATE_IF_COND:                                           // it is mandatory for a host compliant to Physical Spec. Version 2.00 to send the CMD8 to retrieve non-supported voltage range
            if ((iActionResult = fnSendSD_command(ucIF_COND_CMD8, &ucResult, ucData)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            if (ucResult == SDC_CARD_VERSION_2PLUS) {                    // version 2 or higher
                if (ucData[3] == CHECK_PATTERN) {
                    if (ucData[2] == VOLTAGE_2_7__3_6) {                 // check whether the card can operate between 2.7V and 3.6V
                        iMemoryState = SD_STATE_APP_CMD55_CMD41;         // now poll the SD card until it accept the application command 42
                        uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);  // run again
                        break;
                    }
                    fnMemoryDebugMsg("SD-card voltage error\r\n");
                }
                else {
                    fnInitialisationError(0);                            // no valid response
                    break;
                }
            }
            else {                                                       // version 1 or MMC type
                fnMemoryDebugMsg("SD-card V1 or MMC - not supported!\r\n");
            }
            fnInitialisationError(1);                                    // not supported
            break;
        case SD_STATE_APP_CMD55_CMD41:
            if ((iActionResult = fnSendSD_command(ucAPP_CMD_CMD55, &ucResult, 0)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            if (ucResult > R1_IN_IDLE_STATE) {
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // try again
                break;
            }
            iMemoryState = SD_STATE_APP_CMD55_CMD41_2;                   // fall through to send the application command
        case SD_STATE_APP_CMD55_CMD41_2:            
            #ifdef SD_CONTROLLER_AVAILABLE                               // {5} this command returns OCR result in SD card mode
            if ((iActionResult = fnSendSD_command(ucSEND_OP_COND_ACMD_CMD41, &ucResult, ucData)) != UTFAT_SUCCESS)
            #else
            if ((iActionResult = fnSendSD_command(ucSEND_OP_COND_ACMD_CMD41, &ucResult, 0)) != UTFAT_SUCCESS) 
            #endif
            {
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            #ifdef SD_CONTROLLER_AVAILABLE                               // {5}
            if (!(ucData[0] & 0x80))                                     // check card busy bit
            #else
            if (ucResult != 0) 
            #endif
            {
                iMemoryState = SD_STATE_APP_CMD55_CMD41;                 // loop back
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // try again
                break;
            }
            iMemoryState = SD_STATE_OCR;                                 // fall through to issue CMD58
        case SD_STATE_OCR:
            #ifndef SD_CONTROLLER_AVAILABLE                              // OCR value is returned to the CMD41 on SD card interface {5}
            if ((iActionResult = fnSendSD_command(ucREAD_OCR_CMD58, &ucResult, ucData)) != UTFAT_SUCCESS) { // get card capacity information
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            #endif
            fnMemoryDebugMsg("SD-card V2 - ");
            if (ucData[0] & HIGH_CAPACITY_SD_CARD_MEMORY) {              // check the CCS bit
                utDisks[0].usDiskFlags = HIGH_CAPACITY_SD_CARD;
                fnMemoryDebugMsg("High Capacity\r\n");
            }
            else {
                utDisks[0].usDiskFlags = 0;                              // {27}
                fnMemoryDebugMsg("standard\r\n");
            }
            #ifdef SD_CONTROLLER_AVAILABLE                               // {5} SC card mode sequency requires the CID to be read and then an RCA address to be set
            iMemoryState = SD_STATE_GET_INFO;                            // fall through to issue CMD2 and read card information
        case SD_STATE_GET_INFO:
            if ((iActionResult = fnSendSD_command(ucSEND_CID_CMD2, &ucResult, ucData)) != UTFAT_SUCCESS) { // get card information
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            iMemoryState = SD_STATE_SET_ADDRESS;
        case SD_STATE_SET_ADDRESS:
            if ((iActionResult = fnSendSD_command(ucSET_REL_ADD_CMD3, &ucResult, ucData)) != UTFAT_SUCCESS) { // set relative address
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            if ((ucData[2] & (CURRENT_CARD_STATUS_MASK | SD_CARD_READY_FOR_DATA)) == (CURRENT_STATE_IDENT | SD_CARD_READY_FOR_DATA)) {
                ucSEND_CSD_CMD9[1] = ucSELECT_CARD_CMD7[1] = ucAPP_CMD_CMD55[1] = ucSET_BUS_WIDTH_CMD6[1] = ucData[0];  // save the published RCA address
                ucSEND_CSD_CMD9[2] = ucSELECT_CARD_CMD7[2] = ucAPP_CMD_CMD55[2] = ucSET_BUS_WIDTH_CMD6[2] = ucData[1];
            }
            #endif
            iMemoryState = SD_STATE_GET_SECTORS;                         // fall through to issue CMD9 and read card specific data
        case SD_STATE_GET_SECTORS:
            if ((iActionResult = fnSendSD_command(ucSEND_CSD_CMD9, &ucResult, ucData)) != UTFAT_SUCCESS) { // get card capacity information
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            uMemcpy(utDisks[0].utFileInfo.ucCardSpecificData, &ucData[2 - SD_CONTROLLER_SHIFT], 16); // back up the card specific data since it can be of interest later
            if (ucData[2 - SD_CONTROLLER_SHIFT] & HIGH_CAPACITY_SD_CARD_MEMORY) { // high capacity
                utDisks[0].ulSD_sectors = (((ucData[9 - SD_CONTROLLER_SHIFT] << 16) + (ucData[10 - SD_CONTROLLER_SHIFT] << 8)  + ucData[11 - SD_CONTROLLER_SHIFT]) + 1);// SD version 2 assumed
                utDisks[0].ulSD_sectors *= 1024;                         // the number of sectors on the SD card
            }
            else {                                                       // standard capacity
                utDisks[0].ulSD_sectors = ((((ucData[8 - SD_CONTROLLER_SHIFT] & 0x03) << 10) + (ucData[9 - SD_CONTROLLER_SHIFT] << 2) + (ucData[10 - SD_CONTROLLER_SHIFT] >> 6)) + 1);// SD version 2 assumed
                utDisks[0].ulSD_sectors *= (1 << ((((ucData[11 - SD_CONTROLLER_SHIFT] & 0x03) << 1) + (ucData[12 - SD_CONTROLLER_SHIFT] >> 7)) + 2)); // the number of 512 byte sectors on the SD card
                if ((ucData[7 - SD_CONTROLLER_SHIFT] & 0x0f) == 0x0a) {  // 1024 byte block length indicates 2G card
                    utDisks[0].ulSD_sectors *= 2;
                }
            }
            fnMemoryDebugMsg("\r\n");
            if (utDisks[0].ptrSectorData == 0) {
                utDisks[0].ptrSectorData = SDCARD_MALLOC(512);           // {24} allocate a buffer which will contain data read from the present sector (this should be long word aligned and possibly in a specific memory in case DMA operation is to be used)
            }
            #ifdef SD_CONTROLLER_AVAILABLE                               // {5}
            iMemoryState = SD_SELECT_CARD;
        case SD_SELECT_CARD:                                             // select the card before reading/writing
            if ((iActionResult = fnSendSD_command(ucSELECT_CARD_CMD7, &ucResult, ucData)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
          //if ((ucData[2] & (CURRENT_CARD_STATUS_MASK | SD_CARD_READY_FOR_DATA)) == (CURRENT_STATE_STBY | SD_CARD_READY_FOR_DATA)) {
          //}
            iMemoryState = SD_SET_INTERFACE_PREPARE;
        case SD_SET_INTERFACE_PREPARE:                                   // set the interface so that 4 bit data mode is used - first set command mode
            if ((iActionResult = fnSendSD_command(ucAPP_CMD_CMD55, &ucResult, 0)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            if (ucResult > R1_IN_IDLE_STATE) {
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // try again
                break;
            }
            iMemoryState = SD_SET_INTERFACE;                             // fall through to send the application command
        case SD_SET_INTERFACE:
            if ((iActionResult = fnSendSD_command(ucSET_BUS_WIDTH_CMD6, &ucResult, ucData)) != UTFAT_SUCCESS) { // set relative address
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            iMemoryState = SD_SET_BLOCK_LENGTH;
            SET_SPI_SD_INTERFACE_FULL_SPEED();                           // speed up the SPI interface since initialisation is complete (as well as data bus width)
        case SD_SET_BLOCK_LENGTH:
            if ((iActionResult = fnSendSD_command(ucSET_BLOCK_LENGTH_CMD16, &ucResult, ucData)) != UTFAT_SUCCESS) { // set relative address
                if (iActionResult == CARD_BUSY_WAIT) {
                    return;                                              // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(0);                                // the card is behaving incorrectly - stop and try again later
                break;
            }
            iMemoryState = SD_MOUNTING_1;
            uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);              // schedule again to attempt mounting
            utDisks[0].ucDriveNumber = 0;
            #else
            iMemoryState = SD_MOUNTING_1;
            uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);              // schedule again to attempt mounting
            utDisks[0].ucDriveNumber = 0;
            SET_SD_CS_HIGH();
            SET_SPI_SD_INTERFACE_FULL_SPEED();                           // speed up the SPI interface since initialisation is complete
            #endif
            break;
        #endif
        case SD_MOUNTING_1:                                              // start mounting
            utDisks[0].ulPresentSector = 0;                              // the first sector that will be accessed
            if ((iActionResult = ut_read_disk(&utDisks[0])) == CARD_BUSY_WAIT) { // read boot sector from disk
                return;                                                  // read is taking time to complete so quit for the moment
            }
            iMemoryState = SD_MOUNTING_2;
        case SD_MOUNTING_2:
            if (ERROR_SECTOR_INVALID == iActionResult) {
                fnCardNotFormatted();
                break;
            }
            else {
                iMemoryState = SD_MOUNTING_3;
                if (iActionResult != 0) {
                    fnInitialisationError(0);                            // the card is behaving incorrectly - stop and try again later
                    break;
                }
                else {
                    EXTENDED_BOOT_RECORD *ptrExtendedBootSector = (EXTENDED_BOOT_RECORD *)utDisks[0].ptrSectorData; // check to see whether partitions are defined rather than it being a boot record
                    BOOT_SECTOR_FAT32 *ptrBootSector = (BOOT_SECTOR_FAT32 *)utDisks[0].ptrSectorData;
                    if ((ptrExtendedBootSector->EBR_partition_table[0].partition_type != 0) && (ptrBootSector->bs_common.BS_FilSysType[0] != 'F')) {
                        utDisks[0].ulPresentSector = ((ptrExtendedBootSector->EBR_partition_table[0].start_sector[3] << 24) + (ptrExtendedBootSector->EBR_partition_table[0].start_sector[2] << 16) + (ptrExtendedBootSector->EBR_partition_table[0].start_sector[1] << 8) + ptrExtendedBootSector->EBR_partition_table[0].start_sector[0]);
                        if ((iActionResult = ut_read_disk(&utDisks[0])) ==  CARD_BUSY_WAIT) { // read boot sector from disk
                            return;                                      // read is taking time to complete so quit for the moment
                        }
                    }
                    else {
                        utDisks[0].ulPresentSector = 0;                  // the boot sector is 0
                    }
                }
            }                                                            // fall-through
        case SD_MOUNTING_3:
            if (ERROR_SECTOR_INVALID == iActionResult) {
                fnCardNotFormatted();
                break;
            }
            else {
                int iFAT32 = 0;
                BOOT_SECTOR_FAT32 *ptrBootSector;
                unsigned long  ulTotalSections;
                unsigned long  ulFatSize;
                unsigned long  ulFirstDataSector;
                unsigned long  ulCountofClusters;
                unsigned short BPB_RsvdSecCnt;
                unsigned short BPB_FSInfo;
                unsigned short BPB_BytesPerSec;
                unsigned char  BPB_SecPerClus;
                BOOT_SECT_COM  *ptr_common;

                ptrBootSector = (BOOT_SECTOR_FAT32 *)utDisks[0].ptrSectorData; // the buffer content is a boot sector
                BPB_BytesPerSec = (ptrBootSector->boot_sector_bpb.BPB_BytesPerSec[1] << 8);
                ulFatSize = ((ptrBootSector->boot_sector_bpb.BPB_FATSz16[1] << 8) + ptrBootSector->boot_sector_bpb.BPB_FATSz16[0]);
                if (ulFatSize == 0) {                                    // FAT32 will indicate the size in the BPB_FATSz32 field instead
                    ulFatSize = ((ptrBootSector->BPB_FATSz32[3] << 24) + (ptrBootSector->BPB_FATSz32[2] << 16) + (ptrBootSector->BPB_FATSz32[1] << 8) + ptrBootSector->BPB_FATSz32[0]);
                    iFAT32 = 1;                                          // FAT32, irresepctive of the disk's size
                }
                ulTotalSections = ((ptrBootSector->boot_sector_bpb.BPB_TotSec16[1] << 8) + ptrBootSector->boot_sector_bpb.BPB_TotSec16[0]);
                if (ulTotalSections == 0) {                              // FAT32 will indicate the size in the BPB_FATSz32 field instead
                    ulTotalSections = ((ptrBootSector->boot_sector_bpb.BPB_TotSec32[3] << 24) + (ptrBootSector->boot_sector_bpb.BPB_TotSec32[2] << 16) + (ptrBootSector->boot_sector_bpb.BPB_TotSec32[1] << 8) + ptrBootSector->boot_sector_bpb.BPB_TotSec32[0]);
                }
                utDisks[0].utFAT.ulFatSize = ulFatSize;                  // the sectors in a single FAT
                utDisks[0].utFAT.ucNumberOfFATs = ptrBootSector->boot_sector_bpb.BPB_NumFATs; // the number of FAT copies
                ulFatSize *= ptrBootSector->boot_sector_bpb.BPB_NumFATs; // the complete number of sections occupied by all FATs
                BPB_SecPerClus = ptrBootSector->boot_sector_bpb.BPB_SecPerClus;
                utDisks[0].utFAT.ucSectorsPerCluster = BPB_SecPerClus;
                BPB_RsvdSecCnt = ((ptrBootSector->boot_sector_bpb.BPB_RsvdSecCnt[1] << 8) + ptrBootSector->boot_sector_bpb.BPB_RsvdSecCnt[0]);
                utDisks[0].utFAT.ulFAT_start = (utDisks[0].ulPresentSector + BPB_RsvdSecCnt); // the boot sector plus reserved sectors
                if ((BPB_BytesPerSec < 512) || (BPB_SecPerClus == 0) || ((BPB_SecPerClus * BPB_BytesPerSec) > (32*1024))) {
                    fnMemoryDebugMsg("Malformed - ");
                    fnCardNotFormatted();
                    break;
                }
                utDisks[0].usDiskFlags &= ~DISK_UNFORMATTED;
                utDisks[0].usDiskFlags |= DISK_FORMATTED;
                utDisks[0].utFAT.usBytesPerSector = BPB_BytesPerSec;
                ulFirstDataSector = (BPB_RsvdSecCnt + ulFatSize);
                ulCountofClusters = ((ulTotalSections - ulFirstDataSector)/ptrBootSector->boot_sector_bpb.BPB_SecPerClus);
                if ((ulCountofClusters < 65525) && (iFAT32 == 0)) {      // not FAT32
            #ifdef UTFAT16
                    utDisks[0].usDiskFlags |= DISK_FORMAT_FAT16;
                    utDisks[0].ulDirectoryBase = 1;
                    ptr_common = &((BOOT_SECTOR_FAT12_FAT16 *)utDisks[0].ptrSectorData)->bs_common;
            #else
                    fnMemoryDebugMsg("NOT FAT32- ");
                    fnCardNotFormatted();
                    break;
            #endif
                }
                else {
                    BPB_FSInfo = ((ptrBootSector->BPB_FSInfo[1] << 8) + ptrBootSector->BPB_FSInfo[0]);
                    utDisks[0].ulDirectoryBase = ((ptrBootSector->BPB_RootClus[3] << 24) + (ptrBootSector->BPB_RootClus[2] << 16) + (ptrBootSector->BPB_RootClus[1] << 8) + ptrBootSector->BPB_RootClus[0]); // root directory start cluster
                    utDisks[0].ulPresentSector = utDisks[0].utFileInfo.ulInfoSector = (utDisks[0].ulPresentSector + BPB_FSInfo); // sector location of structure
                    ptr_common = &ptrBootSector->bs_common;
                }
                utDisks[0].utFAT.ulClusterCount = ((ulTotalSections - BPB_RsvdSecCnt - ulFatSize)/BPB_SecPerClus); // total cluster count
                utDisks[0].ulLogicalBaseAddress = (utDisks[0].utFAT.ulFAT_start + ulFatSize); // data start sector (logical block address)
                utDisks[0].ulVirtualBaseAddress = (utDisks[0].ulLogicalBaseAddress - (utDisks[0].ulDirectoryBase * utDisks[0].utFAT.ucSectorsPerCluster));
            #ifdef UTFAT16
                if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                    utDisks[0].ulVirtualBaseAddress += (32 - 1);         // fixed 16 kbyte root folder
                }
            #endif
                uMemcpy(utDisks[0].cVolumeLabel, ptr_common->BS_VolLab, sizeof(utDisks[0].cVolumeLabel)); // {30}
                iMemoryState = SD_MOUNTING_4;
                if ((iActionResult = ut_read_disk(&utDisks[0])) ==  CARD_BUSY_WAIT) { // read boot sector from disk
                    return;                                              // read is taking time to complete so quit for the moment
                }
            }                                                            // fall through
        case SD_MOUNTING_4:                                              // optional step for FAT32 SD-cards
            if ((iActionResult == 0) && (!(utDisks[0].usDiskFlags & DISK_FORMAT_FAT16))) { // if the information sector was valid
                INFO_SECTOR_FAT32 *ptrInfo = (INFO_SECTOR_FAT32 *)utDisks[0].ptrSectorData;
                if ((ptrInfo->FSI_TrailSig[3] == 0xaa) && (ptrInfo->FSI_TrailSig[2] == 0x55) &&  // check that the FSInfo sector is valid
                  (ptrInfo->FSI_LeadSig[3] == 0x41) && (ptrInfo->FSI_LeadSig[2] == 0x61) && (ptrInfo->FSI_LeadSig[1] == 0x52) && (ptrInfo->FSI_LeadSig[0] == 0x52) &&
                  (ptrInfo->FSI_StrucSig[3] == 0x61) && (ptrInfo->FSI_StrucSig[2] == 0x41) && (ptrInfo->FSI_StrucSig[1] == 0x72) && (ptrInfo->FSI_StrucSig[0] == 0x72)) {
                    utDisks[0].utFileInfo.ulFreeClusterCount = ((ptrInfo->FSI_Free_Count[3] << 24) + (ptrInfo->FSI_Free_Count[2] << 16) + (ptrInfo->FSI_Free_Count[1] << 8) + ptrInfo->FSI_Free_Count[0]);
                    utDisks[0].utFileInfo.ulNextFreeCluster = ((ptrInfo->FSI_Nxt_Free[3] << 24) + (ptrInfo->FSI_Nxt_Free[2] << 16) + (ptrInfo->FSI_Nxt_Free[1] << 8) + ptrInfo->FSI_Nxt_Free[0]);
                    if (utDisks[0].utFileInfo.ulNextFreeCluster <= (utDisks[0].utFileInfo.ulFreeClusterCount - 2)) { // check that the information is valid
                        utDisks[0].usDiskFlags |= FSINFO_VALID;
                    }
                }
            }
            fnMemoryDebugMsg("Disk D mounted");
            if (GET_SDCARD_WP_STATE()) {                                 // {34}
                utDisks[0].usDiskFlags |= (DISK_MOUNTED | WRITE_PROTECTED_SD_CARD); // the write protected disk is now ready for use
                fnMemoryDebugMsg(" (WP)");
            }
            else {
                utDisks[0].usDiskFlags |= DISK_MOUNTED;                  // the disk is now ready for use (not write protected)
            }
            fnMemoryDebugMsg("\r\n");
            iMemoryOperation |= _INITIALISED_MEMORY;
            iMemoryState = SD_STATE_READY;
        #if defined T_CHECK_CARD_REMOVAL                                 // {50}
            uTaskerMonoTimer(OWN_TASK, T_CHECK_CARD_REMOVAL, E_CHECK_CARD_REMOVAL); // poll the SD card to detect removal
        #endif
        #if defined _WINDOWS                                             // {49}
            if (utDisks[0].usDiskFlags & WRITE_PROTECTED_SD_CARD) {
                SD_card_state((SDCARD_MOUNTED | SDCARD_WR_PROTECTED | SDCARD_INSERTED), SDCARD_REMOVED);
            }
            else {
                SD_card_state((SDCARD_MOUNTED | SDCARD_INSERTED), SDCARD_REMOVED);
            }
            if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                SD_card_state((SDCARD_FORMATTED_16), (SDCARD_FORMATTED_32));
            }
            else {
                SD_card_state((SDCARD_FORMATTED_32), (SDCARD_FORMATTED_16));
            }
        #endif
            break;
        #if defined T_CHECK_CARD_REMOVAL                                 // {50}
        case SD_STATE_CHECKING_DISK_REMOVAL:
            #if defined SDCARD_DETECT_INPUT_POLL                         // {52}
            if (!(SDCARD_DETECTION())) {                                 // if card has been removed start remounting process
                fnInitialisationError(0);                                // try to remount the card
            }
            else {
                iMemoryState = SD_STATE_READY;
                iMemoryOperation |= (_INITIALISED_MEMORY);
                uTaskerMonoTimer(OWN_TASK, T_CHECK_CARD_REMOVAL, E_CHECK_CARD_REMOVAL); // poll the SD card to detect removal
            }
            #else
            if (UTFAT_SUCCESS != utReadDisk(&utDisks[0], utDisks[0].ulPresentSector)) { // {53} read presently selected buffer to verify that the card is still responding
                fnInitialisationError(0);                                // try to remount the card
            }
            else {
                iMemoryState = SD_STATE_READY;
                iMemoryOperation |= (_INITIALISED_MEMORY);
                uTaskerMonoTimer(OWN_TASK, T_CHECK_CARD_REMOVAL, E_CHECK_CARD_REMOVAL); // poll the SD card to detect removal
            }
            #endif
            break;
        #endif
        #if defined UTFAT_WRITE && defined UTFAT_FORMATTING
        case SD_STATE_FORMATTING_DISK_1:                                 // start by creating a partition
            {
                EXTENDED_BOOT_RECORD *ptrExtendedBootRecord = (EXTENDED_BOOT_RECORD *)utDisks[0].ptrSectorData;
                unsigned long ulPartitionSize = (utDisks[0].ulSD_sectors - BOOT_SECTOR_LOCATION);
                uMemset(ptrExtendedBootRecord, 0, 512);                  // ensure content is blank
                ptrExtendedBootRecord->EBR_partition_table[0].starting_cylinder = 2;// fixed values
                ptrExtendedBootRecord->EBR_partition_table[0].starting_head     = 0x0c;
                ptrExtendedBootRecord->EBR_partition_table[0].partition_type    = 0x0b;
                ptrExtendedBootRecord->EBR_partition_table[0].ending_cylinder   = 0x38;
                ptrExtendedBootRecord->EBR_partition_table[0].ending_head       = 0xf8;
                ptrExtendedBootRecord->EBR_partition_table[0].ending_sector     = 0xb8;
                ptrExtendedBootRecord->EBR_partition_table[0].start_sector[0]   = BOOT_SECTOR_LOCATION;
                ptrExtendedBootRecord->EBR_partition_table[0].partition_size[0] = (unsigned char)(ulPartitionSize);
                ptrExtendedBootRecord->EBR_partition_table[0].partition_size[1] = (unsigned char)(ulPartitionSize >> 8);
                ptrExtendedBootRecord->EBR_partition_table[0].partition_size[2] = (unsigned char)(ulPartitionSize >> 16);
                ptrExtendedBootRecord->EBR_partition_table[0].partition_size[3] = (unsigned char)(ulPartitionSize >> 24);
                ptrExtendedBootRecord->ucCheck55 = 0x55;
                ptrExtendedBootRecord->ucCheckAA = 0xaa;
                if ((iActionResult = utCommitSector(&utDisks[0], 0)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        return;                                          // write is taking time to complete so quit for the moment
                    }
                    iMemoryState = SD_NOT_FORMATTED;
                    break;
                }
            #ifdef _WINDOWS
                iFormatCount = 0;
            #endif
                iMemoryState = SD_STATE_FORMATTING_DISK_2;               // fall through
            }
        case SD_STATE_FORMATTING_DISK_2:                                 // add the boot sector
            {
                BOOT_SECTOR_FAT32 *ptrBootSector = (BOOT_SECTOR_FAT32 *)utDisks[0].ptrSectorData;
            #ifdef UTFAT16
                BOOT_SECTOR_FAT12_FAT16 *ptrBootSector_16 = (BOOT_SECTOR_FAT12_FAT16 *)utDisks[0].ptrSectorData;
            #endif
                BOOT_SECT_COM *ptr_common;
                unsigned long ulPartitionSize;
                uMemset(ptrBootSector, 0, 508);                          // ensure content is blank (without overwriting 0xaa550000)
                ptrBootSector->boot_sector_bpb.BS_jmpBoot[0]     = 0xeb; // fixed values
                ptrBootSector->boot_sector_bpb.BS_jmpBoot[1]     = 0x58;
                ptrBootSector->boot_sector_bpb.BS_jmpBoot[2]     = 0x90;
                uStrcpy(ptrBootSector->boot_sector_bpb.BS_OEMName, "MSDOS5.0");
                utDisks[0].utFAT.usBytesPerSector                = BYTES_PER_SECTOR;
                ptrBootSector->boot_sector_bpb.BPB_BytesPerSec[0]= (unsigned char)(BYTES_PER_SECTOR);
                ptrBootSector->boot_sector_bpb.BPB_BytesPerSec[1]= (unsigned char)(BYTES_PER_SECTOR >> 8);
                ptrBootSector->boot_sector_bpb.BPB_SecPerTrk[0]  = 63;
                ptrBootSector->boot_sector_bpb.BPB_NumHeads[0]   = 0xff;                
                ptrBootSector->boot_sector_bpb.BPB_Media         = FIXED_MEDIA;                
            #ifdef UTFAT16
                if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                    unsigned long ulSectorCnt = (unsigned long)(utDisks[0].ulSD_sectors); // total count of sectors on volume
                    ptrBootSector->boot_sector_bpb.BS_jmpBoot[1] = 0x3c;
                    ptrBootSector->boot_sector_bpb.BPB_RsvdSecCnt[0] = 8;
                    ptrBootSector->boot_sector_bpb.BPB_NumFATs   = 1;
                    ptrBootSector->boot_sector_bpb.BPB_RootEntCnt[1] = (unsigned char)(512 >> 8);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec16[0]   = (unsigned char)(ulSectorCnt);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec16[1]   = (unsigned char)(ulSectorCnt >> 8);
                    ulSectorCnt = ((utDisks[0].ulSD_sectors - 8)/257);   // size of FAT16 in sectors
                    if (ulSectorCnt > 0xfe) {
                        ulSectorCnt = (0xfe + 8 + ((0xfe * 512)/sizeof(unsigned short)));
                        ptrBootSector->boot_sector_bpb.BPB_TotSec16[0]   = (unsigned char)(ulSectorCnt);
                        ptrBootSector->boot_sector_bpb.BPB_TotSec16[1]   = (unsigned char)(ulSectorCnt >> 8);
                        ulSectorCnt = 0xfe;                              // limit size of FAT16
                    }
                    ptrBootSector->boot_sector_bpb.BPB_FATSz16[0]    = (unsigned char)(ulSectorCnt);
                    ptrBootSector->boot_sector_bpb.BPB_FATSz16[1]    = (unsigned char)(ulSectorCnt >> 8);
                    utDisks[0].utFAT.ucSectorsPerCluster = 1;            // one sector per cluster assumed since only small systems expected with FAT16
                    ptrBootSector->ucCheck55 = 0x55;                     // mark that the sector is valid
                    ptrBootSector->ucCheckAA = 0xaa;
                    utDisks[0].ulLogicalBaseAddress = (8 + ulSectorCnt);
                    ptr_common = &ptrBootSector_16->bs_common;
                    uMemcpy(ptr_common->BS_FilSysType, "FAT16   ", 8);
                    ulFAT32size = ulSectorCnt;
                    utDisks[0].ulVirtualBaseAddress = (utDisks[0].ulLogicalBaseAddress + (32 - 1));
                }
                else {
            #endif
                    ptrBootSector->boot_sector_bpb.BPB_HiddSec[0]    = BOOT_SECTOR_LOCATION;
                    ulPartitionSize = (utDisks[0].ulSD_sectors - BOOT_SECTOR_LOCATION);
                    ptrBootSector->boot_sector_bpb.BPB_NumFATs       = NUMBER_OF_FATS;
                    ptrBootSector->boot_sector_bpb.BPB_RsvdSecCnt[0] = RESERVED_SECTION_COUNT;
                    ptrBootSector->boot_sector_bpb.BPB_TotSec32[0]   = (unsigned char)(ulPartitionSize);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec32[1]   = (unsigned char)(ulPartitionSize >> 8);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec32[2]   = (unsigned char)(ulPartitionSize >> 16);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec32[3]   = (unsigned char)(ulPartitionSize >> 24);
                    if (ulPartitionSize <= 532480) {                         // disks up to 260MB
                        utDisks[0].utFAT.ucSectorsPerCluster = 1;
                    }
                    else if (ulPartitionSize <= 16777216) {                  // disks up to 8GB
                        utDisks[0].utFAT.ucSectorsPerCluster = 8;
                    }
                    else if (ulPartitionSize <= 33554432) {                  // disks up to 16GB
                        utDisks[0].utFAT.ucSectorsPerCluster = 16;
                    }
                    else if (ulPartitionSize <= 67108864) {                  // disks up to 32GB
                        utDisks[0].utFAT.ucSectorsPerCluster = 32;
                    }
                    else {                                                   // greater than 32GB
                        utDisks[0].utFAT.ucSectorsPerCluster = 64;
                    }                    
                    ulFAT32size = (((256 * utDisks[0].utFAT.ucSectorsPerCluster) + NUMBER_OF_FATS)/2);
                    ulFAT32size = (((ulPartitionSize - RESERVED_SECTION_COUNT) + (ulFAT32size - 1)) / ulFAT32size);
                    ptrBootSector->BPB_FATSz32[0]   = (unsigned char)(ulFAT32size);
                    ptrBootSector->BPB_FATSz32[1]   = (unsigned char)(ulFAT32size >> 8);
                    ptrBootSector->BPB_FATSz32[2]   = (unsigned char)(ulFAT32size >> 16);
                    ptrBootSector->BPB_FATSz32[3]   = (unsigned char)(ulFAT32size >> 24);
                    ptrBootSector->BPB_RootClus[0]  = 2;
                    ptrBootSector->BPB_FSInfo[0]    = 1;
                    ptrBootSector->BPB_BkBootSec[0] = BACKUP_ROOT_SECTOR;
                    utDisks[0].ulLogicalBaseAddress = (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT + (ulFAT32size * NUMBER_OF_FATS));
                    ptr_common = &ptrBootSector->bs_common;
                    uMemcpy(ptr_common->BS_FilSysType, "FAT32   ", 8);
                    utDisks[0].ulVirtualBaseAddress = utDisks[0].ulLogicalBaseAddress - (2 * utDisks[0].utFAT.ucSectorsPerCluster);
            #ifdef UTFAT16
                }
            #endif
                ptrBootSector->boot_sector_bpb.BPB_SecPerClus = utDisks[0].utFAT.ucSectorsPerCluster;
                ptr_common->BS_DrvNum  = 0x80;
                ptr_common->BS_BootSig = 0x29;
            #ifdef RANDOM_NUMBER_GENERATOR
                ptr_common->BS_VolID[0] = (unsigned char)fnRandom();     // can also be generated by combining present data and time
                ptr_common->BS_VolID[1] = (unsigned char)fnRandom();
                ptr_common->BS_VolID[2] = (unsigned char)fnRandom();
                ptr_common->BS_VolID[3] = (unsigned char)fnRandom();
            #else
                ptr_common->BS_VolID[0] = 1;
                ptr_common->BS_VolID[1] = 2;
                ptr_common->BS_VolID[2] = 3;
                ptr_common->BS_VolID[3] = 4;
            #endif
                if (utDisks[0].cVolumeLabel[0] == 0) {
                    uMemcpy(ptr_common->BS_VolLab, "NO NAME    ", 11);
                }
                else {
                    int i = 0;
                    while (i < 11) {
                        if ((ptr_common->BS_VolLab[i] = utDisks[0].cVolumeLabel[i]) == 0) {
                            while (i < 11) {
                                ptr_common->BS_VolLab[i++] = ' ';
                            }
                        }
                        i++;
                    }
                }
              //ptrBootSector->ucCheck55 = 0x55;                         // already exists in buffer
              //ptrBootSector->ucCheckAA = 0xaa;
                iMemoryState = SD_STATE_FORMATTING_DISK_3;               // fall through
            }
        case SD_STATE_FORMATTING_DISK_3:
            #ifdef UTFAT16
            if (!(utDisks[0].usDiskFlags & DISK_FORMAT_FAT16)) {
            #endif
                if ((iActionResult = utCommitSector(&utDisks[0], BOOT_SECTOR_LOCATION)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        return;                                              // write is taking time to complete so quit for the moment
                    }
                    iMemoryState = SD_NOT_FORMATTED;
                    break;
                }
            #ifdef UTFAT16
            }
            #endif
            iMemoryState = SD_STATE_FORMATTING_DISK_4;                   // fall through
        case SD_STATE_FORMATTING_DISK_4:
            {
                unsigned long ulLocation = (BOOT_SECTOR_LOCATION + BACKUP_ROOT_SECTOR);
            #ifdef UTFAT16
                if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                    ulLocation = 0;                                      // FAT16 has a single boot sector at the start of the memory area
                }
            #endif
                if ((iActionResult = utCommitSector(&utDisks[0], ulLocation)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        return;                                          // write is taking time to complete so quit for the moment
                    }
                    iMemoryState = SD_NOT_FORMATTED;
                    break;
                }
                else {
                    DIR_ENTRY_STRUCTURE_FAT32 *ptrVolumeEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)utDisks[0].ptrSectorData;
                    uMemset(utDisks[0].ptrSectorData, 0, 512);
                    uMemcpy(ptrVolumeEntry->DIR_Name, utDisks[0].cVolumeLabel, 11); //add the volume ID
                    ptrVolumeEntry->DIR_Attr = DIR_ATTR_VOLUME_ID;
                    ulFatSectors = 0;
                    iMemoryState = SD_STATE_FORMATTING_DISK_5;
                }
            }
        case SD_STATE_FORMATTING_DISK_5:                                 // ensure that the first boot cluster is blank (with volume ID)
            if (ulFatSectors < utDisks[0].utFAT.ucSectorsPerCluster) {
                if ((iActionResult = utCommitSector(&utDisks[0], (ulFatSectors + utDisks[0].ulLogicalBaseAddress))) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        return;                                          // write is taking time to complete so quit for the moment
                    }
                    iMemoryState = SD_NOT_FORMATTED;
                    break;
                }
                ulFatSectors++;
                uMemset(utDisks[0].ptrSectorData, 0, 512);
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // schedule again to continue with next sector
                break;
            }
            iMemoryState = SD_STATE_FORMATTING_DISK_6;
            ulFatSectors = 0;
            ucFatCopies = 1;
        case SD_STATE_FORMATTING_DISK_6:
          //uMemset(utDisks[0].ptrSectorData, 0, 512);
            #ifdef UTFAT16
            if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                uMemcpy(utDisks[0].ptrSectorData, ucEmptyFAT16, sizeof(ucEmptyFAT16));
            }
            else {
                uMemcpy(utDisks[0].ptrSectorData, ucEmptyFAT32, sizeof(ucEmptyFAT32));
            }
            #else
            uMemcpy(utDisks[0].ptrSectorData, ucEmptyFAT32, sizeof(ucEmptyFAT32));
            #endif
            iMemoryState = SD_STATE_FORMATTING_DISK_7;                   // fall through
        case SD_STATE_FORMATTING_DISK_7:
            {
                unsigned long ulLocation = (ulFatSectors + (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT));
            #ifdef UTFAT16
                if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                    ulLocation = 8;
                }
            #endif
                if ((iActionResult = utCommitSector(&utDisks[0], ulLocation)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        return;                                          // write is taking time to complete so quit for the moment
                    }
                    iMemoryState = SD_NOT_FORMATTED;
                    break;
                }
                uMemset(utDisks[0].ptrSectorData, 0, sizeof(ucEmptyFAT32));
                ulFatSectors++;
                iMemoryState = SD_STATE_FORMATTING_DISK_8;               // fall through
            }
        case SD_STATE_FORMATTING_DISK_8:
            if (ulFatSectors >= (ulFAT32size * ucFatCopies)) {
                if (ucFatCopies < NUMBER_OF_FATS) {
                    ucFatCopies++;
                    iMemoryState = SD_STATE_FORMATTING_DISK_6;           // next FAT copy
                    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);      // schedule again to continue with next sector
                    break;
                }
            #ifdef UTFAT_FULL_FORMATTING
                if (utDisks[0].usDiskFlags & DISK_FORMAT_FULL) {
                    utDisks[0].usDiskFlags &= ~DISK_FORMAT_FULL;
                    ulFatSectors += (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT);
                #ifdef UTFAT16
                    if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                        ulFatSectors -= (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT - 8);
                    }
                #endif
                    iMemoryState = SD_STATE_FORMATTING_DISK_8A;          // the FAT has been cleared but now clear out all data sectors too
                }
                else {
                    iMemoryState = SD_STATE_FORMATTING_DISK_9;           // all FAT copies have been initialised
                }
            #else
                iMemoryState = SD_STATE_FORMATTING_DISK_9;               // all FAT copies have been initialised
            #endif
            }
            else {
                unsigned long ulLocation = (ulFatSectors + (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT));
            #ifdef UTFAT16
                if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                    ulLocation -= (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT - 8);
                }
            #endif
                if ((iActionResult = utCommitSector(&utDisks[0], ulLocation)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        return;                                          // write is taking time to complete so quit for the moment
                    }
                    iMemoryState = SD_NOT_FORMATTED;
                    break;
                }
                ulFatSectors++;
                if ((ulFatSectors % 256) == 0) {
                    fnMemoryDebugMsg("*");
                }
            #ifdef _WINDOWS
                if (++iFormatCount == 0x100) {                           // accelerate simulation
                    ulFatSectors = (ulFAT32size * ucFatCopies);
                    iFormatCount = 0;
                }
            #endif
            }
            uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);              // schedule again to continue with next sector
            break;
            #ifdef UTFAT_FULL_FORMATTING
        case SD_STATE_FORMATTING_DISK_8A:                                // delete disk data
            if (ulFatSectors >= utDisks[0].ulSD_sectors) {
                iMemoryState = SD_STATE_FORMATTING_DISK_9;               // all disk data has been cleaned
            }
            else {
                if ((iActionResult = utCommitSector(&utDisks[0], ulFatSectors)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        return;                                          // write is taking time to complete so quit for the moment
                    }
                    iMemoryState = SD_NOT_FORMATTED;
                    break;
                }
                ulFatSectors++;
                if ((ulFatSectors % 256) == 0) {
                    fnMemoryDebugMsg("X");
                }
            #ifdef _WINDOWS
                if (++iFormatCount == 0x200) {                           // accelerate simulation
                    ulFatSectors = (utDisks[0].utFAT.ulFAT_start + utDisks[0].utFAT.ulFatSize + (utDisks[0].utFAT.ulClusterCount * (utDisks[0].utFAT.usBytesPerSector/512)));
                    iFormatCount = 0;
                }
            #endif
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // schedule again to continue with next sector
                break;
            }
            #endif
        case SD_STATE_FORMATTING_DISK_9:                                 // finally create the FAT32 info record
            {
                INFO_SECTOR_FAT32 *ptrInfoSector;
              //unsigned long ulFreeCount = ((ulFAT32size * (512/sizeof(signed long))) - 3); // total free clusters minus default root cluster and 2 unused ones
                unsigned long ulFreeCount;
            #ifdef UTFAT16
                if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                    fnMemoryDebugMsg("Disk D formatted FAT16\r\n");
                    iMemoryState = SD_MOUNTING_1;
                    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // schedule again to mount the disk which has just been formatted
                    break;
                }
            #endif
                ptrInfoSector = (INFO_SECTOR_FAT32 *)utDisks[0].ptrSectorData;
                ulFreeCount = ((utDisks[0].ulSD_sectors - RESERVED_SECTION_COUNT - BOOT_SECTOR_LOCATION - (NUMBER_OF_FATS * ulFAT32size))/utDisks[0].utFAT.ucSectorsPerCluster); // {1}
                ptrInfoSector->FSI_LeadSig[3] = 0x41;
                ptrInfoSector->FSI_LeadSig[2] = 0x61;
                ptrInfoSector->FSI_LeadSig[1] = 0x52;
                ptrInfoSector->FSI_LeadSig[0] = 0x52;
                ptrInfoSector->FSI_StrucSig[3] = 0x61;
                ptrInfoSector->FSI_StrucSig[2] = 0x41;
                ptrInfoSector->FSI_StrucSig[1] = 0x72;
                ptrInfoSector->FSI_StrucSig[0] = 0x72;                                 
                fnAddInfoSect(ptrInfoSector, (ulFreeCount - 1), 3);      // one cluster occupied by root directory by default and first useable cluster number is 3
                ptrInfoSector->FSI_StrucSig[3] = 0x61;
                ptrInfoSector->FSI_StrucSig[2] = 0x41;
                ptrInfoSector->FSI_StrucSig[1] = 0x72;
                ptrInfoSector->FSI_StrucSig[0] = 0x72;
                ptrInfoSector->FSI_TrailSig[3] = 0xaa;
                ptrInfoSector->FSI_TrailSig[2] = 0x55;
                if ((iActionResult = utCommitSector(&utDisks[0], (BOOT_SECTOR_LOCATION + 1))) != 0) { // write info sector
                    if (iActionResult == CARD_BUSY_WAIT) {
                        return;                                          // write is taking time to complete so quit for the moment
                    }
                    iMemoryState = SD_NOT_FORMATTED;
                    break;
                }
                iMemoryState = SD_STATE_FORMATTING_DISK_10;
                utDisks[0].usDiskFlags |= FSINFO_VALID;
        case SD_STATE_FORMATTING_DISK_10:                                // backup
                if ((iActionResult = utCommitSector(&utDisks[0], (BOOT_SECTOR_LOCATION + 1 + BACKUP_ROOT_SECTOR))) != 0) { // write info sector backup
                    if (iActionResult == CARD_BUSY_WAIT) {
                        return;                                          // write is taking time to complete so quit for the moment
                    }
                    iMemoryState = SD_NOT_FORMATTED;
                    break;
                }
                fnMemoryDebugMsg("Disk D formatted\r\n");
                iMemoryState = SD_MOUNTING_1;
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // schedule again to mount the disk which has just been formatted
            }
            break;
        #endif                                                           // end #if defined UTFAT_WRITE && defined UTFAT_FORMATTING
        }
    }
    else if (iMemoryOperation & _COUNTING_CLUSTERS) {                    // cluster counting in progress
        if (ulClusterSectorCheck >= utDisks[0].utFAT.ulFAT_start) {
            int iRepetitions = 10;                                       // read maximum 10 sectors at a time
            unsigned long ulSectorContent[512/sizeof(signed long)];
            if (ulClusterSectorCheck < (iRepetitions + utDisks[0].utFAT.ulFAT_start)) {
                iRepetitions = (ulClusterSectorCheck - utDisks[0].utFAT.ulFAT_start);
            }
            while (iRepetitions-- >= 0) {
                if ((utReadDiskSector(&utDisks[0], ulClusterSectorCheck--, ulSectorContent)) == UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
        #ifdef UTFAT16
                    if (utDisks[0].usDiskFlags & DISK_FORMAT_FAT16) {
                        int i = 0;                                       // count free FAT16 clusters
                        while (i < (512/sizeof(signed long))) {
                            if (ulSectorContent[i] == 0) {
                                ulActiveFreeClusterCount +=2;
                            }
                            else if (((unsigned short)(ulSectorContent[i]) == 0) || ((unsigned short)(ulSectorContent[i] >> 16) == 0)) {
                                ulActiveFreeClusterCount++;
                            }
                            i++;
                        }
                    }
                    else {
         #endif
                        int i = 0;                                       // count free FAT32 clusters
                        while (i < (512/sizeof(signed long))) {
                            if (ulSectorContent[i++] == 0) {
                                ulActiveFreeClusterCount++;
                            }
                        }
        #ifdef UTFAT16
                    }
        #endif
                }
            }
            uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);              // schedule to run again
        }
        else {
            iMemoryOperation &= ~_COUNTING_CLUSTERS;
            utDisks[0].utFileInfo.ulFreeClusterCount = ulActiveFreeClusterCount;
            fnInterruptMessage(cluster_task, UTFAT_OPERATION_COMPLETED); // inform that the cluster count has completed
            cluster_task = 0;
        }
    }
    #endif                                                               // end #if defined SDCARD_SUPPORT

    while (fnRead(ptrTaskTable->TaskID, ucInputMessage, HEADER_LENGTH)) {// check task input queue
        switch (ucInputMessage[MSG_SOURCE_TASK]) {
        case TIMER_EVENT:
    #ifdef NAND_FLASH_FAT
            if (E_CLEAN_SPARE == ucInputMessage[MSG_TIMER_EVENT]) {
                fnCleanNAND();                                           // if there are area of the NAND Flash marked to be cleaned do the work as a background operation
                break;
            }
    #endif
    #if defined T_CHECK_CARD_REMOVAL
            if (E_CHECK_CARD_REMOVAL == ucInputMessage[MSG_TIMER_EVENT]) { // time to check whether the card is responding
                iMemoryState = SD_STATE_CHECKING_DISK_REMOVAL;
                iMemoryOperation &= ~(_INITIALISED_MEMORY);
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // schedule the task to perform a check to see whether the card is responding (or has been removed)
            }
    #endif
    #if defined MANAGED_FILES && defined TIME_SLICED_FILE_OPERATION
			if (ucInputMessage[MSG_TIMER_EVENT] & _DELAYED_DELETE) {
                uFileManagedDelete(ucInputMessage[MSG_TIMER_EVENT] & ~(_DELAYED_DELETE));
                break;
			}
    #endif
            break;

    #if defined SDCARD_DETECT_INPUT_INTERRUPT                            // {52}
        case INTERRUPT_EVENT:                                            // new state of SD card detection
            if (E_SDCARD_DETECTION_CHANGE == ucInputMessage[MSG_INTERRUPT_EVENT]) {
                if (SDCARD_DETECTION()) {                                // card has been inserted so try to mount it
                                                                         // do nothing in this case since the interrupt event will have already started mounting attempt
        #if defined _WINDOWS
                    SD_card_state(SDCARD_INSERTED, SDCARD_REMOVED);
        #endif
                }
                else {                                                   // card has been removed so stop operation
                    fnInitialisationError(0);
                }
            }
            break;
    #endif
        }
    }
}
#endif                                                                   // end #if defined SDCARD_SUPPORT || defined MANAGED_FILES


#if defined SDCARD_SUPPORT
static void fnCardNotFormatted(void)
{
    utDisks[0].usDiskFlags |= DISK_UNFORMATTED;                          // {27}
    iMemoryState = SD_NOT_FORMATTED;
    fnMemoryDebugMsg("Non-formatted SD-card\r\n");
    #if defined _WINDOWS
    SD_card_state(SDCARD_MOUNTED, (SDCARD_FORMATTED_32 | SDCARD_FORMATTED_16)); // {49}
    #endif
}

// Interrupt initisalisation sequence on error, remove power and start next try after a delay
//
static void fnInitialisationError(int iNotSupported)
{
    SET_SD_CS_HIGH();
    POWER_DOWN_SD_CARD();
    iMemoryState = SD_STATE_WAIT_CARD;                                   // wait for next check of valid card
    uTaskerMonoTimer(OWN_TASK, T_NEXT_CHECK, E_POLL_SD_CARD);            // try again later
    if (iNotSupported != 0) {
        utDisks[0].usDiskFlags = DISK_TYPE_NOT_SUPPORTED;
    #if defined _WINDOWS
        SD_card_state(0, (SDCARD_FORMATTED_32 | SDCARD_FORMATTED_16));   // {49}
    #endif
    }
    else {
        fnMemoryDebugMsg("SD-card not responding\r\n");
        utDisks[0].usDiskFlags = DISK_NOT_PRESENT;                       // {27}
        iMemoryOperation = _IDLE_MEMORY;                                 // {52}
    #if defined _WINDOWS
        SD_card_state(0, (SDCARD_INSERTED | SDCARD_MOUNTED | SDCARD_FORMATTED_32 | SDCARD_FORMATTED_16 | SDCARD_WR_PROTECTED)); // {52}
    #endif
    }
}

#if !defined NAND_FLASH_FAT                                              // {29}
static const unsigned char *fnCreateCommand(unsigned char ucCommand, unsigned long ulValue)
{
    static unsigned char ucCommandBuffer[6];                             // space for a permanent command
    ucCommandBuffer[0] = ucCommand;
    ucCommandBuffer[1] = (unsigned char)(ulValue >> 24);
    ucCommandBuffer[2] = (unsigned char)(ulValue >> 16);
    ucCommandBuffer[3] = (unsigned char)(ulValue >> 8);
    ucCommandBuffer[4] = (unsigned char)(ulValue);
    ucCommandBuffer[5] = 0;                                              // dummy CRC
    return (const unsigned char *)ucCommandBuffer;
}
#endif

#if !defined SD_CONTROLLER_AVAILABLE && !defined NAND_FLASH_FAT          // {5}{29} SPI interface is local - SD card interface uses HW specific external routines
// Read a sector from SD card into the specified data buffer
//
static int fnGetSector(unsigned char *ptrBuf)
{
    unsigned char ucResponse;
    int iLength = 512;
    // this may block so a timer may be required to protect it (100ms?)
    do {
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
    } while ((ucResponse = READ_SPI_DATA()) == 0xff);
    if (ucResponse != 0xfe) {
        return UTFAT_DISK_READ_ERROR;                                    // error
    }
    do {
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        *ptrBuf++ = READ_SPI_DATA();
    } while (--iLength);
    WRITE_SPI_CMD(0xff);                                                 // read and discard two CRC bytes
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    WRITE_SPI_CMD(0xff);
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    return UTFAT_SUCCESS;                                                // read successfully terminated
}

    #if defined UTFAT_WRITE                                              // {43}
// Write present sector with buffer data
//
static int fnPutSector(unsigned char *ptrBuf)
{
    #define DATA_TOKEN 0xfe
    unsigned char ucResponse;
    int iLength = 512;
    // this may block so a timer may be required to protect it (100ms?)
    do {
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
    } while ((ucResponse = READ_SPI_DATA()) != 0xff);
    WRITE_SPI_CMD(DATA_TOKEN);                                           // transmit data byte
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    READ_SPI_DATA();                                                     // clear the receiver
    do {
        WRITE_SPI_CMD(*ptrBuf);                                          // transmit data byte
        ptrBuf++;
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        READ_SPI_DATA();                                                 // clear the receiver
    } while (--iLength);
    WRITE_SPI_CMD(0xff);                                                 // send two dummy CRC bytes
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    READ_SPI_DATA();
    WRITE_SPI_CMD(0xff);
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    READ_SPI_DATA();
    WRITE_SPI_CMD(0xff);                                                 // send two dummy CRC bytes
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    if ((ucResponse & 0x1f) != 0x05) {
        return UTFAT_DISK_WRITE_ERROR;
    }
    return UTFAT_SUCCESS;                                                // write successfully completed
}
    #endif

// Read a specified amount of data from present SD card sector into the specified data buffer (usStart and usStop are offset from start of sector and avoid other data outseide of thsi range being overwritted)
//
static int fnReadPartialSector(unsigned char *ptrBuf, unsigned short usStart, unsigned short usStop)
{
    unsigned char ucResponse;
    unsigned short usLoopCounter = 0;
    // this may block so a timer may be required to protect it (100ms?)
    do {
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
    } while ((ucResponse = READ_SPI_DATA()) == 0xff);
    if (ucResponse != 0xfe) {
        fnMemoryDebugMsg("ERR-1!!!\r\n");
        return 0;                                                        // error
    }
    while (usLoopCounter < usStart) {
        WRITE_SPI_CMD(0xff);
        usLoopCounter++;
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        READ_SPI_DATA();                                                 // discard unwanted data at the start of the sector
    }
    while (usLoopCounter < usStop) {
        WRITE_SPI_CMD(0xff);
        usLoopCounter++;
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        *ptrBuf++ = READ_SPI_DATA();
    }
    while (usLoopCounter < 512) {
        WRITE_SPI_CMD(0xff);
        usLoopCounter++;
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        READ_SPI_DATA();                                                 // discard any additional data, but always read a full 512 byte block
    }
    WRITE_SPI_CMD(0xff);                                                 // read and discard two CRC bytes
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    WRITE_SPI_CMD(0xff);
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    return UTFAT_SUCCESS;                                                // read terminated
}

// Send a command to the SD-card and return a result, plus optional returned arguments
//
static int fnSendSD_command(const unsigned char ucCommand[6], unsigned char *ucResult, unsigned char *ptrReturnData)
{
    static int iCommandYieldCount = 0;
    static int iCommandState = 0;
    int iRtn = 0;

    if (iCommandYieldCount > CommandTimeout) {                           // the present access is taking too long - quit with SD card error
        fnMemoryDebugMsg("TIMEOUT!!!\r\n");
        iRtn = ERROR_CARD_TIMEOUT;
    }
    else {
        switch (iCommandState) {
        case 0:
            if (fnWaitSD_ready(20) != 0) {                               // poll up to 20 times before yielding
                iCommandYieldCount++;                                    // monitor the maximum delay
                uTaskerStateChange(OWN_TASK, UTASKER_GO);                // switch to polling mode of operation
                return CARD_BUSY_WAIT;
            }  
            for ( ; iCommandState < 6; iCommandState++) {                // command content is always 6 bytes in length
                WRITE_SPI_CMD(ucCommand[iCommandState]);                 // send the command and arguments
                WAIT_TRANSMISSON_END();                                  // wait until transmission complete
                READ_SPI_DATA();                                         // read to clear the flag
            }                                                            // fall through intentional after sending the command
            if (iCommandYieldCount != 0) {                               // {12}
                uTaskerStateChange(OWN_TASK, UTASKER_STOP);              // switch to event mode if operation is continuing
                iCommandYieldCount = 0;
            }
        case 6:
            do {
                if (iCommandState++ > 26) {
                    iCommandYieldCount++;                                // monitor the maximum delay
                    iCommandState = 6;
                    uTaskerStateChange(OWN_TASK, UTASKER_GO);            // switch to polling mode of operation
                    return CARD_BUSY_WAIT;                               // poll up to 20 times before yielding
                }
                WRITE_SPI_CMD(0xff);                                     // send idle line
                WAIT_TRANSMISSON_END();                                  // wait until transmission complete
                *ucResult = READ_SPI_DATA();                             // read result byte
            } while (*ucResult & SD_CARD_BUSY);                          // poll the card until it is no longer indicating busy and returns the value
            if (ptrReturnData != 0) {                                    // if the caller requests data, read it here
                int iReturnLength;
                if (ucCommand[0] == SEND_CSD_CMD9) {                     // exception requiring 18 bytes
                    iReturnLength = 18;
                }
                else {
                    iReturnLength = 4;
                }
                for (iCommandState = 0; iCommandState < iReturnLength; iCommandState++) {
                    WRITE_SPI_CMD(0xff);                                 // send the command and arguments
                    WAIT_TRANSMISSON_END();                              // wait until transmission complete
                    *ptrReturnData++ = READ_SPI_DATA();
                }
            }
            break;
        }
    }
    if (iCommandYieldCount != 0) {
        uTaskerStateChange(OWN_TASK, UTASKER_STOP);                      // switch to event mode of operation since write has completed
        iCommandYieldCount = 0;
    }
    iCommandState = 0;
    return iRtn;
}

// Wait until the SD card is ready by reading until 0xff is returned
//
static int fnWaitSD_ready(int iMaxWait)
{
    do {
        if (iMaxWait-- == 0) {
            return 1;                                                    // maximum wait attempts executed
        }
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
    } while (READ_SPI_DATA() != 0xff);                                   // 0xff expected when the device is ready
    return 0;                                                            // the device is now ready
}
#endif

#if defined NAND_FLASH_FAT                                               // {29}
// Read a single sector from the disk - the sector number is specified by ulSectorNumber
//
static int utReadDisk(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    switch (iMemoryOperation & _READING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            if (fnReadNANDsector(ulSectorNumber, 0, ptr_utDisk->ptrSectorData, 512) != 0) {
                fnMemoryDebugMsg(" - ECC check failed\r\n");             // this message can be ignored if the page is blank since it will not have a valid ECC
            }
            ptr_utDisk->ulPresentSector = ulSector;                      // update the sector which is presently valid

            iMemoryOperation &= ~_READING_MEMORY;                        // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Read a part of the specified sector to the buffer (avoiding overwriting all buffer content)
//
static int utReadPartialDiskData(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf, unsigned short usOffset, unsigned short usLength)
{
    static unsigned long ulSector;
    switch (iMemoryOperation & _READING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            unsigned char ucTemp[512];
            if (fnReadNANDsector(ulSectorNumber, 0, ucTemp, 512) != 0) {
                fnMemoryDebugMsg(" - ECC check failed\r\n");             // this message can be ignored if the page is blank since it will not have a valid ECC
            }
            uMemcpy(ptrBuf, &ucTemp[usOffset], usLength);
            iMemoryOperation &= ~_READING_MEMORY;                        // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Read a single, complete sector from the disk to the specified buffer
//
static int utReadDiskSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf)
{
    static unsigned long ulSector;
    switch (iMemoryOperation & _READING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            if (ptr_utDisk->usDiskFlags & DISK_TEST_MODE) {
                _fnReadNANDdata((unsigned short)(ulSectorNumber/NAND_PAGES_IN_BLOCK), (unsigned char)(ulSectorNumber%NAND_PAGES_IN_BLOCK), 0, ptrBuf, 528);
            }
            else {
                if (fnReadNANDsector(ulSectorNumber, 0, ptrBuf, 512) != 0) {
                    fnMemoryDebugMsg(" - ECC check failed\r\n");         // this message can be ignored if the page is blank since it will not have a valid ECC
                }
            }
            iMemoryOperation &= ~_READING_MEMORY;                        // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}
#else
// Read a single sector from the disk - the sector number is specified by ulSectorNumber
//
static int utReadDisk(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation & _READING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            unsigned char ucResult;
            while ((iActionResult = fnSendSD_command(fnCreateCommand(READ_SINGLE_BLOCK_CMD17, ulSector), &ucResult, 0)) == CARD_BUSY_WAIT) {}
            if (iActionResult < 0) {
                SET_SD_CS_HIGH();
                iMemoryOperation &= ~_READING_MEMORY;                    // read operation has failed
                return iActionResult;
            }
            if (ucResult == 0) {
                fnGetSector(ptr_utDisk->ptrSectorData);                  // start reading single sector to disk buffer
                if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
                    ulSector /= 512;                                     // convert back to sector number from byte offset
                }
                ptr_utDisk->ulPresentSector = ulSector;                  // update the sector which is presently valid
            }
            SET_SD_CS_HIGH();
            iMemoryOperation &= ~_READING_MEMORY;                        // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Read a part of the specified sector to the buffer (avoiding overwriting all buffer content)
//
static int utReadPartialDiskData(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf, unsigned short usOffset, unsigned short usLength)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation & _READING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            unsigned char ucResult;
            while ((iActionResult = fnSendSD_command(fnCreateCommand(READ_SINGLE_BLOCK_CMD17, ulSector), &ucResult, 0)) == CARD_BUSY_WAIT) {}
            if (iActionResult < 0) {
                SET_SD_CS_HIGH();
                iMemoryOperation &= ~_READING_MEMORY;                    // read operation has completed
                return iActionResult;
            }
            if (ucResult == 0) {
                fnReadPartialSector(ptrBuf, usOffset, (unsigned short)(usOffset + usLength));// start reading a sector direct to buffer
            }
            SET_SD_CS_HIGH();
            iMemoryOperation &= ~_READING_MEMORY;                        // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Read a single, complete sector from the disk to the specified buffer
//
static int utReadDiskSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation & _READING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            unsigned char ucResult;
            while ((iActionResult = fnSendSD_command(fnCreateCommand(READ_SINGLE_BLOCK_CMD17, ulSector), &ucResult, 0)) == CARD_BUSY_WAIT) {}
            if (iActionResult < 0) {
                SET_SD_CS_HIGH();
                iMemoryOperation &= ~_READING_MEMORY;                    // read operation has completed
                return iActionResult;
            }
            if (ucResult == 0) {
                fnGetSector(ptrBuf);                                     // read a single sector to the buffer
            }
            SET_SD_CS_HIGH();
            iMemoryOperation &= ~_READING_MEMORY;                        // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}
#endif

// Read one sector from the disk specified by the drive number
//
static int ut_read_disk(UTDISK *ptr_utDisk)
{
    int iRtn = utReadDisk(ptr_utDisk, ptr_utDisk->ulPresentSector);      // read specified sector to buffer
    if (iRtn == UTFAT_SUCCESS) {                                         // read from SD media was successful and we expect to have the boot sector in the buffer
        BOOT_SECTOR_FAT32 *ptrBootSector = (BOOT_SECTOR_FAT32 *)ptr_utDisk->ptrSectorData;
        if ((ptrBootSector->ucCheck55 != 0x55) || (ptrBootSector->ucCheckAA != 0xaa)) {
            return ERROR_SECTOR_INVALID;                                 // the sector data is no valid
        }
    }
    return iRtn;                                                         // {51}
}

#ifdef UTFAT_LFN_READ
static int fnMatchLongName(const CHAR **ptrLongFileName, const CHAR *ptrMatchStart, LFN_ENTRY_STRUCTURE_FAT32 *ptrLFN_entry)
{
    int i = 13;                                                          // there are maximum 13 characters in an entry
    unsigned char ucCharacter = 0;
    unsigned char ucMatchCharacter;
    if ((ptrLFN_entry->LFN_Attribute & DIR_ATTR_MASK) != DIR_ATTR_LONG_NAME) {
        if (ptrLFN_entry->LFN_EntryNumber == 0) {                        // end of directory entries
            return END_DIRECTORY_ENTRIES;
        }
        return MATCH_CONTINUE;                                           // this is not a long file name entry so ignore it
    }
    while (--i >= 0) {
        switch (i) {
        case 12:
            ucCharacter = ptrLFN_entry->LFN_Name_12;
            break;
        case 11:
            ucCharacter = ptrLFN_entry->LFN_Name_11;
            break;
        case 10:
            ucCharacter = ptrLFN_entry->LFN_Name_10;
            break;
        case 9:
            ucCharacter = ptrLFN_entry->LFN_Name_9;
            break;
        case 8:
            ucCharacter = ptrLFN_entry->LFN_Name_8;
            break;
        case 7:
            ucCharacter = ptrLFN_entry->LFN_Name_7;
            break;
        case 6:
            ucCharacter = ptrLFN_entry->LFN_Name_6;
            break;
        case 5:
            ucCharacter = ptrLFN_entry->LFN_Name_5;
            break;
        case 4:
            ucCharacter = ptrLFN_entry->LFN_Name_4;
            break;
        case 3:
            ucCharacter = ptrLFN_entry->LFN_Name_3;
            break;
        case 2:
            ucCharacter = ptrLFN_entry->LFN_Name_2;
            break;
        case 1:
            ucCharacter = ptrLFN_entry->LFN_Name_1;
            break;
        case 0:
            ucCharacter = ptrLFN_entry->LFN_Name_0;
            break;
        }
        if (ucCharacter == 0xff) {                                       // past end of entry
            continue;
        }
        if (ucCharacter == 0x00) {                                       // null-terminator
            continue;
        }
        ucMatchCharacter = *(--(*ptrLongFileName));
        if (ucCharacter != ucMatchCharacter) {                           // check the match, working backward, with pre-decrement
            if ((ucCharacter >= 'A') && (ucCharacter <= 'Z')) {
                ucCharacter += ('a' - 'A');
            }
            else if ((ucCharacter >= 'a') && (ucCharacter <= 'z')) {
                ucMatchCharacter += ('a' - 'A');
            }
            else {
                return MATCH_FALSE;
            }
            if (ucCharacter != ucMatchCharacter) {                       // last chance for case-insensitive match
                return MATCH_FALSE;
            }
        }
    }
    if ((ptrLFN_entry->LFN_EntryNumber & ~0x40) == 1) {                  // the end of the long file name
        if (*ptrLongFileName == ptrMatchStart) {
            return MATCH_SUCCESSFUL;
        }
        return MATCH_FALSE;
    }
    return MATCH_CONTINUE;
}
#endif


// Load a sector if the present sector content is not alread valid
//
static int fnLoadSector(UTDISK *ptr_utDisk, unsigned long ulSector)
{
    if (/*(ulSector != 0) && */(ptr_utDisk->ulPresentSector != ulSector)) { // {18} if the requested sector is not the already loaded one
#ifdef UTFAT_WRITE
        if (ptr_utDisk->usDiskFlags & WRITEBACK_BUFFER_FLAG) {           // if changes have been made to the sector since its read it must be written back
            while (utCommitSector(ptr_utDisk, ptr_utDisk->ulPresentSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
            ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;
        }
#endif
        ptr_utDisk->ulPresentSector = ulSector;
        return (utReadDisk(ptr_utDisk, ulSector));                       // read the new sector
    }
    return UTFAT_SUCCESS;
}

// Read a part of a sector directly to a buffer (length is maximum 512 bytes)
//
static int fnLoadPartialData(UTDISK *ptr_utDisk, unsigned long ulSector, void *ptrBuf, unsigned short usOffset, unsigned short usLength)
{
    if ((ptr_utDisk->ulPresentSector == ulSector) && (!(ptr_utDisk->usDiskFlags & WRITEBACK_BUFFER_FLAG))) { // if the requested sector is already loaded and hasn't been modified
        uMemcpy(ptrBuf, &ptr_utDisk->ptrSectorData[usOffset], (usLength - usOffset)); // the data is already in the local sector copy so copy it directly to the caller's buffer
    }
    else {                                                               // the normal case is to load directly
        return (utReadPartialDiskData(ptr_utDisk, ulSector, ptrBuf, usOffset, usLength)); // read data from disk without changing the present sector
    }
    return UTFAT_SUCCESS;
}


static int fnNextSector(UTDISK *ptr_utDisk, FILE_LOCATION *ptr_location)
{
    ptr_location->ulSector++;                                            // next sector
    if (((ptr_location->ulSector - ptr_utDisk->ulLogicalBaseAddress) % ptr_utDisk->utFAT.ucSectorsPerCluster) == 0) { // check whether the present cluster end has been reached
        unsigned long ulClusterSector;
        unsigned long ulCluster;
#ifdef UTFAT16
        unsigned short usCluster;
        if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
            if (ptr_location->ulSector == (ptr_utDisk->ulVirtualBaseAddress + 2)) { // root folder end reached
                return UTFAT_FAT16_ROOT_FOLDER_EXHAUSTED;                // FAT16 root folder exhausted
            }
            else if (ptr_location->ulSector < (ptr_utDisk->ulVirtualBaseAddress + 2)) { // in the FAT16 root folder
                return UTFAT_SUCCESS;                                    // this never grows so its FAT16 entry always remains the same
            }
            ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 8)); // section where the FAT16 responsible for this cluster resides
            if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&usCluster, (unsigned short)((ptr_location->ulCluster & 0xff) * sizeof(unsigned short)), sizeof(unsigned short)) != UTFAT_SUCCESS) { // read directly to cluster value
                return UTFAT_DISK_READ_ERROR;
            }
            ulCluster = LITTLE_SHORT_WORD(usCluster);                    // ensure endien is correct
        }
        else {
            ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT32 responsible for this cluster resides
            if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
                return UTFAT_DISK_READ_ERROR;
            }
            ulCluster = LITTLE_LONG_WORD(ulCluster);                     // ensure endien is correct
        }
#else
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT responsible for this cluster resides
        if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
            return UTFAT_DISK_READ_ERROR;
        }
        ulCluster = LITTLE_LONG_WORD(ulCluster);                         // ensure endien is correct
#endif
        if ((ulCluster <= ptr_utDisk->ulDirectoryBase) || (ulCluster > ptr_utDisk->utFAT.ulClusterCount)) {
            return UTFAT_DIRECTORY_AREA_EXHAUSTED;
        }
        ptr_location->ulCluster = ulCluster;                             // set next cluster
        ptr_location->ulSector = ((ptr_location->ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);
    }
    return UTFAT_SUCCESS;
}

static int fnNextSectorCreate(UTDISK *ptr_utDisk, FILE_LOCATION *ptr_location)
{
    ptr_location->ulSector++;                                            // next sector
    if (((ptr_location->ulSector - ptr_utDisk->ulLogicalBaseAddress) % ptr_utDisk->utFAT.ucSectorsPerCluster) == 0) { // check whether the present clust end has been reached
        unsigned long ulClusterSector;
        unsigned long ulCluster;
        unsigned long ulClusterMask;
    #ifdef UTFAT16
        unsigned short usCluster;
        if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
            ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 8)); // section where the FAT16 responsible for this cluster resides
            if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&usCluster, (unsigned short)((ptr_location->ulCluster & 0xff) * sizeof(unsigned short)), sizeof(unsigned short)) != UTFAT_SUCCESS) { // read directly to cluster value
                return UTFAT_DISK_READ_ERROR;
            }
            ulCluster = LITTLE_SHORT_WORD(usCluster);                    // ensure endien is correct
            ulClusterMask = FAT16_CLUSTER_MASK;
        }
        else {
            ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT32 responsible for this cluster resides
            if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
                return UTFAT_DISK_READ_ERROR;
            }
            ulCluster = LITTLE_LONG_WORD(ulCluster);                     // ensure endien is correct
            ulClusterMask = CLUSTER_MASK;
        }
    #else
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT32 responsible for this cluster resides
        if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
            return UTFAT_DISK_READ_ERROR;
        }
        ulCluster = LITTLE_LONG_WORD(ulCluster);                         // ensure endien is correct
        ulClusterMask = CLUSTER_MASK;
    #endif
        if ((ulCluster <= ptr_utDisk->ulDirectoryBase) || (ulCluster >= ulClusterMask)) { // invalid or end of current cluster - add a cluster so that the file can grow
    #if defined UTFAT_WRITE                                              // {40} allow operation without write support
            ulCluster = fnAllocateCluster(ptr_utDisk, ptr_location->ulCluster, NEW_RELATIVE_CLUSTER); // create a new cluster so that it can be suednext cluster
    #endif
        }
        if (ptr_location->ulCluster > ptr_utDisk->utFAT.ulClusterCount) {
            return UTFAT_DIRECTORY_AREA_EXHAUSTED;
        }
        ptr_location->ulCluster = ulCluster;
        ptr_location->ulSector = ((ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);
    }
    return UTFAT_SUCCESS;
}

#if defined UTFAT_WRITE                                                  // {40} allow operation without write support
static int fnDirectorySectorCreate(UTDISK *ptr_utDisk, FILE_LOCATION *ptr_location)
{
    unsigned long ulClusterSector;
    unsigned long ulCluster;
    unsigned long ulClusterMask;
    #ifdef UTFAT16
    unsigned short usCluster;
    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 8)); // section where the FAT16 responsible for this cluster resides
        if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&usCluster, (unsigned short)((ptr_location->ulCluster & 0xff) * sizeof(unsigned short)), sizeof(unsigned short)) != UTFAT_SUCCESS) { // read directly to cluster value
            return UTFAT_DISK_READ_ERROR;
        }
        ulCluster = LITTLE_SHORT_WORD(usCluster);                    // ensure endien is correct
        ulClusterMask = FAT16_CLUSTER_MASK;
    }
    else {
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT32 responsible for this cluster resides
        if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
            return UTFAT_DISK_READ_ERROR;
        }
        ulCluster = LITTLE_LONG_WORD(ulCluster);                     // ensure endien is correct
        ulClusterMask = CLUSTER_MASK;
    }
    #else
    ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT responsible for this cluster resides
    if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
        return UTFAT_DISK_READ_ERROR;
    }
    ulCluster = LITTLE_LONG_WORD(ulCluster);                         // ensure endien is correct
    ulClusterMask = CLUSTER_MASK;
    #endif
    if ((ulCluster <= ptr_utDisk->ulDirectoryBase) || (ulCluster >= ulClusterMask)) { // invalid or end of current cluster - add a cluster so that the file can grow
        ulCluster = fnAllocateCluster(ptr_utDisk, ptr_location->ulCluster, (INITIALISE_DIR_EXTENSION | NEW_RELATIVE_CLUSTER)); // next cluster
    }
    if (ptr_location->ulCluster > ptr_utDisk->utFAT.ulClusterCount) {
        return UTFAT_DIRECTORY_AREA_EXHAUSTED;
    }
    ptr_location->ulCluster = ulCluster;
    ptr_location->ulSector = ((ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);
    return UTFAT_SUCCESS;
}
#endif


static int fnNextDirectoryEntry(UTDISK *ptr_utDisk, DISK_LOCATION *ptr_location)
{
    if (++(ptr_location->ucDirectoryEntry) >= (ptr_utDisk->utFAT.usBytesPerSector/sizeof(DIR_ENTRY_STRUCTURE_FAT32))) { // end of present sector reached
        ptr_location->ucDirectoryEntry = 0;                              // start at first entry in next sector
        return (fnNextSector(ptr_utDisk, &ptr_location->directory_location));
    }
    return UTFAT_SUCCESS;
}


static void fnLoadShortFileInfo(UTFILEINFO *ptr_utFileInfo, const DIR_ENTRY_STRUCTURE_FAT32 *ptrDirectoryEntry)
{
    int i;
    unsigned char c;
    CHAR *ptrShortName = ptr_utFileInfo->cFileName;
    unsigned char ucNT_info = ptrDirectoryEntry->DIR_NTRes;

    for (i = 0; i < 8; i++) {                                            // name
        c = (unsigned char)ptrDirectoryEntry->DIR_Name[i];
        if (c == ' ') {
            break;
        }
        if (c == 0x05) {
            c = DIR_NAME_FREE;
        }
        if ((ucNT_info & 0x08) && (c >= 'A') && (c <= 'Z')) {
            c += ('a' - 'A');                                            // convert to small letters
        }
        *ptrShortName++ = c;
    }
    if (ptrDirectoryEntry->DIR_Name[8] != ' ') {                         // extension
        *ptrShortName++ = '.';
        for (i = 8; i < 11; i++) {
            c = ptrDirectoryEntry->DIR_Name[i];
            if (c == ' ') {
                break;
            }
            if ((ucNT_info & 0x10) && (c >= 'A') && (c <= 'Z')) {
                c += ('a' - 'A');                                        // convert to small letters
            }
            *ptrShortName++ = c;
        }
    }
    *ptrShortName = 0;                                                   // terminate

    ptr_utFileInfo->ucFileAttributes = ptrDirectoryEntry->DIR_Attr;
    ptr_utFileInfo->ulFileSize = ((ptrDirectoryEntry->DIR_FileSize[3] << 24) + (ptrDirectoryEntry->DIR_FileSize[2] << 16) + (ptrDirectoryEntry->DIR_FileSize[1] << 8) + ptrDirectoryEntry->DIR_FileSize[0]);
    ptr_utFileInfo->usFileData = ((ptrDirectoryEntry->DIR_WrtDate[1] << 8) + ptrDirectoryEntry->DIR_WrtDate[0]);
    ptr_utFileInfo->usFileTime = ((ptrDirectoryEntry->DIR_WrtTime[1] << 8) + ptrDirectoryEntry->DIR_WrtTime[0]);
}

#ifdef UTFAT_LFN_READ
static void fnExtractLongFileName(CHAR cLongFileName[100], LFN_ENTRY_STRUCTURE_FAT32 *ptrLFN_Entry, unsigned char *ucLongFileNameLength)
{
    int i = *ucLongFileNameLength;
    if (i == 0) {
        return;                                                          // {44} protect against invalid length
    }
    if (ptrLFN_Entry->LFN_EntryNumber & 0x40) {                          // last LFN entry
        if ((ptrLFN_Entry->LFN_Name_12 != 0xff) && (ptrLFN_Entry->LFN_Name_12 != 0x00)) { // string ending without null terminator
            cLongFileName[--i] = 0;                                      // force a null terminator
        }
    }
    if (i < 14) {                                                        // {44} protect against over-long names
        uMemset(cLongFileName, 0, i);                                    // pad with zeros
        i = 0;
    }
    else {
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_12;                  // enter the characters backwards in the LFN buffer
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_11;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_10;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_9;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_8;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_7;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_6;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_5;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_4;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_3;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_2;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_1;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_0;
    }
    *ucLongFileNameLength = i;                                           // update the length position
}
#endif


// Read directory or file
//
extern int utReadDirectory(UTLISTDIRECTORY *ptr_utListDirectory, UTFILEINFO *ptr_ut_fileInfo) // {23}
{
#ifdef UTFAT_LFN_READ
    CHAR cLongFileName[100];
    unsigned char ucLongFileNameLength = sizeof(cLongFileName);
#endif
    DIR_ENTRY_STRUCTURE_FAT32 *ptrDirectoryEntry;
    UTDISK *ptr_utDisk = &utDisks[ptr_utListDirectory->ptr_utDirObject->ucDrive]; // the disk that the directory is associated with
    int iFound = 0;

    while (iFound == 0) {
        if (fnLoadSector(ptr_utDisk, ptr_utListDirectory->private_disk_location.directory_location.ulSector) != UTFAT_SUCCESS) { // move to the disk sector containing the directory and read its content
            return UTFAT_DISK_READ_ERROR;
        }
        ptrDirectoryEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData;
        ptrDirectoryEntry += ptr_utListDirectory->private_disk_location.ucDirectoryEntry; // move to the present entry
        switch (ptrDirectoryEntry->DIR_Name[0]) {
        case 0:                                                          // end of directory has been reached
            return UTFAT_END_OF_DIRECTORY;
        case DIR_NAME_FREE:                                              // 0xe5 means that the end of the directory has been reached
#ifdef UTFAT_LFN_READ
            ucLongFileNameLength = sizeof(cLongFileName);                // {44} reset the LFN index
#endif
            break;
        default:
#ifdef UTFAT_LFN_READ                                                    // if long file name read support is enabled
            if ((ptrDirectoryEntry->DIR_Attr & DIR_ATTR_MASK) == (DIR_ATTR_LONG_NAME)) { // check whether the entry is a long file name
                fnExtractLongFileName(cLongFileName, (LFN_ENTRY_STRUCTURE_FAT32 *)ptrDirectoryEntry, &ucLongFileNameLength);
            }
            else if (!(ptrDirectoryEntry->DIR_Attr & DIR_ATTR_VOLUME_ID)) {  // if not a volume entry it is a short file name
                fnLoadShortFileInfo(ptr_ut_fileInfo, ptrDirectoryEntry); // extract the file information
                if (ucLongFileNameLength != sizeof(cLongFileName)) {
                    uStrcpy(ptr_ut_fileInfo->cFileName, &cLongFileName[ucLongFileNameLength]);
                }
                ucLongFileNameLength = sizeof(cLongFileName);            // reset long file name counter
                iFound = 1;                                              // mark that a file has been found and we can terminate after incrementing the directory entry
            }
#else
            if (!(ptrDirectoryEntry->DIR_Attr & DIR_ATTR_VOLUME_ID)) {   // if not a volume entry
                fnLoadShortFileInfo(ptr_ut_fileInfo, ptrDirectoryEntry); // extract the file information
                iFound = 1;                                              // mark that a file has been found and we can terminate after incrementing the directory entry
            }
#endif
            break;
        }
        if (fnNextDirectoryEntry(ptr_utDisk, &(ptr_utListDirectory->private_disk_location)) == UTFAT_DIRECTORY_AREA_EXHAUSTED) { // move to the next entry
            if (iFound == 1) {
                return UTFAT_FINAL_LISTING_ITEM_FOUND;
            }
            return UTFAT_DIRECTORY_AREA_EXHAUSTED;                       // quit since the complete directory space has been exhausted
        }
    }
    return UTFAT_SUCCESS;
}


#if !defined _REMOVE_FORMATTED_OUTPUT                                    // {43} no directory listing possible without formatted output
// Perform a directory listing to a buffer (with options for DOS and FTP styles)
//
extern int utListDir(UTLISTDIRECTORY *ptr_utListDirectory, FILE_LISTING *ptrFileListingInfo)
{
    unsigned char ucLineLength;
    int iResult;
    CHAR *ptrBuf = ptrFileListingInfo->ptrBuffer;
    UTFILEINFO utFileInfo;                                               // temporary file info object
    DISK_LOCATION previous_disk_loc;
    unsigned short usYear;
    unsigned char  ucDay, ucMonth, ucHour, ucMinutes;
    ptrFileListingInfo->usStringLength = 0;
    if (ptrFileListingInfo->ucStyle == FTP_TYPE_LISTING) {
        ucLineLength = FTP_STYLE_LIST_ENTRY_LENGTH;                       // the maximum length of a single entry line - the user's buffer must be adequate to accept this
    }
    else {
        ucLineLength = DOS_STYLE_LIST_ENTRY_LENGTH;                       // the maximum length of a single entry line - the user's buffer must be adequate to accept this
    }
    ptrFileListingInfo->usItemsReturned = 0;
    while (ptrFileListingInfo->usItemsReturned < ptrFileListingInfo->usMaxItems) { // for maximum amount of items
        uMemcpy(&previous_disk_loc, &ptr_utListDirectory->private_disk_location, sizeof (ptr_utListDirectory->private_disk_location)); // backup in case we need to restore due to lack of buffer space
        iResult = utReadDirectory(ptr_utListDirectory, &utFileInfo);
        if ((iResult != UTFAT_SUCCESS) && (iResult != UTFAT_FINAL_LISTING_ITEM_FOUND)) { // read next directory or file in the present directory
            return UTFAT_NO_MORE_LISTING_ITEMS_FOUND;                    // no more items found
        }
        if ((ptrFileListingInfo->usStringLength + ucLineLength + uStrlen(utFileInfo.cFileName)) > ptrFileListingInfo->usBufferLength) {
            uMemcpy(&ptr_utListDirectory->private_disk_location, &previous_disk_loc, sizeof (ptr_utListDirectory->private_disk_location)); // restore location
            return UTFAT_NO_MORE_LISING_SPACE;                           // no more file listings fit in the present buffer
        }
        ucDay = (utFileInfo.usFileData & 0x1f);
        ucMonth = ((utFileInfo.usFileData >> 5) & 0xf);
        usYear = ((utFileInfo.usFileData >> 9) + 1980);
        ucMinutes = ((utFileInfo.usFileTime >> 5) & 0x3f);
        ucHour = (utFileInfo.usFileTime >> 11);
        if (ptrFileListingInfo->ucStyle == FTP_TYPE_LISTING) {
            int i = 10;
            unsigned short usRights;
            CHAR cAccess = 'd';
            if (utFileInfo.ucFileAttributes & DIR_ATTR_DIRECTORY) {
                usRights = 0x3ed;                                        // directory
            }
            else {                                                       // file
                if (utFileInfo.ucFileAttributes & DIR_ATTR_READ_ONLY) {
                    usRights = 0x124;
                }
                else {
                    usRights = 0x1a4;
                }
            }
            while (i--) {
                if (usRights & 0x200) {
                    *ptrBuf++ = cAccess;                                 // rights
                }
                else {
                    *ptrBuf++ = '-';                                     // no rights
                }
                switch (cAccess) {                                       // set next flag
                case 'r':
                    cAccess = 'w';
                    break;
                case 'w':
                    cAccess = 'x';
                    break;
                default:
                    cAccess = 'r';
                    break;
                }
                usRights <<= 1;
            }
            ptrBuf = uStrcpy(ptrBuf, " 1 502 502 ");
            ptrBuf = fnBufferDec(utFileInfo.ulFileSize, 0, ptrBuf);
            *ptrBuf++ = ' ';
            switch (ucMonth) {
            default:
                ptrBuf = uStrcpy(ptrBuf, "Jan");
                break;
            case 2:
                ptrBuf = uStrcpy(ptrBuf, "Feb");
                break;
            case 3:
                ptrBuf = uStrcpy(ptrBuf, "Mar");
                break;
            case 4:
                ptrBuf = uStrcpy(ptrBuf, "Apr");
                break;
            case 5:
                ptrBuf = uStrcpy(ptrBuf, "May");
                break;
            case 6:
                ptrBuf = uStrcpy(ptrBuf, "Jun");
                break;
            case 7:
                ptrBuf = uStrcpy(ptrBuf, "Jul");
                break;
            case 8:
                ptrBuf = uStrcpy(ptrBuf, "Aug");
                break;
            case 9:
                ptrBuf = uStrcpy(ptrBuf, "Sep");
                break;
            case 10:
                ptrBuf = uStrcpy(ptrBuf, "Oct");
                break;
            case 11:
                ptrBuf = uStrcpy(ptrBuf, "Nov");
                break;
            case 12:
                ptrBuf = uStrcpy(ptrBuf, "Dec");
                break;
            }
            ptrBuf = fnBufferDec(ucDay, (LEADING_ZERO | WITH_SPACE), ptrBuf); // day of month
            ptrBuf = fnBufferDec(usYear, WITH_SPACE, ptrBuf);
        }
        else {                                                           // DOS style listing
            if (utFileInfo.ucFileAttributes & DIR_ATTR_READ_ONLY) {
                *ptrBuf++ = 'R';
            }
            else {
                *ptrBuf++ = '-';
            }
            if (utFileInfo.ucFileAttributes & DIR_ATTR_HIDDEN) {
                *ptrBuf++ = 'H';
            }
            else {
                *ptrBuf++ = '-';
            }
            if (utFileInfo.ucFileAttributes & DIR_ATTR_SYSTEM) {
                *ptrBuf++ = 'S';
            }
            else {
                *ptrBuf++ = '-';
            }
            if (utFileInfo.ucFileAttributes & DIR_ATTR_ARCHIVE) {
                *ptrBuf++ = 'A';
            }
            else {
                *ptrBuf++ = '-';
            }
            ptrBuf = fnBufferDec(ucDay, (LEADING_ZERO | WITH_SPACE), ptrBuf);
            *ptrBuf++ = '.';
            ptrBuf = fnBufferDec(ucMonth, (LEADING_ZERO), ptrBuf);
            *ptrBuf++ = '.';
            ptrBuf = fnBufferDec(usYear, 0, ptrBuf);
            *ptrBuf++ = ' ';
            ptrBuf = fnBufferDec(ucHour, (LEADING_ZERO | WITH_SPACE), ptrBuf);
            *ptrBuf++ = ':';
            ptrBuf = fnBufferDec(ucMinutes, (LEADING_ZERO), ptrBuf);
            if (utFileInfo.ucFileAttributes & DIR_ATTR_DIRECTORY) {
                if (ptrFileListingInfo->ucStyle == DOS_TYPE_LISTING) {
                    ptrBuf = uStrcpy(ptrBuf, " <DIR>           ");
                }
                ptrFileListingInfo->usDirectoryCount++;                  // count the directories in this listing
            }
            else {
                int iLen;
                CHAR cLenBuf[18];                                        // maximum decimal file length plus null terminator plus " <DIR> "fill
                CHAR *ptrLen = fnBufferDec(utFileInfo.ulFileSize, WITH_SPACE, cLenBuf);
                iLen = (17 - (ptrLen - cLenBuf));
                while (iLen--) {
                    *ptrBuf++ = ' ';
                }
                ptrBuf = uStrcpy(ptrBuf, cLenBuf);
                ptrFileListingInfo->usFileCount++;                       // count the files in this listing
                ptrFileListingInfo->ulFileSizes += utFileInfo.ulFileSize;// sum of the total file sizes in this listing
            }
        }
        *ptrBuf++ = ' ';
        ptrBuf = uStrcpy(ptrBuf, utFileInfo.cFileName);
        *ptrBuf++ = '\r';
        *ptrBuf++ = '\n';
        ptrFileListingInfo->usStringLength = (ptrBuf - ptrFileListingInfo->ptrBuffer);
        ptrFileListingInfo->usItemsReturned++;
        if (iResult == UTFAT_FINAL_LISTING_ITEM_FOUND) {
            return UTFAT_NO_MORE_LISTING_ITEMS_FOUND;                    // the content is the final content
        }
    }
    return UTFAT_SUCCESS;
}
#endif

static int fnCreateNameParagraph(const CHAR **pptrDirectoryPath, CHAR cDirectoryName[12])
{
    int iLength = 0;
    int iEvenByte = 0;
    int iSeparatorFound = 0;
    CHAR cInputCharacter;
    unsigned char ucCharacterCharacteristics;

    uMemset(cDirectoryName, ' ', 11);                                    // start with blank name
    cDirectoryName[11] = NT_FLAG;
    while (1) {
        cInputCharacter = *(*pptrDirectoryPath)++;
        switch (cInputCharacter) {
        case '.':
            if ((iLength != 0) && (iLength <= 8)) {                      // separator found
                iSeparatorFound = 1;
                iLength = 8;
                continue;
            }
            break;
        case 0x0a:                                                       // for FTP compatibility
        case 0x0d:
            if (iLength != 0) {                                          // regognise line-feed and carriage-return as end of name
                return FULLY_QUALIFIED_SHORT_NAME;                       // return true if the end of the path has been found
            }
            else {
                return INVALID_PARAGRAPH;                                // {37} empty input
            }
          //continue;
        case 0:
        case FORWARD_SLASH:
        case BACK_SLASH:
            if (iLength != 0) {
                return ((cInputCharacter == 0) || (**pptrDirectoryPath == 0)); // {47} return true if the end of the path has been found
            }                                                            // fall through to ignore
            continue;                                                    // ignore at start of string
        default:
            if ((unsigned char)cInputCharacter <= ' ') {                 // reject all invisible characters
                continue;
            }
            else {
                if ((unsigned char)cInputCharacter <= '~') {
                    ucCharacterCharacteristics = ucCharacterTable[cInputCharacter - '!'];
                    if (ucCharacterCharacteristics & (_CHAR_REJECT | _CHAR_REJECT_NON_JAP)) {
                        if ((iEvenByte == 0) || (!(ucCharacterCharacteristics & _CHAR_REJECT_NON_JAP))) { // don't apply these to second byte of Japanese characters
                            continue;                                    // reject
                        }
                    }
                    else if (iEvenByte == 0) {                           // don't apply to second byte of Japanese characters
                        if (ucCharacterCharacteristics & _CHAR_CAPITAL) {
                            if (iSeparatorFound != 1) {
                                cDirectoryName[11] &= ~0x08;
                            }
                            else {
                                cDirectoryName[11] &= ~0x10;
                            }
                        }
                        else if (ucCharacterCharacteristics & _CHAR_SMALL) {
                            cInputCharacter -= ('a' - 'A');              // convert to upper case
                            if (iSeparatorFound != 1) {
                                cDirectoryName[11] |= 0x08;
                            }
                            else {
                                cDirectoryName[11] |= 0x10;
                            }
                        }
                    }
                }
                else if ((((unsigned char)cInputCharacter >= 0x81) && ((unsigned char)cInputCharacter <= 0x9f)) || (((unsigned char)cInputCharacter >= 0xe0) && ((unsigned char)cInputCharacter <= 0xfc))) { // Japanese
                    iEvenByte = 1;
                    if ((iLength == 0) && (cInputCharacter == (CHAR)DIR_NAME_FREE)) { // {31}
                        cDirectoryName[iLength++] = 0x05;                // substitute
                        continue;                                        // continue with even byte
                    }
                }
                else {
                    continue;                                            // reject
                }
            }
            break;                                                       // accept
        }
        if (iLength >= 8) {
            if ((iSeparatorFound == 0) || (iLength >= 11)) {             // name part or extension is too long
#ifdef UTFAT_LFN_READ
                while (1) {
                    switch (*(*pptrDirectoryPath)++) {                   // search the end of the long file name paragraph
                    case 0x0d:                                           // end of line considered fully qualified terminator (for FTP compatibility)
                    case 0x0a:
                    case 0:
                        (*pptrDirectoryPath)--;                          // set back to null-terinator of long file name
                        return FULLY_QUALIFIED_LONG_NAME;
                    case FORWARD_SLASH:
                    case BACK_SLASH:
                        (*pptrDirectoryPath)--;                          // set back to terminator of long file name
                        return LONG_NAME_PARAGRAPH;
                    }
                }
#else
                break;
#endif
            }
        }
        iEvenByte = 0;
        cDirectoryName[iLength++] = cInputCharacter;
    }
    return INVALID_PARAGRAPH;                                            // invalid
}


static UTDIRECTORY utDirectoryObjects[UT_DIRECTORIES_AVAILABLE] = {{0}};

extern UTDIRECTORY*utAllocateDirectory(unsigned char ucDisk, unsigned short usPathLength)
{
    int i = 0;
    while (i < UT_DIRECTORIES_AVAILABLE) {
        if (utDirectoryObjects[i].usDirectoryFlags == 0) {               // directory not allocated
            utDirectoryObjects[i].usDirectoryFlags = UTDIR_ALLOCATED;
            utDirectoryObjects[i].ucDrive = ucDisk;
            if (usPathLength != 0) {
                utDirectoryObjects[i].ptrDirectoryPath = uMalloc((MAX_MALLOC)(usPathLength + 1)); // reserve space for holding the directory path string (plus space for null terminator)
            }
            else {
                utDirectoryObjects[i].ptrDirectoryPath = 0;
            }
            utDirectoryObjects[i].usDirectoryPathLength = usPathLength;  // enter the maximum length that can be stored
            return &utDirectoryObjects[i];
        }
        i++;
    }
    return 0;                                                            // no directory object available for allocation
}


// Open or modify a directory object
//
extern int utOpenDirectory(const CHAR *ptrDirPath, UTDIRECTORY *ptrDirObject)
{
    #define ROOT_DIRECTORY_REFERENCE 0x1
    #define ROOT_DIRECTORY_RELOCATE  0x2
    #define ROOT_DIRECTORY_SET       0x4
    DIR_ENTRY_STRUCTURE_FAT32 *ptrFoundEntry = 0;
    UTDISK *ptr_utDisk = &utDisks[ptrDirObject->ucDrive];
    DISK_LOCATION *ptrDiskLocation;
    int iRootDirectory = 0;
    const CHAR *ptrLocalDirPath;
#ifdef UTFAT_LFN_READ
    const CHAR *ptrLongFileNamePathStart, *ptrLongFileNamePathEnd;
#endif
    unsigned long ulCluster;
    DISK_LOCATION DirStart;
    int iMatchedParagraph;
    int iFullyQualifiedPath;
    CHAR cFileName[8 + 3 + 1];                                           // space for 8:3 format plus an NT specific byte
    unsigned short usDirFlags;

    if (ptrDirPath != 0) {
        int iMoveUp = 0;
        unsigned short usPathTerminator = ptrDirObject->usRelativePathLocation;
        while ((*ptrDirPath == '.') && (*(ptrDirPath + 1) == '.')) {
            if (ptrDirObject->ptrDirectoryPath == 0) {                   // if no directory string path in use only referenced paths possible so always move to root
                ptrDirObject->usDirectoryFlags |= UTDIR_TEST_FULL_PATH;
                ptrDirPath = 0;
                break;
            }

            if (usPathTerminator <= 3) {                                 // can't move up any further
                return UTFAT_PATH_NOT_FOUND;
            }
            if (iMoveUp != 0) {
                usPathTerminator--;
            }
            while ((ptrDirObject->ptrDirectoryPath[usPathTerminator] != BACK_SLASH) && (ptrDirObject->ptrDirectoryPath[usPathTerminator] != FORWARD_SLASH)) {
                usPathTerminator--;
            }
            ptrDirPath += 2;
            iMoveUp++;
            if (*ptrDirPath == 0) {
                break;
            }
            ptrDirPath++;
        }
        if (iMoveUp != 0) {
            ptrDirObject->usDirectoryFlags |= UTDIR_TEST_FULL_PATH;      // use full path to set directory but don't move original (this flag is automatically reset)
            usDirFlags = ptrDirObject->usDirectoryFlags;                 // backup original flags
            if (usPathTerminator > 3) {                                  // not root directory
                int iReturn;
                CHAR cTemp = ptrDirObject->ptrDirectoryPath[usPathTerminator]; // backup the location
                ptrDirObject->ptrDirectoryPath[usPathTerminator] = 0;    // temporary termination
                ptrDirObject->usDirectoryFlags &= ~UTDIR_DIR_AS_FILE;    // this may not be set when moving to next directory location
                if ((iReturn = utOpenDirectory(&ptrDirObject->ptrDirectoryPath[3], ptrDirObject)) != UTFAT_SUCCESS) { // try to locate the new directory
                    ptrDirObject->ptrDirectoryPath[usPathTerminator] = cTemp;// correct the original path
                    return UTFAT_PATH_NOT_FOUND;
                }
                if (*ptrDirPath != 0) {                                   // if the path should go downwards after locating the upward path location
                    ptrDirObject->usDirectoryFlags = ((usDirFlags & ~UTDIR_TEST_FULL_PATH) | UTDIR_TEST_FULL_PATH_TEMP);
                    if ((iReturn = utOpenDirectory(ptrDirPath, ptrDirObject)) != UTFAT_SUCCESS) { // continue downward search form the new location
                        usDirFlags = 0;
                    }
                    else if (usDirFlags & UTDIR_ALLOW_MODIFY_PATH) {
                        CHAR *ptrStart = &ptrDirObject->ptrDirectoryPath[usPathTerminator];                        
                        usPathTerminator += (uStrcpy(ptrStart, (ptrDirPath - 1)) - ptrStart);
                        return UTFAT_SUCCESS_PATH_MODIFIED;
                    }
                }
                if (usDirFlags & UTDIR_ALLOW_MODIFY_PATH) {              // if path modification allowed
                    ptrDirObject->usRelativePathLocation = usPathTerminator;
                    return UTFAT_SUCCESS_PATH_MODIFIED;
                }
                ptrDirObject->ptrDirectoryPath[usPathTerminator] = cTemp;// correct the original path
                return iReturn;
            }
            else {
                if (usDirFlags & UTDIR_ALLOW_MODIFY_PATH) {              // if path modification allowed
                    ptrDirObject->ptrDirectoryPath[3] = 0;
                    ptrDirObject->usRelativePathLocation = 3;
                    uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->root_disk_location, sizeof(ptrDirObject->private_disk_location)); // synchronise to root location
                    return UTFAT_SUCCESS_PATH_MODIFIED;
                }
            }
        }
    }
    ptrLocalDirPath = ptrDirPath;
    usDirFlags = ptrDirObject->usDirectoryFlags;                         // flags on entry

    ptrDirObject->usDirectoryFlags &= ~(UTDIR_TEST_FULL_PATH | UTDIR_TEST_FULL_PATH_TEMP | UTDIR_TEST_REL_PATH | UTDIR_DIR_AS_FILE | UTDIR_SET_START | UTDIR_ALLOW_MODIFY_PATH); // these flags are automatically cleared

    if (!(ptr_utDisk->usDiskFlags & DISK_MOUNTED)) {                     // if the disk is not ready for use quit
        return UTFAT_DISK_NOT_READY;
    }
    if ((ptrLocalDirPath == 0) || (*ptrLocalDirPath == 0) || ((*ptrLocalDirPath == LINE_FEED) || (*ptrLocalDirPath == CARRIAGE_RETURN))) { // if root directory
        iRootDirectory = ROOT_DIRECTORY_REFERENCE;                       // ready to work with root directory or referenced directory
    }
    else if (((*ptrLocalDirPath == BACK_SLASH) || (*ptrLocalDirPath == FORWARD_SLASH))) { // {7}
        usDirFlags |= UTDIR_TEST_REL_PATH;                               // {25}
        if (*(ptrLocalDirPath + 1) != 0) {                               // root directory referenced
            iRootDirectory = ROOT_DIRECTORY_RELOCATE;                    // ready to work with root directory or referenced directory
        }
        else {
            iRootDirectory = ROOT_DIRECTORY_SET;                         // ready to work with root directory explicitly
        }
    }

    if (!(usDirFlags & UTDIR_VALID)) {
        ulCluster = ptr_utDisk->ulDirectoryBase;                         // the first cluster in the root directory
        ptrDirObject->private_disk_location.directory_location.ulCluster = ulCluster;
#ifdef UTFAT16
        if ((ulCluster <= 1) && (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16)) { // if FAT16 root folder
            ptrDirObject->private_disk_location.directory_location.ulSector = (ptr_utDisk->utFAT.ucSectorsPerCluster + ptr_utDisk->ulVirtualBaseAddress - (32 - 1));// the sector in which the directory entries begin
        }
        else {
            ptrDirObject->private_disk_location.directory_location.ulSector = (ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster);// section referenced to logical base address
            ptrDirObject->private_disk_location.directory_location.ulSector += ptr_utDisk->ulVirtualBaseAddress;// the sector in which the directory entries begin
        }
#else
        ptrDirObject->private_disk_location.directory_location.ulSector = (ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster);// section referenced to logical base address
        ptrDirObject->private_disk_location.directory_location.ulSector += ptr_utDisk->ulVirtualBaseAddress;// the sector in which the directory entries begin
#endif
        ptrDirObject->private_disk_location.ucDirectoryEntry = 0;        // reset the entry index
        uMemcpy(&ptrDirObject->root_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->root_disk_location)); // enter the fixed root location
        uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->root_disk_location)); // ensure that the public entry can not be used with invalid value {26}
        ptrDirObject->usDirectoryFlags = (UTDIR_ALLOCATED | UTDIR_VALID); // the entry is now valid
        if ((ptrDirObject->ptrDirectoryPath != 0) && (iRootDirectory != 0)) { // {7} if a root path is being used and root is to be set
            uStrcpy(ptrDirObject->ptrDirectoryPath, "D:\\");             // set root path
            ptrDirObject->usRelativePathLocation = 3;                    // the relative path is equal to the root path
        }
    }

    if (usDirFlags & (UTDIR_TEST_FULL_PATH | UTDIR_TEST_FULL_PATH_TEMP | UTDIR_TEST_REL_PATH | UTDIR_REFERENCED)) {
        if (usDirFlags & UTDIR_TEST_FULL_PATH) {                         // set temporary root directory
            ulCluster = ptr_utDisk->ulDirectoryBase;                     // the first cluster in the root directory
            ptrDirObject->public_disk_location.directory_location.ulCluster = ulCluster;
            ptrDirObject->public_disk_location.directory_location.ulSector = (ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster);
            ptrDirObject->public_disk_location.directory_location.ulSector += ptr_utDisk->ulVirtualBaseAddress;
            ptrDirObject->public_disk_location.ucDirectoryEntry = 0;
        }
        else if (!(usDirFlags & UTDIR_TEST_FULL_PATH_TEMP)) {
            if (iRootDirectory & (ROOT_DIRECTORY_RELOCATE | ROOT_DIRECTORY_SET)) { // {7}
                uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->root_disk_location, sizeof(ptrDirObject->private_disk_location)); // synchronise to root
                if (iRootDirectory == ROOT_DIRECTORY_SET) {
                    return UTFAT_PATH_IS_ROOT;                           // inform that the location is root
                }
            }
            else {
                uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->private_disk_location)); // synchronise to present location
            }
        }                                                                // UTDIR_TEST_FULL_PATH_TEMP continues using present public values to continue a search
        ptrDiskLocation = &ptrDirObject->public_disk_location;        
    }
    else {
        ptrDiskLocation = &ptrDirObject->private_disk_location;
        uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->private_disk_location)); // synchronise to present location
    }
    if (iRootDirectory & (ROOT_DIRECTORY_REFERENCE | ROOT_DIRECTORY_SET)) { // {7} if root directory reference
        return UTFAT_SUCCESS;                                            // ready to work with root directory or referenced directory
    }
    uMemcpy(&DirStart, ptrDiskLocation, sizeof(DirStart));               // {6} backup the original starting directory location

    while (1) {
#ifdef UTFAT_LFN_READ
        ptrLongFileNamePathStart = ptrLocalDirPath;
        if ((*ptrLongFileNamePathStart == BACK_SLASH) || (*ptrLongFileNamePathStart == FORWARD_SLASH)) {
            ptrLongFileNamePathStart++;
        }
#endif
        if ((iFullyQualifiedPath = fnCreateNameParagraph(&ptrLocalDirPath, cFileName)) < 0) { // fill the file name with a single paragraph from the file path
            return UTFAT_PATH_NOT_FOUND;
        }
#ifdef UTFAT_LFN_READ
        ptrLongFileNamePathEnd = ptrLocalDirPath;
#endif
        iMatchedParagraph = 0;
        do {
            if (fnLoadSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) != UTFAT_SUCCESS) {
                return UTFAT_DISK_READ_ERROR;
            }
            ptrFoundEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
            ptrFoundEntry += ptrDiskLocation->ucDirectoryEntry;          // move to the present entry
#ifdef UTFAT_LFN_READ
            if (iFullyQualifiedPath > FULLY_QUALIFIED_SHORT_NAME) {      // the name is not a short name so check for long name match
                int iMatchStatus = fnMatchLongName(&ptrLongFileNamePathEnd, ptrLongFileNamePathStart, (LFN_ENTRY_STRUCTURE_FAT32 *)ptrFoundEntry);
                switch (iMatchStatus) {
                case MATCH_SUCCESSFUL:
                    if (iFullyQualifiedPath == FULLY_QUALIFIED_LONG_NAME) {
                        iFullyQualifiedPath = 1;                         // path name fully matched
                    }
                    else {
                        iFullyQualifiedPath = 0;                         // successful paragraph match, but not completed
                    }
                    ptrLongFileNamePathStart = 0;                        // for an unconditional match on next directory entry
                    break;
                case MATCH_FALSE:
                    ptrLongFileNamePathEnd = ptrLocalDirPath;
                case MATCH_CONTINUE:                                     // match is still correct but not completed
                    break;                                               // not the searched name
                case END_DIRECTORY_ENTRIES:                              // end if end found
                    if (iFullyQualifiedPath == FULLY_QUALIFIED_LONG_NAME) {
                        return UTFAT_FILE_NOT_FOUND;
                    }
                    else {
                        return UTFAT_PATH_NOT_FOUND;
                    }
                }
            }
            else {
#endif
                switch (ptrFoundEntry->DIR_Name[0]) {
                case 0:                                                  // end if end found
                    if (iFullyQualifiedPath != 0) {
                        if (usDirFlags & UTDIR_SET_START) {              // {6}
                            uMemcpy(ptrDiskLocation, &DirStart, sizeof(DirStart)); // set the location to the start of the lowest directory
                        }
                        return UTFAT_FILE_NOT_FOUND;
                    }
                    else {
                        return UTFAT_PATH_NOT_FOUND;
                    }
                case DIR_NAME_FREE:
                    break;
                default:
                    if (
#ifdef UTFAT_LFN_READ
                        (ptrLongFileNamePathStart == 0) ||               // this short file name entry unconditionally belongs to a previous long file name block
#endif
                        ((!(ptrFoundEntry->DIR_Attr & DIR_ATTR_VOLUME_ID)) && (!uMemcmp(ptrFoundEntry->DIR_Name, cFileName, sizeof(ptrFoundEntry->DIR_Name))))) { // if entry has been found
                        ulCluster = ((ptrFoundEntry->DIR_FstClusHI[1] << 24) + (ptrFoundEntry->DIR_FstClusHI[0] << 16) + (ptrFoundEntry->DIR_FstClusLO[1] << 8) + ptrFoundEntry->DIR_FstClusLO[0]);
                        ptrDirObject->ptrEntryStructure = ptrFoundEntry; // set pointer to the entry so that the caller can extract additional information if required
                        if (!(ptrFoundEntry->DIR_Attr & DIR_ATTR_DIRECTORY)) {
                            ptrDirObject->public_file_location.ulCluster = ulCluster; // file's start cluster
                            ptrDirObject->public_file_location.ulSector = ((ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);// section referenced to logical base address
                            return UTFAT_PATH_IS_FILE;                   // not a directory so can not be traced further
                        }
                        if ((iFullyQualifiedPath != 0) && (usDirFlags & UTDIR_DIR_AS_FILE)) { // if a directory is to be treated as a file, its location is preserved rather than moving to its content
                            return UTFAT_SUCCESS;                        // file matched
                        }
                        ptrDiskLocation->directory_location.ulCluster = ulCluster; // move to the new directory
                        ptrDiskLocation->directory_location.ulSector = ((ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);// section referenced to logical base address
                        ptrDiskLocation->ucDirectoryEntry = 2;           // skip "." and ".." entries
                        if (iFullyQualifiedPath != 0) {
                            if (ROOT_DIRECTORY_RELOCATE == iRootDirectory) { // {7}
                                return UTFAT_PATH_IS_ROOT_REF;           // directory path successfully found but relative to root
                            }
                            return UTFAT_SUCCESS;                        // file matched
                        }
                        uMemcpy(&DirStart, ptrDiskLocation, sizeof(DirStart)); // {6} set the location to the start of this directory
                        iMatchedParagraph = 1;                           // this paragraph matched, but not last one
                    }
                    break;
                }
#ifdef UTFAT_LFN_READ
            }
#endif
            if (iMatchedParagraph != 0) {
                break;
            }
            if (fnNextDirectoryEntry(ptr_utDisk, ptrDiskLocation) != UTFAT_SUCCESS) { // move to next directory entry
                if (iMatchedParagraph != 0) {
                    return UTFAT_SUCCESS;
                }
                if (iFullyQualifiedPath != 0) {
                    if (usDirFlags & UTDIR_SET_START) {                  // {16}
                        uMemcpy(ptrDiskLocation, &DirStart, sizeof(DirStart)); // set the location to the start of the lowest directory
                    }
                    return UTFAT_FILE_NOT_FOUND;
                }
                else {
                    return UTFAT_PATH_NOT_FOUND;
                }
            }
        } while (1);
    }
}


// Read linear file data content directly to an external buffer
//
extern int utReadFile(UTFILE *ptr_utFile, void *ptrBuffer, unsigned short usReadLength)
{
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
    unsigned long ulContentRemaining;
    unsigned short usAccessLength;
    unsigned short usAccessOffset;

    ptr_utFile->usLastReadWriteLength = 0;
    if (!(ptr_utFile->ucFileMode & UTFAT_OPEN_FOR_READ)) {
        return UTFAT_FILE_NOT_READABLE;                                  // don't allow read of files not opened for read
    }
    ulContentRemaining = (ptr_utFile->ulFileSize - ptr_utFile->ulFilePosition);
    if (usReadLength > ulContentRemaining) {
        usReadLength = (unsigned short)ulContentRemaining;               // limit  to remaining size
    }

    while (usReadLength != 0) {                                          // while requested length of data has not yet been returned
        usAccessOffset = (unsigned short)(ptr_utFile->ulFilePosition & 0x1ff);
        usAccessLength = (512 - usAccessOffset);
        if (usAccessLength > usReadLength) {
            usAccessLength = usReadLength;
        }
        if ((usAccessOffset != 0) || (usAccessLength != 512)) {          // only load partical data if a complete sector can not be read directly to the buffer
            if (fnLoadPartialData(ptr_utDisk, ptr_utFile->public_file_location.ulSector, (unsigned char *)ptrBuffer, usAccessOffset, usAccessLength) != UTFAT_SUCCESS) { // read directly to buffer
                return UTFAT_DISK_READ_ERROR;
            }
        }
        else {                                                           // load a complete sector            
            if (utReadDiskSector(ptr_utDisk, ptr_utFile->public_file_location.ulSector, (unsigned char *)ptrBuffer) != UTFAT_SUCCESS) { // read complete sector directly to buffer
                return UTFAT_DISK_READ_ERROR;
            }
        }
        usReadLength -= usAccessLength;
        ptr_utFile->ulFilePosition += usAccessLength;
        ptr_utFile->usLastReadWriteLength += usAccessLength;
        ptrBuffer = (void *)((CAST_POINTER_ARITHMETIC)ptrBuffer + usAccessLength);
        if ((ptr_utFile->ulFilePosition % ptr_utDisk->utFAT.usBytesPerSector) == 0) { // {13} on a sector boundary
            int iResult = fnNextSector(ptr_utDisk, &ptr_utFile->public_file_location);
            if (iResult != UTFAT_SUCCESS) {
                return iResult;
            }
        }
    }
    return UTFAT_SUCCESS;
}


// Open a list directory referenced to its main directory object, which must already have been configured
//
extern int utLocateDirectory(const CHAR *ptrDirPath, UTLISTDIRECTORY *ptrListDirectory)
{
    int iReturn = UTFAT_SUCCESS;
    if (ptrListDirectory->ptr_utDirObject == 0) {
        return UTFAT_DIRECTORY_OBJECT_MISSING;
    }
    ptrListDirectory->ptr_utDirObject->usDirectoryFlags |= UTDIR_REFERENCED; // the next open is a referenced open
    if ((iReturn = utOpenDirectory(ptrDirPath, ptrListDirectory->ptr_utDirObject)) >= UTFAT_SUCCESS) {
        uMemcpy(&ptrListDirectory->private_disk_location, &ptrListDirectory->ptr_utDirObject->public_disk_location, sizeof(ptrListDirectory->private_disk_location)); // copy the referenced directory details to the list directory object
    }
    return iReturn;
}

// This routine sets a fixed time and data - it can be replaced by a routine to set the information from a RTC or similar
//
#define CREATION_HOURS   15
#define CREATION_MINUTES 58
#define CREATION_SECONDS 43

#define CREATION_DAY_OF_MONTH  31
#define CREATION_MONTH_OF_YEAR 1
#define CREATION_YEAR    (2010 - 1980)

#if defined UTFAT_WRITE                                                  // {43}
static void fnSetTimeDate(DIR_ENTRY_STRUCTURE_FAT32 *ptrEntry, int iCreation) // {3}
{
    unsigned short usCreationTime;
    unsigned short usCreationDate;
    #ifdef SUPPORT_FILE_TIME_STAMP
    if (fnGetLocalFileTime(&usCreationTime, &usCreationDate) != 0) {     // {19}
        usCreationTime = (CREATION_SECONDS | (CREATION_MINUTES << 5) | (CREATION_HOURS << 11));
        usCreationDate = (CREATION_DAY_OF_MONTH | (CREATION_MONTH_OF_YEAR << 5) | (CREATION_YEAR << 9));
    }
    #else
    usCreationTime = (CREATION_SECONDS | (CREATION_MINUTES << 5) | (CREATION_HOURS << 11));
    usCreationDate = (CREATION_DAY_OF_MONTH | (CREATION_MONTH_OF_YEAR << 5) | (CREATION_YEAR << 9));
    #endif
    ptrEntry->DIR_WrtTime[0] = (unsigned char)(usCreationTime);
    ptrEntry->DIR_WrtTime[1] = (unsigned char)(usCreationTime >> 8);
    ptrEntry->DIR_LstAccDate[0] = ptrEntry->DIR_WrtDate[0] = (unsigned char)(usCreationDate);
    ptrEntry->DIR_LstAccDate[1] = ptrEntry->DIR_WrtDate[1] = (unsigned char)(usCreationDate >> 8);
    if (iCreation != 0) {
        ptrEntry->DIR_CrtTime[0] = ptrEntry->DIR_WrtTime[0];
        ptrEntry->DIR_CrtTime[1] = ptrEntry->DIR_WrtTime[1];
        ptrEntry->DIR_CrtDate[0] = ptrEntry->DIR_LstAccDate[0];
        ptrEntry->DIR_CrtDate[1] = ptrEntry->DIR_LstAccDate[1];
    }
}

static void fnAddEntry(DIR_ENTRY_STRUCTURE_FAT32 *ptrEntry, unsigned long ulPresentCluster, unsigned char ucAttributes)
{
    ptrEntry->DIR_FstClusLO[0] = (unsigned char)ulPresentCluster;
    ptrEntry->DIR_FstClusLO[1] = (unsigned char)(ulPresentCluster >> 8);
    ptrEntry->DIR_FstClusHI[0] = (unsigned char)(ulPresentCluster >> 16);
    ptrEntry->DIR_FstClusHI[1] = (unsigned char)(ulPresentCluster >> 24);
    ptrEntry->DIR_Attr = ucAttributes;
    uMemset(ptrEntry->DIR_FileSize, 0, sizeof(ptrEntry->DIR_FileSize));  // {2} set the file size to zero
    fnSetTimeDate(ptrEntry, 1);                                          // {3} add creation time
}

static void fnSetFileInformation(DIR_ENTRY_STRUCTURE_FAT32 *ptrFileEntry, unsigned long ulFileSize)
{
    ptrFileEntry->DIR_FileSize[0] = (unsigned char)(ulFileSize);         // update the file size
    ptrFileEntry->DIR_FileSize[1] = (unsigned char)(ulFileSize >> 8);
    ptrFileEntry->DIR_FileSize[2] = (unsigned char)(ulFileSize >> 16);
    ptrFileEntry->DIR_FileSize[3] = (unsigned char)(ulFileSize >> 24);
    fnSetTimeDate(ptrFileEntry, 0);                                      // {3} set the modification time
}

// Allocate a single new cluster
//
static unsigned long fnAllocateCluster(UTDISK *ptr_utDisk, unsigned long ulPresentCluster, unsigned char ucClusterType)
{
    unsigned long  ulFAT = ptr_utDisk->utFAT.ulFAT_start;                // sector in which the file's first cluster is located in
    unsigned long  ulFatOriginal; // = ulFAT;                            {15}
    unsigned long  ulSectorContent[512/sizeof(signed long)];             // sector to read FAT32 sectors to
    unsigned long  ulAbsoluteCluster;
    unsigned char  ucClusterMax;
    unsigned char  ucClusterEntry;
    unsigned char  ucOriginalCluster;
    if (ucClusterType & NEW_RELATIVE_CLUSTER) {
        ulAbsoluteCluster = ulPresentCluster;
      //ulFAT += (ulPresentCluster/(512/sizeof(signed long)));           // {4}{14} moved to be used also by info block referenced case
    }
    else {
        if ((ptr_utDisk->usDiskFlags & FSINFO_VALID) && (ptr_utDisk->utFileInfo.ulNextFreeCluster <= ptr_utDisk->utFileInfo.ulFreeClusterCount)) { // the info block is valid so reference to next free cluster
            ulAbsoluteCluster = ptr_utDisk->utFileInfo.ulNextFreeCluster;
        }
        else {
            ulAbsoluteCluster = ptr_utDisk->ulDirectoryBase;
        }
        ulPresentCluster = ulAbsoluteCluster;
    }
    ucClusterEntry = (unsigned char)ulPresentCluster;
    #ifdef UTFAT16
    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
        ulFAT += (ulPresentCluster/(512/sizeof(signed short)));
        ucClusterMax = ((512/sizeof(unsigned short)) - 1);
      //ucClusterEntry &= ((512/sizeof(signed short)) - 1);
    }
    else {
        ulFAT += (ulPresentCluster/(512/sizeof(signed long)));
        ucClusterEntry &= ((512/sizeof(signed long)) - 1);
        ucClusterMax = ((512/sizeof(unsigned long)) - 1);
    }
    #else
    ulFAT += (ulPresentCluster/(512/sizeof(signed long)));               // {14}
    ucClusterEntry &= ((512/sizeof(signed long)) - 1);
    ucClusterMax = ((512/sizeof(unsigned long)) - 1);
    #endif
    ulFatOriginal = ulFAT;                                               // {15}
    ucOriginalCluster = ucClusterEntry;
    while (1) {
        if ((utReadDiskSector(ptr_utDisk, ulFAT, ulSectorContent)) != UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
            return (unsigned long)UTFAT_DISK_READ_ERROR;
        }
        while (1) {
            unsigned long ulClusterEntryContent;
    #ifdef UTFAT16
            if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                ulClusterEntryContent = ulSectorContent[ucClusterEntry/2];
                if (ucClusterEntry & 1) {
                    ulClusterEntryContent &= LITTLE_LONG_WORD(0xffff0000);
                }
                else {
                    ulClusterEntryContent &= LITTLE_LONG_WORD(0x0000ffff);
                }
            }
            else {
                ulClusterEntryContent = ulSectorContent[ucClusterEntry];
            }
    #else
            ulClusterEntryContent = ulSectorContent[ucClusterEntry];
    #endif
            if (ulClusterEntryContent == 0) {                            // next free cluster location found
                unsigned char ucFatCopies = 0;
                if (ucClusterType & (INITIALISE_DIR_CLUSTER | INITIALISE_DIR_EXTENSION)) { // the new cluster must be configured as empty directory
                    unsigned char ucSectors = ptr_utDisk->utFAT.ucSectorsPerCluster;
                    unsigned long ulSectorToDelete = ((ulAbsoluteCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);
                    while (ucSectors != 0) {
                        if ((ucSectors-- == ptr_utDisk->utFAT.ucSectorsPerCluster) && (ucClusterType & INITIALISE_DIR_CLUSTER)) { // first sector - add "." and ".." entries
                            unsigned char ucDirectoryDefault[512];
                            DIR_ENTRY_STRUCTURE_FAT32 *ptrEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ucDirectoryDefault;
                            uMemset(ucDirectoryDefault, 0x00, sizeof(ucDirectoryDefault));
                            uMemset(ptrEntry->DIR_Name, ' ', sizeof(ptrEntry->DIR_Name));
                            ptrEntry->DIR_Name[0] = '.';
                            fnAddEntry(ptrEntry, ulPresentCluster, DIR_ATTR_DIRECTORY);
                            ptrEntry++;
                            uMemset(ptrEntry->DIR_Name, ' ', sizeof(ptrEntry->DIR_Name));
                            ptrEntry->DIR_Name[0] = '.';
                            ptrEntry->DIR_Name[1] = '.';
                            fnAddEntry(ptrEntry, 2, DIR_ATTR_DIRECTORY);
                            while (utCommitSectorData(ptr_utDisk, ucDirectoryDefault, ulSectorToDelete) == CARD_BUSY_WAIT) {}
                        }
                        else {
                            while (utDeleteSector(ptr_utDisk, ulSectorToDelete) == CARD_BUSY_WAIT) {} // delete sector
                        }
                        ulSectorToDelete++;
                    }
                }
    #ifdef UTFAT16
                if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                    if (ucClusterEntry & 0x01) {
                        ulSectorContent[ucClusterEntry/2] |= (LITTLE_LONG_WORD(FAT16_CLUSTER_MASK << 16)); // mark last cluster in extension
                    }
                    else {
                        ulSectorContent[ucClusterEntry/2] |= (LITTLE_LONG_WORD(FAT16_CLUSTER_MASK)); // mark last cluster in extension
                    }
                }
                else {                                                   // else FAT32
                    ulSectorContent[ucClusterEntry] = LITTLE_LONG_WORD(CLUSTER_MASK); // mark last cluster in extension
                }
    #else
                ulSectorContent[ucClusterEntry] = LITTLE_LONG_WORD(CLUSTER_MASK); // mark last cluster in extension
    #endif
                if (ucClusterType & UPDATE_FAT_END) {
                    ucClusterType = 0;
    #ifdef UTFAT16
                    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                        if (ucClusterEntry & 0x01) {
                            ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0x0000ffff);
                            ulSectorContent[ucClusterEntry/2] |= LITTLE_LONG_WORD(ulAbsoluteCluster); // mark last cluster in extension
                        }
                        else {
                            ucClusterEntry -= 2;                         // {38}
                            ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0xffff0000);
                            ulSectorContent[ucClusterEntry/2] |= LITTLE_LONG_WORD(ulAbsoluteCluster << 16); // mark last cluster in extension
                        }
                    }
                    else {
                        ulSectorContent[ucOriginalCluster] = LITTLE_LONG_WORD(ulAbsoluteCluster); // modify the original end marker to point to the additional cluster
                    }
    #else
                    ulSectorContent[ucOriginalCluster] = LITTLE_LONG_WORD(ulAbsoluteCluster); // modify the original end marker to point to the additional cluster
    #endif
                }
                while (ucFatCopies < ptr_utDisk->utFAT.ucNumberOfFATs) {
                    while (utCommitSectorData(ptr_utDisk, ulSectorContent, (ulFAT + (ucFatCopies * ptr_utDisk->utFAT.ulFatSize))) == CARD_BUSY_WAIT) {} // write the new FAT entry
                    ucFatCopies++;
                }
                if (ucClusterType & UPDATE_FAT_END_IN_DIFF_SECT) {       // the new cluster end has been marked but the original end must be modified to point to it (it is in a different sector so needs to be modified seperately)
                    ucFatCopies = 0;
                    if ((utReadDiskSector(ptr_utDisk, ulFatOriginal, ulSectorContent)) != UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
                        return (unsigned long)UTFAT_DISK_READ_ERROR;
                    }
    #ifdef UTFAT16
                    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                        if (ucClusterEntry & 0x01) {
                            ulSectorContent[((unsigned char)ulPresentCluster/2)] &= ~LITTLE_LONG_WORD(0x0000ffff); // {38} unsigned char casting before divide by 2!!
                            ulSectorContent[((unsigned char)ulPresentCluster/2)] |= LITTLE_LONG_WORD(ulAbsoluteCluster); // mark last cluster in extension
                        }
                        else {
                            ulSectorContent[((unsigned char)ulPresentCluster/2)] &= ~LITTLE_LONG_WORD(0xffff0000);
                            ulSectorContent[((unsigned char)ulPresentCluster/2)] |= LITTLE_LONG_WORD(ulAbsoluteCluster << 16); // mark last cluster in extension
                        }
                    }
                    else {
                        ulSectorContent[ulPresentCluster & ((512/sizeof(signed long)) - 1)] = LITTLE_LONG_WORD(ulAbsoluteCluster);
                    }
    #else
                    ulSectorContent[ulPresentCluster & ((512/sizeof(signed long)) - 1)] = LITTLE_LONG_WORD(ulAbsoluteCluster);
    #endif
                    while (ucFatCopies < ptr_utDisk->utFAT.ucNumberOfFATs) {
                        while (utCommitSectorData(ptr_utDisk, ulSectorContent, (ulFatOriginal + (ucFatCopies * ptr_utDisk->utFAT.ulFatSize))) == CARD_BUSY_WAIT) {} // write the new FAT entry
                        ucFatCopies++;
                    }
                }
                if (ptr_utDisk->usDiskFlags & FSINFO_VALID) {
                    if (ulAbsoluteCluster >= ptr_utDisk->utFileInfo.ulNextFreeCluster) {                        
                        ptr_utDisk->utFileInfo.ulNextFreeCluster = (ulAbsoluteCluster + 1);
                    }
                    ptr_utDisk->usDiskFlags |= WRITEBACK_INFO_FLAG;      // mark that the info block information has changed
                    ptr_utDisk->utFileInfo.ulFreeClusterCount--;
                }
                return (ulAbsoluteCluster);
            }
            ulAbsoluteCluster++;                                         // {35}
            if (ucClusterEntry >= ucClusterMax) {
                break;
            }
            ucClusterEntry++;
          //ulAbsoluteCluster++;                                         // {35}
        }
        ucClusterType &= ~UPDATE_FAT_END;                                // the new FAT entry is in a different FAT sector so needs to be modified later
        ucClusterEntry = 0;
        ulFAT++;
    }
}

static void fnAddInfoSect(INFO_SECTOR_FAT32 *ptrInfoSector, unsigned long ulFreeCount, unsigned long ulNextFree)
{
    ptrInfoSector->FSI_Free_Count[0] = (unsigned char)(ulFreeCount);
    ptrInfoSector->FSI_Free_Count[1] = (unsigned char)(ulFreeCount >> 8);
    ptrInfoSector->FSI_Free_Count[2] = (unsigned char)(ulFreeCount >> 16);
    ptrInfoSector->FSI_Free_Count[3] = (unsigned char)(ulFreeCount >> 24);
    ptrInfoSector->FSI_Nxt_Free[0]   = (unsigned char)(ulNextFree);
    ptrInfoSector->FSI_Nxt_Free[1]   = (unsigned char)(ulNextFree >> 8);
    ptrInfoSector->FSI_Nxt_Free[2]   = (unsigned char)(ulNextFree >> 16);
    ptrInfoSector->FSI_Nxt_Free[3]   = (unsigned char)(ulNextFree >> 24);
}

static int fnCommitInfoChanges(UTDISK *ptr_utDisk)
{
    if (ptr_utDisk->usDiskFlags & WRITEBACK_INFO_FLAG) {                 // info sector content has changed
        INFO_SECTOR_FAT32 info_Sector;
        unsigned long ulInfoSector_Location = ptr_utDisk->utFileInfo.ulInfoSector;
        if ((utReadDiskSector(ptr_utDisk, ulInfoSector_Location, &info_Sector)) != UTFAT_SUCCESS) { // read the information sector
            return UTFAT_DISK_READ_ERROR;
        }
        fnAddInfoSect(&info_Sector, ptr_utDisk->utFileInfo.ulFreeClusterCount, ptr_utDisk->utFileInfo.ulNextFreeCluster);
        while (utCommitSectorData(ptr_utDisk, &info_Sector, ulInfoSector_Location) == CARD_BUSY_WAIT) {}
        ptr_utDisk->usDiskFlags &= ~WRITEBACK_INFO_FLAG;
    }
    return UTFAT_SUCCESS;
}
#endif
// Open a list directory referenced to its main directory object, which must already have been configured
//
extern int utOpenFile(const CHAR *ptrFilePath, UTFILE *ptr_utFile, unsigned short usAccessMode) // {8}
{
    int iReturn = UTFAT_SUCCESS;
    if (ptr_utFile->ptr_utDirObject == 0) {
        return UTFAT_SEARCH_INVALID;
    }    
    if ((utDisks[ptr_utFile->ptr_utDirObject->ucDrive].usDiskFlags & WRITE_PROTECTED_SD_CARD) && (usAccessMode & (UTFAT_OPEN_FOR_RENAME | UTFAT_OPEN_FOR_WRITE | UTFAT_OPEN_FOR_DELETE | UTFAT_OPEN_FOR_RENAME))) { // {34}
        return UTFAT_DISK_WRITE_PROTECTED;
    }
#if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0             // {40} allow SD card interface to be used without UTMANAGED_FILE_COUNT
    if (usAccessMode & UTFAT_MANAGED_MODE) {
        int iFileHandle = 0;
        UTMANAGED_FILE *ptrManagedFile = utManagedFiles;
        while (iFileHandle < UTMANAGED_FILE_COUNT) {
            if (ptrManagedFile->managed_owner == 0) {                
                if ((iReturn = utOpenFile(ptrFilePath, ptr_utFile, (unsigned short)(usAccessMode & ~UTFAT_MANAGED_MODE)) != UTFAT_PATH_IS_FILE)) {
                    return UTFAT_FILE_NOT_FOUND;                         // file doesn't exist
                }
                if (fnFileLocked(ptr_utFile) != 0) {
                    return UTFAT_FILE_LOCKED;                            // the file is locked by another user so may not be opened
                }
                ptrManagedFile->utManagedFile = ptr_utFile;
                ptrManagedFile->managed_owner = ptr_utFile->ownerTask;   // validate the entry with the owner task
                ptr_utFile->ucFileMode = (unsigned char)usAccessMode;
                ptrManagedFile->managed_mode = (unsigned char)usAccessMode;
                ptr_utFile->iFileHandle = iFileHandle;
                return UTFAT_PATH_IS_FILE;                               // open in managed mode was successful
            }
            iFileHandle++;
            ptrManagedFile++;
        }
        return MANAGED_FILE_NO_FILE_HANDLE;                              // all managed file spaces are presently occupied
    }
#endif
    if (usAccessMode & UTFAT_OPEN_FOR_RENAME) {
        ptr_utFile->ptr_utDirObject->usDirectoryFlags |= (UTDIR_DIR_AS_FILE | UTDIR_REFERENCED | UTDIR_SET_START); // opens for renames are set to allow directories to be handled as files 
    }
    else {
        ptr_utFile->ptr_utDirObject->usDirectoryFlags |= (UTDIR_REFERENCED | UTDIR_SET_START); // {6} the next open is a referenced open; set to start of lowest directory when file not found
    }
    if ((iReturn = utOpenDirectory(ptrFilePath, ptr_utFile->ptr_utDirObject)) == UTFAT_PATH_IS_FILE) {
        uMemcpy(&ptr_utFile->private_disk_location, &ptr_utFile->ptr_utDirObject->public_disk_location, sizeof(ptr_utFile->private_disk_location)); // copy the referenced directory details
        uMemcpy(&ptr_utFile->private_file_location, &ptr_utFile->ptr_utDirObject->public_file_location, sizeof(ptr_utFile->private_file_location)); // copy the referenced file start details
        uMemcpy(&ptr_utFile->public_file_location, &ptr_utFile->ptr_utDirObject->public_file_location, sizeof(ptr_utFile->public_file_location)); // copy the referenced file start details
        if (ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_Attr & DIR_ATTR_READ_ONLY) { // {10}
            usAccessMode &= ~(UTFAT_OPEN_FOR_DELETE | UTFAT_OPEN_FOR_WRITE); // the file is read-only so remove possible delete and write modes
        }
        ptr_utFile->ucFileMode = (unsigned char)usAccessMode;
        ptr_utFile->ulFileSize = ((ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_FileSize[3] << 24) + (ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_FileSize[2] << 16) + (ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_FileSize[1] << 8) + ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_FileSize[0]);
        ptr_utFile->ulFilePosition = 0;                                  // set position to start of file on open
        ptr_utFile->ucDrive = ptr_utFile->ptr_utDirObject->ucDrive;
#ifdef UTFAT_WRITE
        if (usAccessMode & UTFAT_TRUNCATE) {                             // the file is to be overwritten so delete its content
            fnDeleteFileContent(ptr_utFile, &utDisks[ptr_utFile->ucDrive], 0);
            ptr_utFile->ulFileSize = 0;
        }
        else if (UTFAT_APPEND & usAccessMode) {                          // {39}
            if ((iReturn = utSeek(ptr_utFile, 0, UTFAT_SEEK_END)) != UTFAT_SUCCESS) {// seek to the end of the file so that writes cause append
                return iReturn;
            }
        }
#endif
        return (UTFAT_PATH_IS_FILE);
    }
#ifdef UTFAT_WRITE
    else if (usAccessMode & UTFAT_CREATE) {                              // file doesn't exist so we should create it
        const CHAR *ptrFilename = ptrFilePath;
        DISK_LOCATION *ptrDiskLocation = &ptr_utFile->ptr_utDirObject->public_disk_location; // this is the start of the directory in which the new file is to be placed - this allows reuse of deleted file locations in the directory
        DIR_ENTRY_STRUCTURE_FAT32 *ptrFoundEntry;
        UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ptr_utDirObject->ucDrive];
        CHAR cFileName[8 + 3 + 1];                                       // space for 8:3 format plus an NT specific byte
        if (iReturn != UTFAT_FILE_NOT_FOUND) {                           // invalid path has been entered so return
            return iReturn;
        }
        do {
            iReturn = fnCreateNameParagraph(&ptrFilename, cFileName);    // run down the path until the file name is loaded
            if (iReturn < 0) {
                return UTFAT_INVALID_NAME;                               // {37}
            }
        } while ((iReturn != FULLY_QUALIFIED_SHORT_NAME) && (iReturn != FULLY_QUALIFIED_LONG_NAME));
        while (1) {
            if (fnLoadSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) != UTFAT_SUCCESS) {
                return UTFAT_DISK_READ_ERROR;
            }
            ptrFoundEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
            ptrFoundEntry += ptrDiskLocation->ucDirectoryEntry;          // move to the present entry
            if ((DIR_NAME_FREE == ptrFoundEntry->DIR_Name[0]) || (ptrFoundEntry->DIR_Name[0] == 0)) { // deleted entry or end of root entries found
                unsigned long ulFreeCluster = fnAllocateCluster(ptr_utDisk, 0, 0);
                uMemcpy(ptrFoundEntry->DIR_Name, cFileName, 11);         // add the short name
                ptrFoundEntry->DIR_NTRes = cFileName[11];
                fnAddEntry(ptrFoundEntry, ulFreeCluster, DIR_ATTR_ARCHIVE);
                while (utCommitSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
                ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;
                uMemcpy(&ptr_utFile->private_disk_location, &ptr_utFile->ptr_utDirObject->public_disk_location, sizeof(ptr_utFile->private_disk_location)); // copy the referenced directory details
                ptr_utFile->private_file_location.ulCluster = ulFreeCluster;
                ptr_utFile->private_file_location.ulSector = (ulFreeCluster * ptr_utDisk->utFAT.ucSectorsPerCluster);// section referenced to logical base address
                ptr_utFile->private_file_location.ulSector += ptr_utDisk->ulVirtualBaseAddress;// the sector in which the file content begins
                uMemcpy(&ptr_utFile->public_file_location, &ptr_utFile->private_file_location, sizeof(ptr_utFile->private_file_location)); // copy the referenced file start details
                ptr_utFile->ucFileMode = (unsigned char)usAccessMode;
                ptr_utFile->ulFileSize = 0;
                ptr_utFile->ulFilePosition = 0;                          // set position to start of file on open
                ptr_utFile->ucDrive = ptr_utFile->ptr_utDirObject->ucDrive;
                fnCommitInfoChanges(ptr_utDisk);
                return UTFAT_PATH_IS_FILE;
            }
            else {
                int iResult = fnNextDirectoryEntry(ptr_utDisk, ptrDiskLocation); // {17}
                if (UTFAT_DIRECTORY_AREA_EXHAUSTED == iResult) {         // the present directory cluster end has been reached so a new one must be started
                    ptrDiskLocation->directory_location.ulSector--;      // set back to previous sector
                    iResult = fnDirectorySectorCreate(ptr_utDisk, &ptrDiskLocation->directory_location); // create additional directory cluster
                }
                if (iResult != UTFAT_SUCCESS) {
                    return iResult;                                      // return error
                }                                                        // else continue to create the file entry in the new cluster
            }
        }
    }
#endif
    else if (iReturn == UTFAT_SUCCESS) {                                 // a directory was matched 
        uMemcpy(&ptr_utFile->private_disk_location, &ptr_utFile->ptr_utDirObject->public_disk_location, sizeof(ptr_utFile->ptr_utDirObject->public_disk_location)); // copy the referenced directory details
        ptr_utFile->ucFileMode = UTFAT_FILE_IS_DIR;                      // mark that the entry is not a file but a directory
        ptr_utFile->ucDrive = ptr_utFile->ptr_utDirObject->ucDrive;
    }
    return iReturn;
}

#if defined UTFAT_WRITE                                                  // {40} allow operation without write support
// Rename a referenced file
//
extern int utRenameFile(const CHAR *ptrFilePath, UTFILE *ptr_utFile)
{
    DIR_ENTRY_STRUCTURE_FAT32 *ptrFileEntry;
    CHAR cFileName[8 + 3 + 1];                                           // space for 8:3 format plus an NT specific byte
    const CHAR *ptrFilename = ptrFilePath;
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
    if (ptr_utDisk->usDiskFlags & WRITE_PROTECTED_SD_CARD) {             // {34}
        return UTFAT_DISK_WRITE_PROTECTED;
    }
    if (!(ptr_utFile->ucFileMode & (UTFAT_OPEN_FOR_RENAME | UTFAT_OPEN_FOR_DELETE))) { // {9} if the file is neither open for rename or write don't allow a rename
        return UTFAT_FILE_NOT_WRITEABLE;
    }
    if ((fnCreateNameParagraph(&ptrFilename, cFileName)) < 0) {          // fill the file name with a single paragraph from the file path
        return UTFAT_PATH_NOT_FOUND;
    }
    ptrFileEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
    ptrFileEntry += ptr_utFile->private_disk_location.ucDirectoryEntry;  // move to the file entry
    if (fnLoadSector(ptr_utDisk, ptr_utFile->private_disk_location.directory_location.ulSector) != UTFAT_SUCCESS) { // ensure that the directory sector is loaded
        return UTFAT_DISK_READ_ERROR;
    }

    uMemcpy(ptrFileEntry->DIR_Name, cFileName, 11);                      // add the short name
    ptrFileEntry->DIR_NTRes = cFileName[11];
    while (utCommitSector(ptr_utDisk, ptr_utFile->private_disk_location.directory_location.ulSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
    ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;
    return UTFAT_SUCCESS;
}
#endif

// A close only has real significance when the file was opened as a managed file
//
extern void utCloseFile(UTFILE *ptr_utFile)
{
#if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0             // {40} allow SD card interface to be used without UTMANAGED_FILE_COUNT
    if (ptr_utFile->ucFileMode & UTFAT_MANAGED_MODE) {
        utManagedFiles[ptr_utFile->iFileHandle].managed_owner = 0;       // free the managed file entry
    }
#endif
    uMemset(ptr_utFile, 0, sizeof(UTFILE));
}


#if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0             // {40} allow SD card interface to be used without UTMANAGED_FILE_COUNT
// The file has changed so update all user information
//
static void fnSynchroniseManagedFile(UTFILE *ptr_utFile)
{
    int iFileHandle = 0;
    while (iFileHandle < UTMANAGED_FILE_COUNT) {
        if (utManagedFiles[iFileHandle].managed_owner != 0) {
            if (uMemcmp(&ptr_utFile->private_file_location, &utManagedFiles[iFileHandle].utManagedFile->private_file_location, sizeof(FILE_LOCATION)) == 0) {
                utManagedFiles[iFileHandle].utManagedFile->ulFileSize = ptr_utFile->ulFileSize; // update the file size
                if (utManagedFiles[iFileHandle].utManagedFile->ulFileSize < utManagedFiles[iFileHandle].utManagedFile->ulFilePosition) {
                    utManagedFiles[iFileHandle].utManagedFile->ulFilePosition = utManagedFiles[iFileHandle].utManagedFile->ulFileSize;
                }
            }
        }
        iFileHandle++;
    }
}

// Check whether the file is locked by another user
//
static int fnFileLocked(UTFILE *ptr_utFile)
{
    int iFileHandle = 0;
    while (iFileHandle < UTMANAGED_FILE_COUNT) {
        if (utManagedFiles[iFileHandle].managed_owner != 0) {
            if (uMemcmp(&ptr_utFile->private_file_location, &utManagedFiles[iFileHandle].utManagedFile->private_file_location, sizeof(FILE_LOCATION)) == 0) {
                if (utManagedFiles[iFileHandle].managed_mode & UTFAT_PROTECTED) {
                    return 1;                                            // another user is locking this file
                }
            }
        }
        iFileHandle++;
    }
    return 0;                                                            // this file is not being locked by another user
}
#endif

#if defined UTFAT_WRITE                                                  // {40} allow operation without write support
extern int utWriteFile(UTFILE *ptr_utFile, unsigned char *ptrBuffer, unsigned short usLength)
{
    unsigned short usWriteLength;
    unsigned short usRemainingBuffer;
    unsigned char  ucDataContent[512];                                   // temporary buffer
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
    ptr_utFile->usLastReadWriteLength = 0;
    if (ptr_utDisk->usDiskFlags & WRITE_PROTECTED_SD_CARD) {             // {34}
        return UTFAT_DISK_WRITE_PROTECTED;
    }
    if (!(ptr_utFile->ucFileMode & UTFAT_OPEN_FOR_WRITE)) {              // only allow writes if the file is open for writing
        return UTFAT_FILE_NOT_WRITEABLE;
    }
    while (usLength != 0) {
        usRemainingBuffer = (unsigned short)(ptr_utFile->ulFilePosition % 512);
        if ((usRemainingBuffer != 0) || ((usLength < 512) && (usLength < (ptr_utFile->ulFileSize - ptr_utFile->ulFilePosition)))) { // if the complete content is not being overwritten            
            if (utReadDiskSector(ptr_utDisk, ptr_utFile->public_file_location.ulSector, ucDataContent) != UTFAT_SUCCESS) { // read directly to buffer
                return UTFAT_DISK_READ_ERROR;
            }
        }
        else {
            uMemset(ucDataContent, 0, sizeof(ucDataContent));
        }
        usWriteLength = usLength;
        if (usWriteLength > (512 - usRemainingBuffer)) {
            usWriteLength = (512 - usRemainingBuffer);
        }
        uMemcpy(&ucDataContent[usRemainingBuffer], ptrBuffer, usWriteLength);
        while (utCommitSectorData(ptr_utDisk, ucDataContent, ptr_utFile->public_file_location.ulSector) == CARD_BUSY_WAIT) {}
        ptrBuffer += usWriteLength;
        ptr_utFile->ulFilePosition += usWriteLength;
        if (ptr_utFile->ulFilePosition > ptr_utFile->ulFileSize) {
            ptr_utFile->ulFileSize = ptr_utFile->ulFilePosition;
        }
        ptr_utFile->usLastReadWriteLength += usWriteLength;              // the number of bytes written
        usLength -= usWriteLength;
        if ((usRemainingBuffer + usWriteLength) >= 512) {
            int iResult = fnNextSectorCreate(ptr_utDisk, &(ptr_utFile->public_file_location));
            if (iResult != UTFAT_SUCCESS) {                              // move to next sector/cluster and create additional cluster if necessary
                return iResult;
            }
        }
    }
    if (ptr_utFile->usLastReadWriteLength != 0) {                        // updata the file entry if there was a change
        DIR_ENTRY_STRUCTURE_FAT32 *ptrFileEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
        ptrFileEntry += ptr_utFile->private_disk_location.ucDirectoryEntry; // move to the file entry
        if (fnLoadSector(ptr_utDisk, ptr_utFile->private_disk_location.directory_location.ulSector) != UTFAT_SUCCESS) { // ensure that the directory sector is loaded
            return UTFAT_DISK_READ_ERROR;
        }
        fnSetFileInformation(ptrFileEntry, ptr_utFile->ulFileSize);
        while (utCommitSector(ptr_utDisk, ptr_utFile->private_disk_location.directory_location.ulSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
        ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;
    }
    #if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0         // {40} allow SD card interface to be used without UTMANAGED_FILE_COUNT
    fnSynchroniseManagedFile(ptr_utFile);
    #endif
    return UTFAT_SUCCESS;
}
#endif

// Change the directory location with reference to its present position
//
extern int utChangeDirectory(const CHAR *ptrDirPath, UTDIRECTORY *ptrDirObject)
{
    unsigned short usPathTerminator = ptrDirObject->usRelativePathLocation; // original path location
    ptrDirObject->usDirectoryFlags |= UTDIR_ALLOW_MODIFY_PATH;           // allow the open to modify the path if necessary
    switch (utOpenDirectory(ptrDirPath, ptrDirObject)) {                 // {7} try to locate the new directory
    case UTFAT_PATH_IS_ROOT_REF:
        ptrDirObject->usRelativePathLocation = usPathTerminator = 3;     // reset the root string
        ptrDirPath++;                                                    // jump the root reference and fall through to set new path
    case UTFAT_SUCCESS:
        if (ptrDirObject->ptrDirectoryPath != 0) {
            if (*ptrDirPath != 0) {
                if (usPathTerminator > 3) {
                    ptrDirObject->ptrDirectoryPath[usPathTerminator] = '\\';
                }
                if (usPathTerminator != 3) {
                    usPathTerminator++;
                }
                ptrDirObject->usRelativePathLocation = (uStrcpy(&ptrDirObject->ptrDirectoryPath[usPathTerminator], ptrDirPath) - ptrDirObject->ptrDirectoryPath);
            }
            else {
                ptrDirObject->usRelativePathLocation = usPathTerminator;
                if (usPathTerminator == 2) {
                    usPathTerminator = 3;
                }
                ptrDirObject->ptrDirectoryPath[usPathTerminator] = 0;
            }
        }
        break;
    case UTFAT_PATH_IS_ROOT:                                             // change was to the root
        ptrDirObject->usRelativePathLocation = 3;                        // reset the root string
        ptrDirObject->ptrDirectoryPath[3] = 0;
        break;
    case UTFAT_SUCCESS_PATH_MODIFIED:                                    // successful and no further action required since it has been performed
        break;
    default:
         return UTFAT_PATH_NOT_FOUND;
    }
    uMemcpy(&ptrDirObject->private_disk_location, &ptrDirObject->public_disk_location, sizeof(ptrDirObject->private_disk_location)); // commit the new directory location
    return UTFAT_SUCCESS;
}

// Move through sectors and cluster chain until the end sector is reached
//
static int fnRunThroughClusters(UTDISK *ptr_utDisk, FILE_LOCATION *prtFileLocation, unsigned long ulStartPosition, unsigned long ulEndPosition)
{
    int iResult = UTFAT_SUCCESS;
    unsigned short usBytesPerSector = ptr_utDisk->utFAT.usBytesPerSector;
    ulStartPosition &= ~(usBytesPerSector - 1);                          // round down to start of present sector
    ulEndPosition &= ~(usBytesPerSector - 1);                            // round down to start of end sector
    while (ulStartPosition < ulEndPosition) {
        ulStartPosition += usBytesPerSector;
        if ((iResult = fnNextSectorCreate(ptr_utDisk, prtFileLocation)) != UTFAT_SUCCESS) { // {42} if the end of the file is reached, which is at the end of a cluster, a new cluster will be created if write is supported
            break;
        }
    }
    return iResult;
}

extern int utSeek(UTFILE *ptr_utFile, unsigned long ulPosition, int iSeekType)
{
    int iResult = UTFAT_SUCCESS;
    unsigned long ulNewPosition = 0;                                     // default to start of file
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
    switch (iSeekType) {
    case UTFAT_SEEK_SET:                                                 // relative to the start of the file - always forwards
        ulNewPosition = ulPosition;
        break;
    case UTFAT_SEEK_CUR:                                                 // relative to current position
        if ((signed long)ulPosition < 0) {                               // relative backwards
            if (-ulPosition > ptr_utFile->ulFilePosition) {
                break;                                                   // move to beginning of file
            }
        }
        else {                                                           // relative forwards
            if (ulPosition > (ptr_utFile->ulFileSize - ptr_utFile->ulFilePosition)) {
                ulPosition = (ptr_utFile->ulFileSize - ptr_utFile->ulFilePosition);
            }
        }
        ulNewPosition = ptr_utFile->ulFilePosition + ulPosition;
        break;
    case UTFAT_SEEK_END:                                                 // relative to the end of the file (always backwards)
        if (ulPosition < ptr_utFile->ulFileSize) {
            ulNewPosition = (ptr_utFile->ulFileSize - ulPosition);
        }
        break;
    }
    if (ulNewPosition != ptr_utFile->ulFilePosition) {                   // update the cluster pointer
        if (ulNewPosition < ptr_utFile->ulFilePosition) {                // move back
            uMemcpy(&ptr_utFile->public_file_location, &ptr_utFile->private_file_location, sizeof(ptr_utFile->private_file_location)); // rewind to start and search from there
            ptr_utFile->ulFilePosition = 0;
        }
        else if (ulNewPosition > ptr_utFile->ulFileSize) {               // {42} limit seek to end of file
            ulNewPosition = ptr_utFile->ulFileSize;                      
        }
        iResult = fnRunThroughClusters(ptr_utDisk, &ptr_utFile->public_file_location, ptr_utFile->ulFilePosition, ulNewPosition);
        if (iResult == UTFAT_SUCCESS) {                                  // if the cluster location could be successfully set
            ptr_utFile->ulFilePosition = ulNewPosition;                  // set the file position to the new position
        }
    }
    return iResult;
}

#if defined UTFAT_WRITE                                                  // support writing as well as reading

    #if defined NAND_FLASH_FAT                                           // {29}

static int utCommitSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    switch (iMemoryOperation & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            if (fnCheckBlankPage(ulSectorNumber) != 0) {
                if (fnOverWriteSector(ulSectorNumber, ptr_utDisk->ptrSectorData) != UTFAT_SUCCESS) {
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            else {
                if (fnWriteNANDsector(ulSectorNumber, 0, ptr_utDisk->ptrSectorData, 512) != 0) {
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            ptr_utDisk->ulPresentSector = ulSector;                      // update the sector which is presently valid
            iMemoryOperation &= ~_WRITING_MEMORY;                        // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

static int utCommitSectorData(UTDISK *ptr_utDisk, void *ptrBuffer, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    switch (iMemoryOperation & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            if (fnCheckBlankPage(ulSectorNumber) != 0) {
                if (fnOverWriteSector(ulSectorNumber, ptrBuffer) != UTFAT_SUCCESS) {
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            else {
                if (fnWriteNANDsector(ulSector, 0, ptrBuffer, 512) != UTFAT_SUCCESS) {
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            iMemoryOperation &= ~_WRITING_MEMORY;                        // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Delete the specified sector by writing data content of 0x00
//
static int utDeleteSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    switch (iMemoryOperation & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            unsigned long ulTemp[512/sizeof(unsigned long)];         // temporary long-word aligned buffer
    #ifdef LONG_UMEMSET
            uMemset_long(ulTemp, 0x00, sizeof(ulTemp));              // zero buffer content for delete
    #else
            uMemset(ulTemp, 0x00, sizeof(ulTemp));
    #endif
            if (fnWriteNANDsector(ulSector, 0, (unsigned char *)ulTemp, 512) != 0) {
                fnMemoryDebugMsg("Write error\r\n");
                return UTFAT_DISK_WRITE_ERROR;
            } 
            iMemoryOperation &= ~_WRITING_MEMORY;                        // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}
    #else
static int utCommitSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            unsigned char ucResult;
            if ((iActionResult = fnSendSD_command(fnCreateCommand(WRITE_BLOCK_CMD24, ulSector), &ucResult, 0)) != 0) {
                return iActionResult;
            }
            if (ucResult == 0) {
                if (fnPutSector(ptr_utDisk->ptrSectorData) != UTFAT_SUCCESS) { // start writing buffer to single sector
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
                if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
                    ulSector /= 512;                                     // convert back to sector number from byte offset
                }
                ptr_utDisk->ulPresentSector = ulSector;                  // update the sector which is presently valid
            }
            SET_SD_CS_HIGH();
            iMemoryOperation &= ~_WRITING_MEMORY;                        // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

static int utCommitSectorData(UTDISK *ptr_utDisk, void *ptrBuffer, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            unsigned char ucResult;
            if ((iActionResult = fnSendSD_command(fnCreateCommand(WRITE_BLOCK_CMD24, ulSector), &ucResult, 0)) != 0) {
                return iActionResult;
            }
            if (ucResult == 0) {
                if (fnPutSector((unsigned char *)ptrBuffer) != UTFAT_SUCCESS) { // start writing buffer to single sector
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            SET_SD_CS_HIGH();
            iMemoryOperation &= ~_WRITING_MEMORY;                        // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Delete the specified sector by writing data content of 0x00
//
static int utDeleteSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            unsigned char ucResult;
            if ((iActionResult = fnSendSD_command(fnCreateCommand(WRITE_BLOCK_CMD24, ulSector), &ucResult, 0)) != 0) {
                return iActionResult;
            }
            if (ucResult == 0) {
                unsigned long ulTemp[512/sizeof(unsigned long)];         // temporary long-word aligned buffer
    #ifdef LONG_UMEMSET
                uMemset_long(ulTemp, 0x00, sizeof(ulTemp));              // zero buffer content for delete
    #else
                uMemset(ulTemp, 0x00, sizeof(ulTemp));
    #endif
                if (fnPutSector((unsigned char *)ulTemp) != UTFAT_SUCCESS) { // write sector content to 0x00
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            SET_SD_CS_HIGH();
            iMemoryOperation &= ~_WRITING_MEMORY;                        // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}
    #endif

static int fnDeleteClusterChain(unsigned long ulClusterStart, unsigned char ucDrive, int iDestroyClusters)
{
    UTDISK *ptr_utDisk = &utDisks[ucDrive];
    unsigned long ulCluster = (((ulClusterStart - ptr_utDisk->ulLogicalBaseAddress)/ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulDirectoryBase);
    unsigned long ulClusterSector;
    unsigned long ulNextCluster, ulNextClusterSector;
    unsigned long ulRemovedClusters = 0;
    unsigned char ucClusterEntry;
    unsigned long ulSectorContent[512/sizeof(signed long)];              // temporary long-word aligned buffer
#ifdef UTFAT16
    unsigned long ulClusterMask;
    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulCluster >> 8)); // section where the FAT responsible for this cluster resides
        ulCluster -= (32 - 1);                                           // {38} compensate for fixed 16k boot sector
        ucClusterEntry = (unsigned char)ulCluster;
        ulClusterMask = FAT16_CLUSTER_MASK;
    }
    else {
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulCluster >> 7)); // section where the FAT responsible for this cluster resides
        ucClusterEntry = (unsigned char)(ulCluster & 0x7f);
        ulClusterMask = CLUSTER_MASK;
    }
#else
    ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulCluster >> 7)); // section where the FAT responsible for this cluster resides
    ucClusterEntry = (unsigned char)(ulCluster & 0x7f);
#endif
    if ((utReadDiskSector(ptr_utDisk, ulClusterSector, ulSectorContent)) != UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
        return UTFAT_DISK_READ_ERROR;
    }
    while (1) {
#ifdef UTFAT16
        if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
            ulNextCluster = LITTLE_LONG_WORD(ulSectorContent[ucClusterEntry/2]); // {36}
            if (ucClusterEntry & 1) {
                ulNextCluster &= LITTLE_LONG_WORD(0xffff0000);
                ulNextCluster >>= 16;
            }
            else {
                ulNextCluster &= LITTLE_LONG_WORD(0x0000ffff);
            }
        }
        else {
            ulNextCluster = LITTLE_LONG_WORD(ulSectorContent[ucClusterEntry]); // {36} read the next entry
        }
        if (((ulNextCluster & ulClusterMask) == ulClusterMask) || (ulNextCluster <= ptr_utDisk->ulDirectoryBase)) // check whether the end of the cluster chain has been reached (if the next cluster is 0, 1 or 2 it is ignored since it must be corrupt - 0, 1 are not used and 2 is always the start of the root directory! [FAT16 is 0..1])
#else
        ulNextCluster = LITTLE_LONG_WORD(ulSectorContent[ucClusterEntry]); // {36} read the next entry
        if (((ulNextCluster & CLUSTER_MASK) == CLUSTER_MASK) || (ulNextCluster <= ptr_utDisk->ulDirectoryBase)) // check whether the end of the cluster chain has been reached (if the next cluster is 0, 1 or 2 it is ignored since it must be corrupt - 0, 1 are not used and 2 is always the start of the root directory!)
#endif
        {
            int iResult = UTFAT_SUCCESS;
            unsigned char ucFatCopies = 0;
            if (iDestroyClusters == 0) {                                 // if the file's cluster is to be reused and only occupied one cluster, don't delete it
                break;
            }
            else {
#ifdef UTFAT16
                if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                    if (ucClusterEntry & 1) {
                        ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0xffff0000);
                    }
                    else {
                        ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0x0000ffff);
                    }
                }
                else {
                    ulSectorContent[ucClusterEntry] = 0;                 // delete the previous entry
                }
#else
                ulSectorContent[ucClusterEntry] = 0;                     // delete the previous entry
#endif
                if (ptr_utDisk->usDiskFlags & FSINFO_VALID) {
                    ptr_utDisk->utFileInfo.ulFreeClusterCount += ++ulRemovedClusters;
                    ptr_utDisk->usDiskFlags |= WRITEBACK_INFO_FLAG;      // mark that the info block information has changed
                }
            }
            while (ucFatCopies < ptr_utDisk->utFAT.ucNumberOfFATs) {
                while ((iResult = utCommitSectorData(ptr_utDisk, ulSectorContent, (ulClusterSector + (ucFatCopies * ptr_utDisk->utFAT.ulFatSize)))) == CARD_BUSY_WAIT) {}
                ucFatCopies++;
            }
            return iResult;
        }
        if (iDestroyClusters == 0) {                                     // if the file's cluster is to be reused
#ifdef UTFAT16
            if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                if (ucClusterEntry & 1) {
                    ulSectorContent[ucClusterEntry/2] = ulSectorContent[ucClusterEntry/2] & ~LITTLE_LONG_WORD(0xffff0000);
                    ulSectorContent[ucClusterEntry/2] = (ulSectorContent[ucClusterEntry/2] | LITTLE_LONG_WORD(FAT16_CLUSTER_MASK << 16)); // mark last cluster in extension
                }
                else {
                    ulSectorContent[ucClusterEntry/2] = ulSectorContent[ucClusterEntry/2] & ~LITTLE_LONG_WORD(0x0000ffff);
                    ulSectorContent[ucClusterEntry/2] = (ulSectorContent[ucClusterEntry/2] | LITTLE_LONG_WORD(FAT16_CLUSTER_MASK)); // mark last cluster in extension
                }
            }
            else {
                ulSectorContent[ucClusterEntry] = CLUSTER_MASK;          // mark the end of the cluster chain
            }
#else
            ulSectorContent[ucClusterEntry] = CLUSTER_MASK;              // mark the end of the cluster chain
#endif
            iDestroyClusters = 1;                                        // from here on, destroy rest of chain
        }
        else {
#ifdef UTFAT16
            if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                if (ucClusterEntry & 1) {
                    ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0xffff0000);
                }
                else {
                    ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0x0000ffff);
                }
            }
            else {
                ulSectorContent[ucClusterEntry] = 0;                     // delete the previous entry
            }
#else
            ulSectorContent[ucClusterEntry] = 0;                         // delete the previous entry
#endif
            ulRemovedClusters++;
        }
#ifdef UTFAT16
        if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
            ulNextClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulNextCluster >> 8));
            ucClusterEntry = (unsigned char)(ulNextCluster);
        }
        else {
            ulNextClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulNextCluster >> 7));
            ucClusterEntry = (unsigned char)(ulNextCluster & 0x7f);
        }
#else
        ulNextClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulNextCluster >> 7));
        ucClusterEntry = (unsigned char)(ulNextCluster & 0x7f);
#endif
        if (ulNextClusterSector != ulClusterSector) {                    // moving to a new sector so update the FAT
            unsigned char ucFatCopies = 0;
            while (ucFatCopies < ptr_utDisk->utFAT.ucNumberOfFATs) {
                while (utCommitSectorData(ptr_utDisk, ulSectorContent, (ulClusterSector + (ucFatCopies * ptr_utDisk->utFAT.ulFatSize))) == CARD_BUSY_WAIT) {}
                ucFatCopies++;
            }
            while (utCommitSectorData(ptr_utDisk, ulSectorContent, ulClusterSector) == CARD_BUSY_WAIT) {}
            ulClusterSector = ulNextClusterSector;
            if ((utReadDiskSector(ptr_utDisk, ulClusterSector, ulSectorContent)) != UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
                return UTFAT_DISK_READ_ERROR;
            }
        }
    }
    return UTFAT_SUCCESS;
}

static int fnDeleteFileContent(UTFILE *ptr_utFile, UTDISK *ptr_utDisk, int iDestroyClusters)
{
#if defined UTFAT_LFN_READ && defined UTFAT_LFN_DELETE                   // {45}  
    LFN_ENTRY_STRUCTURE_FAT32 *ptrFileEntry;
    DISK_LOCATION FileLocation;
    unsigned long ulSectorToWrite = ptr_utDisk->ulPresentSector;
#endif
    int iResult;
    if (iDestroyClusters == 0) {                                         // the file is to be re-used so set length to zero
        fnSetFileInformation(ptr_utFile->ptr_utDirObject->ptrEntryStructure, 0);// set the file size to zero
    }
#if defined UTFAT_LFN_READ && defined UTFAT_LFN_DELETE                   // {45} if long file names are in use, delete also possible long file name    
    uMemcpy(&FileLocation, &(ptr_utFile->private_disk_location), sizeof(DISK_LOCATION)); // make a copy of the file entry details
    ptr_utDisk->usDiskFlags |= WRITEBACK_BUFFER_FLAG;                    // mark that the original sector content must be committed
    while (1) {
        if (FileLocation.ucDirectoryEntry == 0) {                        // if at a sector boundary
            while (utCommitSector(ptr_utDisk, ulSectorToWrite) == CARD_BUSY_WAIT) {} // commit the sector since we will be loading new data
            ptr_utDisk->usDiskFlags &= ~(WRITEBACK_BUFFER_FLAG);         // original sector must no longer be committed
            FileLocation.ucDirectoryEntry = ((ptr_utDisk->utFAT.usBytesPerSector/sizeof(DIR_ENTRY_STRUCTURE_FAT32)) - 1);
            FileLocation.directory_location.ulSector--;            
            if (fnLoadSector(ptr_utDisk, FileLocation.directory_location.ulSector) != UTFAT_SUCCESS) { // load previous sector
                return UTFAT_DISK_READ_ERROR;
            }
            ulSectorToWrite = FileLocation.directory_location.ulSector;
        }
        else {
            FileLocation.ucDirectoryEntry--;
        }
        ptrFileEntry = (LFN_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData;
        ptrFileEntry += FileLocation.ucDirectoryEntry;
        if ((ptrFileEntry->LFN_Attribute & DIR_ATTR_MASK) == DIR_ATTR_LONG_NAME) {
            unsigned char ucEntryNumber = ptrFileEntry->LFN_EntryNumber;
            ptrFileEntry->LFN_EntryNumber = DIR_NAME_FREE;               // delete the entry in the long file name
            ptr_utDisk->usDiskFlags |= WRITEBACK_BUFFER_FLAG;            // a change has been made to the present sector so ensure that it is committed
            if (ucEntryNumber & 0x40) {                                  // if this is the initial entry we can quit
                break;
            }
            continue;
        }
        break;                                                           // all parts of the long file name have been deleted
    }
    if (ptr_utDisk->usDiskFlags & WRITEBACK_BUFFER_FLAG) {               // if there is sector content not yet committed it is done now
        while (utCommitSector(ptr_utDisk, ulSectorToWrite) == CARD_BUSY_WAIT) {} // commit the sector
        ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;
    }
#else 
    while (utCommitSector(ptr_utDisk, ptr_utDisk->ulPresentSector) == CARD_BUSY_WAIT) {} // commit the file entry change
    ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;
#endif
    if (ptr_utFile->private_file_location.ulSector >= ptr_utDisk->ulLogicalBaseAddress) { // {46} don't delete cluster chain if there is none allocated
        iResult = fnDeleteClusterChain(ptr_utFile->private_file_location.ulSector, ptr_utDisk->ucDriveNumber, iDestroyClusters); // free up all clusters belonging to the file content
        if (iResult < 0) {
            return iResult;                                              // return error cause
        }
    }
    return fnCommitInfoChanges(ptr_utDisk);                              // update the info sector accordingly if it has changed
}

extern int utDeleteFile(const CHAR *ptrFilePath, UTDIRECTORY *ptrDirObject)
{
    int iResult;
    UTFILE utFile;
    UTDISK *ptr_utDisk = &utDisks[ptrDirObject->ucDrive];
    utFile.ptr_utDirObject = ptrDirObject;
    ptrDirObject->usDirectoryFlags |= (UTDIR_TEST_REL_PATH);
    iResult = utOpenFile(ptrFilePath, &utFile, UTFAT_OPEN_FOR_DELETE);
    if (iResult == UTFAT_SUCCESS) {                                      // directory has been located
        DISK_LOCATION *ptrDirContent = &utFile.private_disk_location;
        DIR_ENTRY_STRUCTURE_FAT32 *ptrDirEntry;
        uMemcpy(&utFile.private_file_location, &utFile.private_disk_location.directory_location, sizeof(utFile.private_disk_location.directory_location));
        do {
            if (fnLoadSector(ptr_utDisk, utFile.private_disk_location.directory_location.ulSector) != UTFAT_SUCCESS) {
                return UTFAT_DISK_READ_ERROR;
            }
            ptrDirEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
            ptrDirEntry += ptrDirContent->ucDirectoryEntry;              // move to the present entry
            if (ptrDirEntry->DIR_Name[0] == 0) {
                break;                                                   // directory is empty so allow delete
            }
            else if (ptrDirEntry->DIR_Name[0] != DIR_NAME_FREE) {
                return UTFAT_DIR_NOT_EMPTY;
            }
            if (fnNextDirectoryEntry(ptr_utDisk, ptrDirContent) == UTFAT_DIRECTORY_AREA_EXHAUSTED) { // move to the next entry
                return UTFAT_DIRECTORY_AREA_EXHAUSTED;
            }
        } while (1);
        ptrDirObject->usDirectoryFlags |= (UTDIR_TEST_REL_PATH | UTDIR_DIR_AS_FILE); // handle a directory as a file so that it can also be deleted if found
        iResult = utOpenFile(ptrFilePath, &utFile, UTFAT_OPEN_FOR_DELETE);
        if (iResult == UTFAT_SUCCESS) {
            utFile.ptr_utDirObject->ptrEntryStructure->DIR_Name[0] = DIR_NAME_FREE; // delete the (short) file name
            return (fnDeleteFileContent(&utFile, &utDisks[ptrDirObject->ucDrive], 1));
        }
    }
    else if (iResult == UTFAT_PATH_IS_FILE) {                            // if file has been located
        if (utFile.ptr_utDirObject->ptrEntryStructure->DIR_Attr & DIR_ATTR_READ_ONLY) { // {10}
            return UTFAT_FILE_NOT_WRITEABLE;                             // the file is read-only so can not be deleted
        }
        utFile.ptr_utDirObject->ptrEntryStructure->DIR_Name[0] = DIR_NAME_FREE; // delete the (short) file name
        return (fnDeleteFileContent(&utFile, &utDisks[ptrDirObject->ucDrive], 1));
    }
    return iResult;
}

// Make a new directory - don't allow if a directory of the same name already exists
//
extern int utMakeDirectory(const CHAR *ptrDirPath, UTDIRECTORY *ptrDirObject)
{
    int iReturn;
    DISK_LOCATION *ptrDiskLocation = &(ptrDirObject->public_disk_location);
    DISK_LOCATION found_location;
    UTDISK *ptr_utDisk = &utDisks[ptrDirObject->ucDrive];
    DIR_ENTRY_STRUCTURE_FAT32 *ptrFoundEntry;
    const CHAR *ptrDirname = ptrDirPath;
    CHAR cFileName[8 + 3 + 1];                                           // space for 8:3 format plus an NT specific byte

    if (ptr_utDisk->usDiskFlags & WRITE_PROTECTED_SD_CARD) {             // {34}
        return UTFAT_DISK_WRITE_PROTECTED;
    }
    ptrDirObject->usDirectoryFlags |= UTDIR_SET_START;                   // {6} set to start of lowest directory when file not found
    if ((iReturn = utOpenDirectory(ptrDirPath, ptrDirObject)) != UTFAT_FILE_NOT_FOUND) {
        return UTFAT_PATH_NOT_FOUND;
    }
    do {
        iReturn = fnCreateNameParagraph(&ptrDirname, cFileName);         // run down the path until the directory name is loaded
        if (iReturn < 0) {
            return UTFAT_INVALID_NAME;                                   // {37}
        }
    } while ((iReturn != FULLY_QUALIFIED_SHORT_NAME) && (iReturn != FULLY_QUALIFIED_LONG_NAME));

    found_location.directory_location.ulCluster = 0;

    while (1) {
        if (fnLoadSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) != UTFAT_SUCCESS) {
            return UTFAT_DISK_READ_ERROR;
        }
        ptrFoundEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
        ptrFoundEntry += ptrDiskLocation->ucDirectoryEntry;              // move to the present entry
        if (ptrFoundEntry->DIR_Name[0] == 0) {                           // end of root entries found
            unsigned long ulFreeCluster;
            FILE_LOCATION *ptr_newDir;
            if (found_location.directory_location.ulCluster != 0) {      // fill a deleted location which was found during the check of existing directory
                ptr_newDir = &found_location.directory_location;
                if (fnLoadSector(ptr_utDisk, ptr_newDir->ulSector) != UTFAT_SUCCESS) {
                    return UTFAT_DISK_READ_ERROR;
                }
                ptrFoundEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
                ptrFoundEntry += found_location.ucDirectoryEntry;        // move to the present entry
            }
            else {
                ptr_newDir = &ptrDiskLocation->directory_location;
            }
            ulFreeCluster = fnAllocateCluster(ptr_utDisk, ptr_newDir->ulCluster, INITIALISE_DIR_CLUSTER);
            uMemcpy(ptrFoundEntry->DIR_Name, cFileName, 11);             // add the short name
            ptrFoundEntry->DIR_NTRes = cFileName[11];
            fnAddEntry(ptrFoundEntry, ulFreeCluster, DIR_ATTR_DIRECTORY);
            while (utCommitSector(ptr_utDisk, ptr_newDir->ulSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
            ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;
            return (fnCommitInfoChanges(ptr_utDisk));
        }
        else if (ptrFoundEntry->DIR_Name[0] == DIR_NAME_FREE) {          // a free deleted entry found
            if (found_location.directory_location.ulCluster == 0) {
                uMemcpy(&found_location, ptrDiskLocation, sizeof(found_location)); // remember the entry location
            }
        }
        else {                                                           // a valid entry found
            if (ptrFoundEntry->DIR_Attr == DIR_ATTR_DIRECTORY) {         // if it is a directory
                if (!(uMemcmp(ptrFoundEntry->DIR_Name, cFileName, sizeof(ptrFoundEntry->DIR_Name)))) { // check that the directory doesn't already exist
                    return UTFAT_DIRECTORY_EXISTS_ALREADY;               // return an error since it is not allowed to have two directories with the same name
                }
            }
        }
        if ((fnNextDirectoryEntry(ptr_utDisk, ptrDiskLocation) != UTFAT_SUCCESS) && (found_location.directory_location.ulCluster == 0)) { // move to next directory entry
            int iResult;
            iResult = fnDirectorySectorCreate(ptr_utDisk, &ptrDiskLocation->directory_location); // move to next sector/cluster and create additional cluster if necessary
            if (iResult != UTFAT_SUCCESS) {
                return iResult;
            }
        }
    }
    return UTFAT_SUCCESS;
}

static int utReFormat(const unsigned char ucDrive, const CHAR *cVolumeLabel, unsigned char ucFlags) // {33} static and pass flags
{
    if (utDisks[ucDrive].usDiskFlags & WRITE_PROTECTED_SD_CARD) {        // {20}
        return UTFAT_DISK_WRITE_PROTECTED;
    }
    iMemoryOperation = 0;
    utDisks[ucDrive].usDiskFlags &= (HIGH_CAPACITY_SD_CARD);             // remove all flags apart from high capacity information
    utDisks[ucDrive].usDiskFlags |= DISK_UNFORMATTED;                    // consider unformatted from this point on
    #ifdef UTFAT16
    if (ucFlags & UTFAT_FORMAT_16) {                                     // {33}
        utDisks[ucDrive].usDiskFlags |= DISK_FORMAT_FAT16;               // FAT16 to be formatted rather than FAT32
        iMemoryState = SD_STATE_FORMATTING_DISK_2;                       // no extended record added to FAT16
    }
    else {
        iMemoryState = SD_STATE_FORMATTING_DISK_1;                       // start with an extended record when formatting to FAT32
    }
    #else
    iMemoryState = SD_STATE_FORMATTING_DISK_1;
    #endif
    if (ucFlags & UTFAT_FULL_FORMAT) {                                   // {33}
        utDisks[ucDrive].usDiskFlags |= DISK_FORMAT_FULL;                // delete all data content as well as just FAT content
    }
    uMemcpy(utDisks[ucDrive].cVolumeLabel, cVolumeLabel, 11);
    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);                      // schedule the task to start formatting
    return UTFAT_SUCCESS;
}

extern int utFormat(const unsigned char ucDrive, const CHAR *cVolumeLabel, unsigned char ucFlags) // {33}
{
    if ((!(ucFlags & UTFAT_REFORMAT)) && (iMemoryState != SD_NOT_FORMATTED)) {
        return UTFAT_DISK_NOT_READY;
    }
    return utReFormat(ucDrive, cVolumeLabel, ucFlags);
}
#endif


// This routine is called to count the free clusters and update the disk information
//
extern int utFreeClusters(unsigned char ucDisk, UTASK_TASK owner_task)
{
    if (cluster_task != 0) {                                             // cluster count already in progress
        return MISSING_USER_TASK_REFERENCE;                              // the user must pass a task reference
    }
    ulActiveFreeClusterCount = 0;
    ulClusterSectorCheck = (utDisks[ucDisk].utFAT.ulFAT_start + (utDisks[ucDisk].utFAT.ulFatSize - 1));
    iMemoryOperation |= _COUNTING_CLUSTERS;
    cluster_task = owner_task;
    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);                      // start cluster counting operation
    return UTFAT_SUCCESS;
}

// Routine to check for a certain file extension (type of file)          {28}
//
extern int uMatchFileExtension(UTFILEINFO *ptrFileInfo, const CHAR *ptrExtension)
{
    int i = 0;
    CHAR *ptrName = ptrFileInfo->cFileName;
    while (i++ < MAX_UTFAT_FILE_NAME) {
        if (*ptrName++ == '.') {
            int iMatch = uStrEquiv(ptrName, ptrExtension);
            if (iMatch > 0) {
                if ((*(ptrName + iMatch) == 0) && (*(ptrExtension + iMatch) == 0)) {
                    return 0;                                            // successful match
                }
            }
            break;                                                       // match failed
        }
    }
    return -1;                                                           // file type doesn't match
}

// Low level access routines
//
extern int fnReadSector(unsigned char ucDisk, unsigned char *ptrBuffer, unsigned long ulSectorNumber)
{
    if (ptrBuffer == 0) {                                                // {18} if a zero pointer is given force a load but don't copy to a buffer
        return (fnLoadSector(&utDisks[ucDisk], ulSectorNumber));
    }
    return (utReadDiskSector(&utDisks[ucDisk], ulSectorNumber, ptrBuffer));
}

#if defined UTFAT_WRITE                                                  // {40} allow operation without write support
extern int fnWriteSector(unsigned char ucDisk, unsigned char *ptrBuffer, unsigned long ulSectorNumber) // {22}
{
    int iResult;
    UTDISK *ptr_utDisk = &utDisks[ucDisk];
    while ((iResult = utCommitSectorData(ptr_utDisk, ptrBuffer, ulSectorNumber)) == CARD_BUSY_WAIT) {}
    return iResult;
}
#endif

extern const UTDISK *fnGetDiskInfo(unsigned char ucDisk)
{
    return (&utDisks[ucDisk]);
}

#if defined HTTP_ROOT || defined FTP_ROOT
extern int utServer(UTDIRECTORY *ptr_utDirectory, unsigned long ulServerType) // {48}
{
    if (ptr_utDirectory == 0) {                                          // change setting
        usServerStates &= ~(unsigned short)(ulServerType >> 16);         // disable
        usServerResets |= ((unsigned short)ulServerType & ~usServerStates); // the servers that have just been enabled short be reset once
        usServerStates |= (unsigned short)(ulServerType);                // enable
    }
    else {
        if (utDisks[ptr_utDirectory->ucDrive].usDiskFlags & (DISK_NOT_PRESENT | DISK_TYPE_NOT_SUPPORTED | DISK_UNFORMATTED)) { // unusable disk states
            usServerResets = 0;
            ptr_utDirectory->usDirectoryFlags &= (UTDIR_ALLOCATED);
            return UTFAT_DISK_NOT_READY;
        }
        if (!(usServerStates & ulServerType)) {                          // the server type has no access rights so return an error
            return UTFAT_DISK_NOT_READY;
        }
        if (usServerResets & ulServerType) {                             // if a server has been reset its root has to be confirmed
            ptr_utDirectory->usDirectoryFlags &= (UTDIR_ALLOCATED);
            usServerResets &= ~ulServerType;                             // only once
        }
        if (!(ptr_utDirectory->usDirectoryFlags & UTDIR_VALID)) {        // if the disk is not known to be valid try to locate the root directory
            const CHAR *ptrRoot;
            if (ulServerType & UTFAT_HTTP_SERVER) {
    #if defined HTTP_ROOT
                ptrRoot = HTTP_ROOT;
    #endif
            }
            else if (ulServerType & UTFAT_FTP_SERVER) {
    #if defined FTP_ROOT
                ptrRoot = FTP_ROOT;
    #endif
            }
            else {
                return UTFAT_DISK_NOT_READY;                             // unknown server
            }
            if (utOpenDirectory(ptrRoot, ptr_utDirectory) < 0) {         // open the root directory of disk D on connection (if no disk or no directory this fails and the HTTP server falls back to other file systems)
                ptr_utDirectory->usDirectoryFlags &= (UTDIR_ALLOCATED);  // if the directory doesn't exist reset so that it is checked each time (it may be added during operation)
                return UTFAT_DISK_NOT_READY;
            }
            else {
                ptr_utDirectory->usDirectoryFlags |= UTDIR_REFERENCED;   // all accesses are referenced to the root directory
            }
        }
    }
    return UTFAT_SUCCESS;
}
#endif

#if defined SDCARD_DETECT_INPUT_INTERRUPT
// Interrupt call-back on change in SD-card presence
//
static void sdcard_detection_change(void)
{
    fnInterruptMessage(OWN_TASK, E_SDCARD_DETECTION_CHANGE);             // send interrupt event to task so that it can respond accordingly
}

// SD-card detection interrupt configuration
//
static void fnPrepareDetectInterrupt(void)                               // {52}
{
    INTERRUPT_SETUP interrupt_setup;                                     // interrupt configuration parameters
    #if defined _HW_AVR32
    interrupt_setup.int_type = PORT_INTERRUPT;                           // identifier when configuring
    interrupt_setup.int_handler = sdcard_detection_change;               // handling function
    interrupt_setup.int_priority = PRIORITY_GPIO;                        // port interrupt priority
    interrupt_setup.int_port = SDCARD_DETECT_PORT;                       // the port used
    interrupt_setup.int_port_sense = (IRQ_BOTH_EDGES | IRQ_ENABLE_GLITCH_FILER); // interrupt on both edges with active glitch filter
    interrupt_setup.int_port_bits = SDCARD_DETECT_PIN;                   // the input connected
    #endif
    fnConfigureInterrupt((void *)&interrupt_setup);                      // configure test interrupt
}
#endif

#endif
