/************************************************************************
    Mark Butcher    Bsc (Hons) MPhil MIET

    M.J.Butcher Consulting
    Birchstrasse 20f,    CH-5406, Rtihof
    Switzerland

    www.uTasker.com    Skype: M_J_Butcher

    ---------------------------------------------------------------------
    File:      mass_storage.c
    Project:   uTasker project
    ---------------------------------------------------------------------
    Copyright (C) M.J.Butcher Consulting 2004..2014
    *********************************************************************
    11.07.2014 utFATV2.0

*/


/* =================================================================== */
/*                           include files                             */
/* =================================================================== */

#include "config.h"


/* =================================================================== */
/*                          local definitions                          */
/* =================================================================== */


#if !defined UTFAT_DISABLE_DEBUG_OUT                                     // allow all debug messages to be disabled
    #define fnMemoryDebugMsg(x) fnDebugMsg(x)                            // enable debug output
    #define fnMemoryDebugHex(x, y) fnDebugHex((x), (y))                  // enable debug output
    #define fnMemoryDebugDec(x, y) fnDebugDec((x), (y))                  // enable debug output
#else
    #define fnMemoryDebugMsg(x)                                          // disable debug output
    #define fnMemoryDebugHex(x, y)                                       // disable debug output
    #define fnMemoryDebugDec(x, y)                                       // disable debug output
#endif

#if !defined SDCARD_MALLOC
    #define SDCARD_MALLOC(x)             uMalloc((MAX_MALLOC)(x))
#endif

#define OWN_TASK                         TASK_MASS_STORAGE

#if defined SDCARD_SUPPORT

#define T_POWER_STABILISE                (DELAY_LIMIT)(SEC * 0.05)
#define T_NEXT_CHECK                     (DELAY_LIMIT)(SEC * SD_CARD_RETRY_INTERVAL)
#define T_CLEAN_SPARE                    (DELAY_LIMIT)(SEC * 0.10)

// Timer events
//
#define E_POWER_STABILISED               1
#define E_POLL_SD_CARD                   2
#define E_CLEAN_SPARE                    3
#define E_CHECK_CARD_REMOVAL             4                               // used when trying to detect removal by reading register

// Interrpt events
//
#define E_SDCARD_DETECTION_CHANGE        1

#define SD_STATE_STARTING                0
#define SD_STATE_WAIT_CARD               1
#define SD_STATE_STABILISING             2
#define SD_STATE_GO_IDLE                 3
#define SD_STATE_IF_COND                 4
#define SD_STATE_APP_CMD55_CMD41         5
#define SD_STATE_APP_CMD55_CMD41_2       6
#define SD_STATE_OCR                     7
#define SD_STATE_GET_INFO                8
#define SD_STATE_SET_ADDRESS             9
#define SD_STATE_GET_SECTORS             10
#define SD_SET_INTERFACE_PREPARE         11
#define SD_SET_INTERFACE                 12
#define SD_SELECT_CARD                   13
#define SD_SET_BLOCK_LENGTH              14
#define DISK_MOUNTING_1                  15
#define DISK_MOUNTING_2                  16
#define DISK_MOUNTING_3                  17
#define DISK_MOUNTING_4                  18
#define DISK_STATE_READY                 19

#define STATE_FORMATTING_DISK_1          20
#define STATE_FORMATTING_DISK_2          21
#define STATE_FORMATTING_DISK_3          22
#define STATE_FORMATTING_DISK_4          23
#define STATE_FORMATTING_DISK_5          24
#define STATE_FORMATTING_DISK_6          25
#define STATE_FORMATTING_DISK_7          26
#define STATE_FORMATTING_DISK_8          31
#define STATE_FORMATTING_DISK_8A         32
#define STATE_FORMATTING_DISK_9          33
#define STATE_FORMATTING_DISK_10         34
#define STATE_FORMATTING_DISK_11         35
#define STATE_CHECKING_DISK_REMOVAL      36

#define DISK_NOT_FORMATTED               50


#define _IDLE_MEMORY                     0x00
#define _INITIALISED_MEMORY              0x01
#define _READING_MEMORY                  0x02
#define _WRITING_MEMORY                  0x04
#define _FORMATTING_DISK                 0x08
#define _COUNTING_CLUSTERS               0x10

#define ENTRY_VOLUME_ID                 -5
#define ENTRY_DELETED                   -4
#define MATCH_NOT_LFN                   -3
#define END_DIRECTORY_ENTRIES           -2
#define MATCH_FALSE                     -1
#define MATCH_CONTINUE                   0
#define MATCH_SUCCESSFUL                 1
#define DELETED_LFN_MATCH_SUCCESSFUL     2


// File search results
//
#define FULLY_QUALIFIED_LONG_NAME        4
#define FULLY_QUALIFIED_LONG_NAME_SFNM   3
#define LONG_NAME_PARAGRAPH              2
#define FULLY_QUALIFIED_SHORT_NAME       1
#define SHORT_NAME_PARAGRAPH             0
#define INVALID_PARAGRAPH                -1


// FAT32 BPB_ExtFlags flags
//
#define BPB_ExtFlags_0_ZERO_BASED_NUMBER 0x0f                            // only valid if mirroring is disabled
#define BPB_ExtFlags_0_MIRRORED_FAT      0x00
#define BPB_ExtFlags_0_ONE_FAT           0x80                            // mirroring disabled

#define DIR_NAME_FREE                    0xe5

#define NT_FLAG                          0x18

#define NEW_ABSOLUTE_CLUSTER             0x00
#define INITIALISE_DIR_CLUSTER           0x01
#define INITIALISE_DIR_EXTENSION         0x02
#define UPDATE_FAT_END                   0x04
#define UPDATE_FAT_END_IN_DIFF_SECT      0x08
#define NEW_RELATIVE_CLUSTER             (UPDATE_FAT_END | UPDATE_FAT_END_IN_DIFF_SECT)


#define _CHAR_REJECT                     0x01
#define _CHAR_TERMINATE                  0x02
#define _CHAR_CAPITAL                    0x04
#define _CHAR_SMALL                      0x08
#define _CHAR_NUMBER                     0x10
#define _CHAR_REJECT_NON_JAP             0x20
#if defined UTFAT_LFN_WRITE
    #define _CHAR_ACCEPT_LFN             0x40
#else
    #define _CHAR_ACCEPT_LFN             0x00
#endif

    #if defined SD_CONTROLLER_AVAILABLE
        #define SD_CONTROLLER_SHIFT      2
    #else
        #define SD_CONTROLLER_SHIFT      0
    #endif
#endif

#if !defined SFN_ENTRY_CACHE_SIZE
    #define SFN_ENTRY_CACHE_SIZE         1
#endif

#define SIMPLE_DELETE                    0
#define SAFE_DELETE                      1
#define REUSE_CLUSTERS                   2

#define ROOT_DIRECTORY_REFERENCE         0x01
#define ROOT_DIRECTORY_RELOCATE          0x02
#define ROOT_DIRECTORY_SET               0x04
#define ROOT_DIRECTORY_SETTING           0x08

/* =================================================================== */
/*                      local structure definitions                    */
/* =================================================================== */

#if defined SDCARD_SUPPORT
#define DELETED_ENTRY_COUNT 21                                           // 0 is a reference to a single space, 1 to a double deleted hole in the directory objects, 1 to a tripple,.. 20 to a row of 21 deleted objects
typedef struct stOPEN_FILE_BLOCK
{
    const CHAR *ptrLocalDirPath;
    DISK_LOCATION *ptrDiskLocation;
    UTDISK *ptr_utDisk;
    unsigned long ulCluster;
    int iContinue;
    int iRootDirectory;
    int iQualifiedPathType;
    unsigned short usDirFlags;
    CHAR cShortFileName[8 + 3 + 1];                                      // space for 8:3 short file name format plus an NT specific byte

    const CHAR   *ptrFileNameStart;
    CHAR         *ptrFileNameMatch;
    const CHAR   *ptrFileNameEnd;
    DISK_LOCATION present_location;                                      // present location being treated
    #if defined UTFAT_LFN_READ && ((defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE) || defined UTFAT_EXPERT_FUNCTIONS)
    DISK_LOCATION lfn_file_location;                                     // the location where the LFN file name begins
    #endif
    unsigned long ulSFN_found;                                           // count of short file names found in the present directory
    #if defined UTFAT_LFN_WRITE
    DISK_LOCATION DirectoryEndLocation;                                  // location of the end of the present directory (after a complete scan)
    DISK_LOCATION DeleteLocationRef;                                     // temporary sector reference to start of contiguous deleted entries
    DISK_LOCATION DeleteLocation[DELETED_ENTRY_COUNT];                   // location of double, tripple entry deletes found to ease reuse by new LFN entries (21 is maximum possible length of entries in LFN including a final SFN entry)
    #endif
    CHAR          cSFN_entry[SFN_ENTRY_CACHE_SIZE][13];                  // cache of short file names found in a directory
    unsigned char ucSFN_alias_checksum;
    #if defined UTFAT_LFN_WRITE
    unsigned char ucDeleteCount;                                         // temporary counter of contiguous deleted entries
    #endif
    #if defined UTFAT_LFN_READ && ((defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE) || defined UTFAT_EXPERT_FUNCTIONS)
    unsigned char ucLFN_entries;                                         // directory entries occupied by this LFN entry
    #endif
} OPEN_FILE_BLOCK;

#endif

/* =================================================================== */
/*                 local function prototype declarations               */
/* =================================================================== */

#if defined SDCARD_SUPPORT || defined USB_MSD_HOST
    #if defined SD_CONTROLLER_AVAILABLE                                  // routines supplied by HW specific module
        extern void fnInitSDCardInterface(void);
        extern int  fnSendSD_command(const unsigned char ucCommand[6], unsigned char *ucResult, unsigned char *ptrReturnData);
        extern int  fnGetSector(unsigned char *ptrBuf);       
        extern int  fnReadPartialSector(unsigned char *ptrBuf, unsigned short usStart, unsigned short usStop);
        extern int  fnPutSector(unsigned char *ptrBuf, int iMultiBlock);
    #elif defined NAND_FLASH_FAT
        #include "NAND_driver.h"                                         // include NAND driver code
    #else
        static int  fnWaitSD_ready(int iMaxWait);
        static int  fnSendSD_command(const unsigned char ucCommand[6], unsigned char *ucResult, unsigned char *ptrReturnData);
        static int  fnGetSector(unsigned char *ptrBuf);
        static int  fnReadPartialSector(unsigned char *ptrBuf, unsigned short usStart, unsigned short usStop);
    #endif
    static int  utReadSDsector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber);
    extern int  utReadMSDsector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber);
    
    static int  utReadDiskSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf);
    static int  utReadPartialDiskData(UTDISK *ptr_utDisk, unsigned long ulSector, void *ptrBuf, unsigned short usOffset, unsigned short usLength);
    static int  fnLoadSector(UTDISK *ptr_utDisk, unsigned long ulSector);
    #if !defined NAND_FLASH_FAT
        static const unsigned char *fnCreateCommand(unsigned char ucCommand, unsigned long ulValue);
    #endif
    static int  ut_read_disk(UTDISK *ptr_utDisk);
    static void fnCardNotFormatted(int iDisk);
    static void fnInitialisationError(int iDisk, int iNotSupported);
    #if defined UTFAT_WRITE
        static int  utCommitSector(UTDISK *ptr_utDisk, unsigned long ulSector);
        static int  utCommitSectorData(UTDISK *ptr_utDisk, void *ptrBuffer, unsigned long ulSectorNumber);
        static int  utDeleteSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber);
        static unsigned long fnAllocateCluster(UTDISK *ptr_utDisk, unsigned long ulPresentCluster, unsigned char ucClusterType);
        static int  fnDeleteFileContent(UTFILE *ptr_utFile, UTDISK *ptr_utDisk, int iDestroyClusters);
        static int  fnDeleteClusterChain(unsigned long ulClusterStart, unsigned char ucDrive, int iDestroyClusters);
        static void fnAddInfoSect(INFO_SECTOR_FAT32 *ptrInfoSector, unsigned long ulFreeCount, unsigned long ulNextFree);
        static void fnSetTimeDate(DIR_ENTRY_STRUCTURE_FAT32 *ptrEntry, int iCreation);
    #endif
    #if defined UTFAT_LFN_READ && (defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE)
        static int fnDeleteLFN_entry(UTFILE *ptr_utFile);
    #endif
    #if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0
        static int fnFileLocked(UTFILE *ptr_utFile);
    #endif
    #if defined SDCARD_DETECT_INPUT_INTERRUPT 
        static void fnPrepareDetectInterrupt(void);
    #endif
    #if defined UTFAT_FILE_CACHE_POOL && (UTFAT_FILE_CACHE_POOL > 0) && (UTMANAGED_FILE_COUNT > 0)
        static int fnGetManagedFileCache(unsigned long ulSector, unsigned char *ptrBuffer, unsigned short usAccessOffset, unsigned short usAccessLength);
    #endif
#endif

/* =================================================================== */
/*                             constants                               */
/* =================================================================== */

#if defined SDCARD_SUPPORT

#if !defined NAND_FLASH_FAT
    static const unsigned char ucGO_IDLE_STATE_CMD0[6]      = {GO_IDLE_STATE_CMD0, 0x00, 0x00, 0x00, 0x00, CS_GO_IDLE_STATE_CMD0};
    static const unsigned char ucIF_COND_CMD8[6]            = {SEND_IF_COND_CMD8, 0x00, 0x00, VOLTAGE_2_7__3_6, CHECK_PATTERN, CS_SEND_IF_COND_CMD8};
    #if defined SD_CONTROLLER_AVAILABLE                                  // no CRC is set to buffer since the SD controller appends this automatically {5}
        static const unsigned char ucSEND_OP_COND_ACMD_CMD41[5] = {SEND_OP_COND_ACMD_CMD41, HIGH_CAPACITY_SD_CARD_MEMORY, 0xff, 0x80, 0x00};
        static const unsigned char ucSET_REL_ADD_CMD3[5]    = {SET_REL_ADD_CMD3, 0x00, 0x00, 0x00, 0x00};
        static const unsigned char ucSEND_CID_CMD2[5]       = {SEND_CID_CMD2, 0x00, 0x00, 0x00, 0x00};
        static const unsigned char ucSET_BLOCK_LENGTH_CMD16[5]  = {SET_BLOCKLEN_CMD16, 0x00, 0x02, 0x00, 0x02}; // 512 byte block length
        static unsigned char ucSELECT_CARD_CMD7[5]          = {SELECT_CARD_CMD7, 0x00, 0x00, 0x00, 0x00};
        static unsigned char ucSEND_CSD_CMD9[5]             = {SEND_CSD_CMD9, 0x00, 0x00, 0x00, 0x00};
        static unsigned char ucAPP_CMD_CMD55[5]             = {APP_CMD_CMD55, 0x00, 0x00, 0x00, 0x00};
        static unsigned char ucSET_BUS_WIDTH_CMD6[5]        = {SET_BUS_WIDTH_CMD6, 0x00, 0x00, 0x00, 0x02};
    #else
        static const unsigned char ucSEND_OP_COND_ACMD_CMD41[6] = {SEND_OP_COND_ACMD_CMD41, HIGH_CAPACITY_SD_CARD_MEMORY, 0x00, 0x00, 0x00, CS_SEND_OP_COND_ACMD_CMD41};
        static const unsigned char ucSEND_CSD_CMD9[6]       = {SEND_CSD_CMD9, 0x00, 0x00, 0x00, 0x00, CS_SEND_CSD_CMD9};
        static const unsigned char ucAPP_CMD_CMD55[6]       = {APP_CMD_CMD55, 0x00, 0x00, 0x00, 0x00, CS_APP_CMD_CMD55};
        static const unsigned char ucREAD_OCR_CMD58[6]      = {READ_OCR_CMD58, 0x00, 0x00, 0x00, 0x00, CS_READ_OCR_CMD58};
    #endif
#endif

#if defined UTFAT_WRITE && defined UTFAT_FORMATTING
static const unsigned char ucEmptyFAT32[12] = {
    LITTLE_LONG_WORD_BYTES(MEDIA_VALUE_FIXED),
    LITTLE_LONG_WORD_BYTES(0xffffffff),
    LITTLE_LONG_WORD_BYTES(CLUSTER_MASK)
};

    #if defined UTFAT16
static const unsigned char ucEmptyFAT16[4] = {
    LITTLE_LONG_WORD_BYTES(0xfffffff8)
};
    #endif
#endif

static const unsigned char ucCharacterTable[] = {
    (0),                                                                 // !
    (_CHAR_REJECT),                                                      // "
    (0),                                                                 // #
    (0),                                                                 // $
    (0),                                                                 // %
    (0),                                                                 // &
    (0),                                                                 // 
    (0),                                                                 // (
    (0),                                                                 // )
    (_CHAR_REJECT),                                                      // *
    (_CHAR_REJECT),                                                      // +
    (0),                                                                 // ,
    (0),                                                                 // -
    (0),                                                                 // .
    (_CHAR_TERMINATE),                                                   // /
    (_CHAR_NUMBER),                                                      // 0
    (_CHAR_NUMBER),                                                      // 1
    (_CHAR_NUMBER),                                                      // 2
    (_CHAR_NUMBER),                                                      // 3
    (_CHAR_NUMBER),                                                      // 4
    (_CHAR_NUMBER),                                                      // 5
    (_CHAR_NUMBER),                                                      // 6
    (_CHAR_NUMBER),                                                      // 7
    (_CHAR_NUMBER),                                                      // 8
    (_CHAR_NUMBER),                                                      // 9
    (_CHAR_REJECT),                                                      // :
    (_CHAR_REJECT),                                                      // ;
    (_CHAR_REJECT),                                                      // <
    (_CHAR_REJECT),                                                      // =
    (_CHAR_REJECT),                                                      // >
    (0),                                                                 // ?
    (0),                                                                 // @
    (_CHAR_CAPITAL),                                                     // A
    (_CHAR_CAPITAL),                                                     // B
    (_CHAR_CAPITAL),                                                     // C
    (_CHAR_CAPITAL),                                                     // D
    (_CHAR_CAPITAL),                                                     // E
    (_CHAR_CAPITAL),                                                     // F
    (_CHAR_CAPITAL),                                                     // G
    (_CHAR_CAPITAL),                                                     // H
    (_CHAR_CAPITAL),                                                     // I
    (_CHAR_CAPITAL),                                                     // J
    (_CHAR_CAPITAL),                                                     // K
    (_CHAR_CAPITAL),                                                     // L
    (_CHAR_CAPITAL),                                                     // M
    (_CHAR_CAPITAL),                                                     // N
    (_CHAR_CAPITAL),                                                     // O
    (_CHAR_CAPITAL),                                                     // P
    (_CHAR_CAPITAL),                                                     // Q
    (_CHAR_CAPITAL),                                                     // R
    (_CHAR_CAPITAL),                                                     // S
    (_CHAR_CAPITAL),                                                     // T
    (_CHAR_CAPITAL),                                                     // U
    (_CHAR_CAPITAL),                                                     // V
    (_CHAR_CAPITAL),                                                     // W
    (_CHAR_CAPITAL),                                                     // X
    (_CHAR_CAPITAL),                                                     // Y
    (_CHAR_CAPITAL),                                                     // Z
    (_CHAR_REJECT_NON_JAP),                                              // [
    (_CHAR_TERMINATE),                                                   // back slash
    (_CHAR_REJECT_NON_JAP),                                              // ]
    (0),                                                                 // ^
    (0),                                                                 // _
    (0),                                                                 // '
    (_CHAR_SMALL),                                                       // a
    (_CHAR_SMALL),                                                       // b
    (_CHAR_SMALL),                                                       // c
    (_CHAR_SMALL),                                                       // d
    (_CHAR_SMALL),                                                       // e
    (_CHAR_SMALL),                                                       // f
    (_CHAR_SMALL),                                                       // g
    (_CHAR_SMALL),                                                       // h
    (_CHAR_SMALL),                                                       // i
    (_CHAR_SMALL),                                                       // j
    (_CHAR_SMALL),                                                       // k
    (_CHAR_SMALL),                                                       // l
    (_CHAR_SMALL),                                                       // m
    (_CHAR_SMALL),                                                       // n
    (_CHAR_SMALL),                                                       // o
    (_CHAR_SMALL),                                                       // p
    (_CHAR_SMALL),                                                       // q
    (_CHAR_SMALL),                                                       // r
    (_CHAR_SMALL),                                                       // s
    (_CHAR_SMALL),                                                       // t
    (_CHAR_SMALL),                                                       // u
    (_CHAR_SMALL),                                                       // v
    (_CHAR_SMALL),                                                       // w
    (_CHAR_SMALL),                                                       // x
    (_CHAR_SMALL),                                                       // y
    (_CHAR_SMALL),                                                       // z
    (0),                                                                 // {
    (_CHAR_REJECT_NON_JAP),                                              // |
    (0),                                                                 // }
    (0),                                                                 // ~
};
#endif

/* =================================================================== */
/*                     global variable definitions                     */
/* =================================================================== */


/* =================================================================== */
/*                      local variable definitions                     */
/* =================================================================== */

#if defined SDCARD_SUPPORT || defined USB_MSD_HOST
static UTDISK utDisks[DISK_COUNT] = {{0}};                               // disk D is SD-card / disk E is USB-MSD host

#if !defined SD_CONTROLLER_AVAILABLE && !defined NAND_FLASH_FAT
    #if defined _WINDOWS
        static int CommandTimeout = 10;
    #else
        static int CommandTimeout = 20000;                               // change depending on SPI speed
    #endif
#endif

static int iMemoryOperation[DISK_COUNT] = {0};
static int iMemoryState[DISK_COUNT] = {0};

#if defined UTFAT_MULTIPLE_BLOCK_WRITE
    static unsigned long ulBlockWriteLength = 0;                         // multiple block writing to speed up write operations
    static unsigned long ulMultiBlockAddress = 0;
#endif

static unsigned long  ulClusterSectorCheck[DISK_COUNT] = {0};
static unsigned long  ulActiveFreeClusterCount[DISK_COUNT] = {0};
static UTASK_TASK     cluster_task[DISK_COUNT] = {0};
#if defined HTTP_ROOT || defined FTP_ROOT
    static unsigned short usServerStates = 0;
    static unsigned short usServerResets = 0;
#endif

#if defined UTFAT_WRITE && (defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0)
    #define FILE_BUFFER_FREE      0x00
    #define FILE_BUFFER_IN_USE    0x01
    #define FILE_BUFFER_VALID     0x02
    #define FILE_BUFFER_MODIFIED  0x04
    static FILE_DATA_CACHE FileDataCache[UTFAT_FILE_CACHE_POOL] = {{0}}; // initially all area free to be allocated
#endif

#if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0
typedef struct stUTMANAGED_FILE
{
    unsigned char   managed_mode;
    unsigned char   managed_owner;
    UTFILE         *utManagedFile;
}   UTMANAGED_FILE;

static UTMANAGED_FILE utManagedFiles[UTMANAGED_FILE_COUNT] = {{0}};
#endif
#endif                                                                   // end #if defined SDCARD_SUPPORT || defined USB_MSD_HOST


#if defined MANAGED_FILES
    #if defined SUB_FILE_SIZE
        #define SUBFILE_WRITE  , ucSubFileInProgress
        #define SUB_FILE_ON    ,SUB_FILE_TYPE
    #else
        #define SUBFILE_WRITE
        #define SUB_FILE_ON
    #endif

static int iManagedMedia = 0;

static MANAGED_FILE managed_files[MANAGED_FILE_COUNT] = {{0}};

extern int uFileManagedDelete(int fileHandle)
{
	int iEraseStatus = 0;
    if (managed_files[fileHandle].managed_size != 0) {
        iEraseStatus = fnEraseFlashSector(managed_files[fileHandle].managed_write, 0); // start single page erase
    }
    #if defined TIME_SLICED_FILE_OPERATION
    else {                                                               // final sector delete has already started
        managed_files[fileHandle].period = 0;
    }
    #endif
	if (iEraseStatus != MEDIA_BUSY) {
        if (managed_files[fileHandle].managed_size == 0) {               // delete has completed (after one polly)
			managed_files[fileHandle].managed_mode = 0;
          //managed_files[fileHandle].managed_mode &= ~WAITING_DELETE;
			if (managed_files[fileHandle].ucParameters & AUTO_CLOSE) {
				managed_files[fileHandle].managed_owner = 0;
			}
			if (managed_files[fileHandle].fileOperationCallback) {
				managed_files[fileHandle].fileOperationCallback(fileHandle, 0);
			}
			return 0;                                                    // complete
		}
		else {
            managed_files[fileHandle].managed_write += iEraseStatus;     // erase of block started so increment the erase pointer
            if ((unsigned int)iEraseStatus >= managed_files[fileHandle].managed_size) {
                managed_files[fileHandle].managed_size = 0;              // final sector delete started
            }
            else {
			    managed_files[fileHandle].managed_size -= iEraseStatus;
            }
		}
    #if defined TIME_SLICED_FILE_OPERATION
		if (managed_files[fileHandle].period != 0) {                     // if a page delete period rate is defined wait until this expires before continuing
			uTaskerGlobalMonoTimer(OWN_TASK, managed_files[fileHandle].period, (unsigned char)(fileHandle | _DELAYED_DELETE));
            return 1;
		}
    #endif
	}
	managed_files[fileHandle].managed_mode |= WAITING_DELETE;            // either waiting to delete a busy page or waiting for a page to complete deletion
    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);                      // set to run again
    iManagedMedia = 1;                                                   // mark that this media type needs monitoring
    return 1;                                                            // delete not yet complete
}

    #if defined MANAGED_FILE_WRITE
extern int uFileManagedWrite(int fileHandle)
{
    int iWriteStatus;
    MAX_FILE_LENGTH write_chunk_length = managed_files[fileHandle].managed_chunk_size; // the maximum write size that is specified
    if ((write_chunk_length == 0) || (write_chunk_length > managed_files[fileHandle].managed_size)) {
        write_chunk_length = managed_files[fileHandle].managed_size;     // maximum remaining size possible
    }
	iWriteStatus = fnWriteBytesFlashNonBlocking(managed_files[fileHandle].managed_write, managed_files[fileHandle].managed_buffer, write_chunk_length); // start a single block write
    if (iWriteStatus != MEDIA_BUSY) {                                    // if the storage media was not busy the write has been started
        if (managed_files[fileHandle].managed_size <= write_chunk_length) { // final write completed
            if (managed_files[fileHandle].managed_size == 0) {           // we never terminate immediately after the final write has started but always perform a dummy poll to avoid recursive calls (when writes always terminate immediately)
                if (managed_files[fileHandle].ucParameters & AUTO_CLOSE) {
                    managed_files[fileHandle].managed_owner = 0;         // automatically close on completion
                }
                if (managed_files[fileHandle].fileOperationCallback) {   // if there is a user callback defined
                    managed_files[fileHandle].fileOperationCallback(fileHandle, 0);
                }
                return 0;                                                // complete
            }
        #if defined TIME_SLICED_FILE_OPERATION
            else {
                managed_files[fileHandle].period = 0;
            }
        #endif
		}
        else {
            managed_files[fileHandle].managed_write += write_chunk_length; // write of block started so increment the write pointer
            managed_files[fileHandle].managed_buffer += write_chunk_length; // and increment the buffer pointer
        }
        managed_files[fileHandle].managed_size -= write_chunk_length;
        #if defined TIME_SLICED_FILE_OPERATION
		if (managed_files[fileHandle].period != 0) {                     // if a block write period rate is defined wait until this expires before continuing
			uTaskerGlobalMonoTimer(OWN_TASK, managed_files[fileHandle].period, (unsigned char)(fileHandle | _DELAYED_WRITE));
            return 1;
		}
        #endif
	}
	managed_files[fileHandle].managed_mode |= WAITING_WRITE;             // waiting to write a busy page
    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);                      // set to run again
    iManagedMedia = 1;                                                   // mark that this media type needs monitoring
    return 1;                                                            // write not yet complete
}
    #endif

    #if defined MANAGED_FILE_READ
extern int uFileManagedRead(int fileHandle)
{
    int iReadStatus;
    MAX_FILE_LENGTH read_chunk_length = managed_files[fileHandle].managed_chunk_size; // the maximum read size that is specified
    if (read_chunk_length > managed_files[fileHandle].managed_size) {
        read_chunk_length = managed_files[fileHandle].managed_size;      // remaining size
    }
	iReadStatus = fnReadBytesFlashNonBlocking(managed_files[fileHandle].managed_write, managed_files[fileHandle].managed_buffer, read_chunk_length); // start a single block read
    if (iReadStatus != MEDIA_BUSY) {                                     // if the storage media was not busy the read has completed
        if (managed_files[fileHandle].managed_size <= read_chunk_length) { // final read completed
            if (managed_files[fileHandle].managed_size == 0) {           // we never terminate immediately after the final read has started but always perform a dummy poll to avoid recursive calls (when writes always terminate immediately)
                if (managed_files[fileHandle].ucParameters & AUTO_CLOSE) {
                    managed_files[fileHandle].managed_owner = 0;         // automatically close on completion
                }
                if (managed_files[fileHandle].fileOperationCallback) {   // if there is a user callback defined
                    managed_files[fileHandle].fileOperationCallback(fileHandle, 0);
                }
                return 0;                                                // complete
            }
        #if defined TIME_SLICED_FILE_OPERATION
            else {
                managed_files[fileHandle].period = 0;
            }
        #endif
		}
        else {
            managed_files[fileHandle].managed_write += read_chunk_length; // read from block completed so increment the read pointer
            managed_files[fileHandle].managed_buffer += read_chunk_length; // and increment the buffer pointer
        }
        managed_files[fileHandle].managed_size -= read_chunk_length;
        #if defined TIME_SLICED_FILE_OPERATION
		if (managed_files[fileHandle].period != 0) {                     // if a block write period rate is defined wait until this expires before continuing
			uTaskerGlobalMonoTimer(OWN_TASK, managed_files[fileHandle].period, (unsigned char)(fileHandle | _DELAYED_READ));
            return 1;
		}
        #endif
	}
	managed_files[fileHandle].managed_mode |= WAITING_READ;              // waiting to read a busy page
    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);                      // set to run again
    iManagedMedia = 1;                                                   // mark that this media type needs monitoring
    return 1;                                                            // write not yet complete
}
    #endif

extern int uOpenManagedFile(void *ptrFileName, UTASK_TASK owner_task, unsigned char ucMode)
{
    int i = 0;
    int iFree = 0;
    unsigned char *ptrFile;
	MAX_FILE_LENGTH file_length;
	if (ucMode & MANAGED_MEMORY_AREA) {
		MANAGED_MEMORY_AREA_BLOCK *ptrMemoryArea = (MANAGED_MEMORY_AREA_BLOCK *)ptrFileName;
		ptrFile = ptrMemoryArea->ptrStart;
		file_length = ptrMemoryArea->size;                               // size of the area
	}
	else {
        ptrFile = uOpenFile((CHAR *)ptrFileName);                        // get the file to be opened from the uFileSystem
		file_length = uGetFileLength(ptrFile);                           // file length of existing file
	}
    while (i < MANAGED_FILE_COUNT) {                                     // check whether this file is presently being protected
        if (managed_files[i].managed_owner != 0) {
            if (managed_files[i].managed_mode & (MANAGED_WRITE | MANAGED_LOCK | MANAGED_DELETE)) {
                if ((ptrFile >= managed_files[i].managed_start) && (ptrFile < (managed_files[i].managed_start + managed_files[i].managed_size))) {
                    return MANAGED_FILE_NO_ACCESS;                       // this file can presently not be accessed
                }
            }
        }
        else {
            iFree = (i + 1);                                             // free space found
        }
        i++;
    }
    if (iFree-- == 0) {
        return MANAGED_FILE_NO_FILE_HANDLE;                              // all managed file spaces are presently occupied
    }
    managed_files[iFree].managed_owner = owner_task;
    managed_files[iFree].managed_write = managed_files[iFree].managed_start = ptrFile;
    managed_files[iFree].managed_mode = ucMode;
    managed_files[iFree].managed_size = file_length;                     // file length of existing file
    if (ucMode & MANAGED_MEMORY_AREA) {
		MANAGED_MEMORY_AREA_BLOCK *ptrMemoryArea = (MANAGED_MEMORY_AREA_BLOCK *)ptrFileName;
		managed_files[iFree].ucParameters = ptrMemoryArea->ucParameters;
		managed_files[iFree].fileOperationCallback = ptrMemoryArea->fileOperationCallback;
		managed_files[iFree].period = ptrMemoryArea->period;
    #if defined MANAGED_FILE_WRITE || defined MANAGED_FILE_READ          // parameters only used by writes
        managed_files[iFree].managed_buffer = ptrMemoryArea->ptrBuffer;
        managed_files[iFree].managed_chunk_size = ptrMemoryArea->chunk_size;
    #endif
		if (ptrMemoryArea->ucParameters & AUTO_DELETE) {                 // if a delete is to be performed immediately call the function
			uFileManagedDelete(iFree);                                   // start deletion
		}
    #if defined MANAGED_FILE_WRITE
        else if (ptrMemoryArea->ucParameters & AUTO_WRITE) {             // if a write is to be performed immediately call the function
			uFileManagedWrite(iFree);                                    // start write
        }
    #endif
    #if defined MANAGED_FILE_READ
        else if (ptrMemoryArea->ucParameters & AUTO_READ) {              // if a write is to be performed immediately call the function
			uFileManagedRead(iFree);                                     // start read
        }
    #endif
	}
    return iFree;                                                        // return file handle
}

static void fnManagedMediaCheck(void)
{
    int iManagedFileEntry = MANAGED_FILE_COUNT;
    while (iManagedFileEntry > 0) {                                      // check all managed files
        iManagedFileEntry--;
        if (managed_files[iManagedFileEntry].managed_owner != 0) {       // for each one that is owned
            if (managed_files[iManagedFileEntry].managed_mode & WAITING_DELETE) { // waiting to start or continue delete operation
                uFileManagedDelete(iManagedFileEntry);
			}
    #if defined MANAGED_FILE_WRITE
            else if (managed_files[iManagedFileEntry].managed_mode & WAITING_WRITE) { // waiting to start or continue write operation
                uFileManagedWrite(iManagedFileEntry);
            }
    #endif
    #if defined MANAGED_FILE_READ
            else if (managed_files[iManagedFileEntry].managed_mode & WAITING_READ) { // waiting to start or continue write operation
                uFileManagedRead(iManagedFileEntry);
            }
    #endif
        }
    }
}
#endif



#if defined SDCARD_SUPPORT || defined USB_MSD_HOST
static int (*_utReadDisk[DISK_COUNT])(UTDISK *ptr_utDisk, unsigned long ulSectorNumber) = {
    utReadSDsector,                                                      // SD-card function to read a sector
    #if DISK_COUNT > 1
    utReadMSDsector,                                                     // USB-MSD host function to read a sector
    #endif
};
#endif

#if defined SDCARD_SUPPORT || defined USB_MSD_HOST || defined MANAGED_FILES
// Mass storage task
//
extern void fnMassStorage(TTASKTABLE *ptrTaskTable)
{
    #if defined _WINDOWS
    static int iFormatCount;
    #endif
    #if DISK_COUNT > 1
        int iDiskNumber = -1;
        #define _return continue
    #else
        #define iDiskNumber DISK_D
        #define _return return
    #endif
    unsigned char ucInputMessage[HEADER_LENGTH];                         // reserve space for receiving messages
    #if defined SDCARD_SUPPORT || defined USB_MSD_HOST
    int iActionResult = 0;
        #if !defined NAND_FLASH_FAT
    unsigned char ucData[18];
    unsigned char ucResult = 0;
        #endif
    #endif
    #if defined MANAGED_FILES
    if (iManagedMedia != 0) {
        iManagedMedia = 0;
        fnManagedMediaCheck();
    }
    #endif
    #if DISK_COUNT > 1
    while (++iDiskNumber < DISK_COUNT) {                                 // for each disk
    #endif
    #if defined SDCARD_SUPPORT || defined USB_MSD_HOST
    if (iMemoryOperation[iDiskNumber] & _READING_MEMORY) {               // reading
        if ((iActionResult = _utReadDisk[iDiskNumber](&utDisks[iDiskNumber], 0)) == CARD_BUSY_WAIT) {
            return;                                                      // still reading so keep waiting
        }
    }
    if (!(iMemoryOperation[iDiskNumber] & _INITIALISED_MEMORY)) {        // if initialisation in progress
        #if defined UTFAT_WRITE && defined UTFAT_FORMATTING
        static unsigned long ulFatSectors;
        static unsigned long ulFAT32size;
        static unsigned char ucFatCopies;
        #endif
        switch (iMemoryState[iDiskNumber]) {                             // perform the initialisation state-event process
        case SD_STATE_STARTING:
        case SD_STATE_WAIT_CARD:
        #if DISK_COUNT > 1
            if (iDiskNumber > DISK_D) {
                continue;                                                // non SD-card not presently operational
            }
        #endif
        #if defined NAND_FLASH_FAT
            if (utDisks[iDiskNumber].ptrSectorData == 0) {
                fnInitNAND();                                            // initialise the NAND interface
                utDisks[iDiskNumber].ptrSectorData = SDCARD_MALLOC(512); // allocate a buffer which will contain data read from the present sector (this should be long word aligned and possibly in a specific mememroy are in case DMA operation is to be used)
                utDisks[iDiskNumber].ulSD_sectors = (USER_AREA_BLOCKS * NAND_PAGES_IN_BLOCK);
                uTaskerStateChange(OWN_TASK, UTASKER_GO);                // switch to polling mode of operation
            }
            if (fnGetBlankBlocks() != 0) {                               // start the process of reading all blocks to get their present state
                _return;                                                 // repeat until the complete NAND flash has been checked for blank blocks
            }
            uTaskerStateChange(OWN_TASK, UTASKER_STOP);
        #elif defined SDCARD_SUPPORT                                     // initialisation only required by SD card
            #if defined SD_CONTROLLER_AVAILABLE
            fnInitSDCardInterface();                                     // HW interface initialisation
            ucSEND_CSD_CMD9[1] = ucSELECT_CARD_CMD7[1] = ucAPP_CMD_CMD55[1] = ucSET_BUS_WIDTH_CMD6[1] = 0; // start with zeroed RCA address
            ucSEND_CSD_CMD9[2] = ucSELECT_CARD_CMD7[2] = ucAPP_CMD_CMD55[2] = ucSET_BUS_WIDTH_CMD6[2] = 0;
            iMemoryState[DISK_D] = SD_STATE_STABILISING;                 // move to stabilisation delay state
            uTaskerMonoTimer(OWN_TASK, T_POWER_STABILISE, E_POWER_STABILISED); // wait until SD card power stabilised
            #else
            INITIALISE_SPI_SD_INTERFACE();                               // initialise the SPI interface to the card
                #if defined SDCARD_DETECT_INPUT_POLL || defined SDCARD_DETECT_INPUT_INTERRUPT
                    #if defined SDCARD_DETECT_INPUT_INTERRUPT 
            fnPrepareDetectInterrupt();                                  // prepare interrupt detection of SD card presence
                    #endif
            if (!(SDCARD_DETECTION())) {                                 // if card is not detected immediately abort mounting process
                fnInitialisationError(DISK_D, 0);                        // try to remount the card
                break;
            }
                    #if defined _WINDOWS
            else {
                SD_card_state(SDCARD_INSERTED, SDCARD_REMOVED);
            }
                    #endif
                #endif
            SET_SD_DI_CS_HIGH();                                         // prepare chip select and DI ready for the initialisation
            POWER_UP_SD_CARD();                                          // power up if necessary
            iMemoryState[DISK_D] = SD_STATE_STABILISING;                 // move to stabilisation delay state
            ENABLE_SPI_SD_OPERATION();                                   // dummy to solve an AVR32 GCC optimising problem
            uTaskerMonoTimer(OWN_TASK, T_POWER_STABILISE, E_POWER_STABILISED); // wait until SD card power stabilised
            #endif
            break;

        case SD_STATE_STABILISING:                                       // delay after applying power to the SD-card
            #if defined SD_CONTROLLER_AVAILABLE
            POWER_UP_SD_CARD();                                          // apply power
                #if defined SDCARD_DETECT_INPUT_POLL || defined SDCARD_DETECT_INPUT_INTERRUPT
                    #if defined SDCARD_DETECT_INPUT_INTERRUPT 
            fnPrepareDetectInterrupt();                                  // prepare interrupt detection of SD card presence
                    #endif
            if (!(SDCARD_DETECTION())) {                                 // if card is not detected immediately abort mounting process
                fnInitialisationError(DISK_D, 0);                        // try to remount the card
                break;
            }
                    #if defined _WINDOWS
            else {
                SD_card_state(SDCARD_INSERTED, SDCARD_REMOVED);
            }
                    #endif
                #endif
            iMemoryState[DISK_D] = SD_STATE_GO_IDLE;                     // prepare to communicate
            uTaskerMonoTimer(OWN_TASK, T_POWER_STABILISE, E_POWER_STABILISED); // wait until SD card power stabilised
            break;
            #else
            {                                                            // send at least 74 clocks to the SD-card
                int i = 10;
                while (i--) {                                            // set the SD card to native command mode by sending 80 clocks
                    WRITE_SPI_CMD(0xff);                                 // write dummy tx
                    WAIT_TRANSMISSON_END();                              // wait until transmission complete
                    READ_SPI_DATA();                                     // read 10 dummy bytes from the interface in order to generate 80 clock pulses on the interface (at least 74 needed)
                }
                SET_SD_CS_LOW();                                         // assert the chip select line to the SD-card ready to start the initialisation sequence
                iMemoryState[DISK_D] = SD_STATE_GO_IDLE;                 // move to next state
                SET_SD_CARD_MODE();                                      // allow final mode in case the DIN line had to be pulled up during native mode sequence
            }                                                            // fall through
            #endif
        case SD_STATE_GO_IDLE:
            if ((iActionResult = fnSendSD_command(ucGO_IDLE_STATE_CMD0, &ucResult, 0)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                ucResult = 0;                                            // set error since the result is not as expected
            }
            if (R1_IN_IDLE_STATE != ucResult) {                          // no valid card detected so disable power and try again after a delay
                fnInitialisationError(DISK_D, 0);                        // the card is no present or is behaving incorrectly - stop and try again later
                break;
            }                                                            // SD card must return the idle state to be able to continue
                                                                         // in the idle state the card accepts only commands 0, 1, ACMD41 and 58
            iMemoryState[DISK_D] = SD_STATE_IF_COND;                     // fall through
        case SD_STATE_IF_COND:                                           // it is mandatory for a host compliant to Physical Spec. Version 2.00 to send the CMD8 to retrieve non-supported voltage range
            if ((iActionResult = fnSendSD_command(ucIF_COND_CMD8, &ucResult, ucData)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            if (ucResult == SDC_CARD_VERSION_2PLUS) {                    // version 2 or higher
                if (ucData[3] == CHECK_PATTERN) {
                    if (ucData[2] == VOLTAGE_2_7__3_6) {                 // check whether the card can operate between 2.7V and 3.6V
                        iMemoryState[DISK_D] = SD_STATE_APP_CMD55_CMD41; // now poll the SD card until it accept the application command 42
                        uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);  // run again
                        break;
                    }
                    fnMemoryDebugMsg("SD-card voltage error\r\n");
                }
                else {
                    fnInitialisationError(DISK_D, 0);                    // no valid response
                    break;
                }
            }
            else {                                                       // version 1 or MMC type
                fnMemoryDebugMsg("SD-card V1 or MMC - not supported!\r\n");
            }
            fnInitialisationError(DISK_D, 1);                            // not supported
            break;
        case SD_STATE_APP_CMD55_CMD41:
            if ((iActionResult = fnSendSD_command(ucAPP_CMD_CMD55, &ucResult, 0)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            if (ucResult > R1_IN_IDLE_STATE) {
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // try again
                break;
            }
            iMemoryState[DISK_D] = SD_STATE_APP_CMD55_CMD41_2;           // fall through to send the application command
        case SD_STATE_APP_CMD55_CMD41_2:            
            #if defined SD_CONTROLLER_AVAILABLE                          // this command returns OCR result in SD card mode
            if ((iActionResult = fnSendSD_command(ucSEND_OP_COND_ACMD_CMD41, &ucResult, ucData)) != UTFAT_SUCCESS)
            #else
            if ((iActionResult = fnSendSD_command(ucSEND_OP_COND_ACMD_CMD41, &ucResult, 0)) != UTFAT_SUCCESS) 
            #endif
            {
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            #if defined SD_CONTROLLER_AVAILABLE
            if (!(ucData[0] & 0x80))                                     // check card busy bit
            #else
            if (ucResult != 0) 
            #endif
            {
                iMemoryState[DISK_D] = SD_STATE_APP_CMD55_CMD41;         // loop back
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // try again
                break;
            }
            iMemoryState[DISK_D] = SD_STATE_OCR;                         // fall through to issue CMD58
        case SD_STATE_OCR:
            #if !defined SD_CONTROLLER_AVAILABLE                         // OCR value is returned to the CMD41 on SD card interface {5}
            if ((iActionResult = fnSendSD_command(ucREAD_OCR_CMD58, &ucResult, ucData)) != UTFAT_SUCCESS) { // get card capacity information
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            #endif
            fnMemoryDebugMsg("SD-card V2 - ");
            if (ucData[0] & HIGH_CAPACITY_SD_CARD_MEMORY) {              // check the CCS bit
                utDisks[DISK_D].usDiskFlags = HIGH_CAPACITY_SD_CARD;
                fnMemoryDebugMsg("High Capacity\r\n");
            }
            else {
                utDisks[DISK_D].usDiskFlags = 0;
                fnMemoryDebugMsg("standard\r\n");
            }
            #if defined SD_CONTROLLER_AVAILABLE                          // SC card mode sequency requires the CID to be read and then an RCA address to be set
            iMemoryState[DISK_D] = SD_STATE_GET_INFO;                    // fall through to issue CMD2 and read card information
        case SD_STATE_GET_INFO:
            if ((iActionResult = fnSendSD_command(ucSEND_CID_CMD2, &ucResult, ucData)) != UTFAT_SUCCESS) { // get card information
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            iMemoryState[DISK_D] = SD_STATE_SET_ADDRESS;
        case SD_STATE_SET_ADDRESS:
            if ((iActionResult = fnSendSD_command(ucSET_REL_ADD_CMD3, &ucResult, ucData)) != UTFAT_SUCCESS) { // set relative address
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            if ((ucData[2] & (CURRENT_CARD_STATUS_MASK | SD_CARD_READY_FOR_DATA)) == (CURRENT_STATE_IDENT | SD_CARD_READY_FOR_DATA)) {
                ucSEND_CSD_CMD9[1] = ucSELECT_CARD_CMD7[1] = ucAPP_CMD_CMD55[1] = ucSET_BUS_WIDTH_CMD6[1] = ucData[0];  // save the published RCA address
                ucSEND_CSD_CMD9[2] = ucSELECT_CARD_CMD7[2] = ucAPP_CMD_CMD55[2] = ucSET_BUS_WIDTH_CMD6[2] = ucData[1];
            }
            #endif
            iMemoryState[DISK_D] = SD_STATE_GET_SECTORS;                 // fall through to issue CMD9 and read card specific data
        case SD_STATE_GET_SECTORS:
            if ((iActionResult = fnSendSD_command(ucSEND_CSD_CMD9, &ucResult, ucData)) != UTFAT_SUCCESS) { // get card capacity information
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            uMemcpy(utDisks[DISK_D].utFileInfo.ucCardSpecificData, &ucData[2 - SD_CONTROLLER_SHIFT], 16); // back up the card specific data since it can be of interest later
            if (ucData[2 - SD_CONTROLLER_SHIFT] & HIGH_CAPACITY_SD_CARD_MEMORY) { // high capacity
                utDisks[DISK_D].ulSD_sectors = (((ucData[9 - SD_CONTROLLER_SHIFT] << 16) + (ucData[10 - SD_CONTROLLER_SHIFT] << 8)  + ucData[11 - SD_CONTROLLER_SHIFT]) + 1);// SD version 2 assumed
                utDisks[DISK_D].ulSD_sectors *= 1024;                    // the number of sectors on the SD card
            }
            else {                                                       // standard capacity
                utDisks[DISK_D].ulSD_sectors = ((((ucData[8 - SD_CONTROLLER_SHIFT] & 0x03) << 10) + (ucData[9 - SD_CONTROLLER_SHIFT] << 2) + (ucData[10 - SD_CONTROLLER_SHIFT] >> 6)) + 1);// SD version 2 assumed
                utDisks[DISK_D].ulSD_sectors *= (1 << ((((ucData[11 - SD_CONTROLLER_SHIFT] & 0x03) << 1) + (ucData[12 - SD_CONTROLLER_SHIFT] >> 7)) + 2)); // the number of 512 byte sectors on the SD card
                if ((ucData[7 - SD_CONTROLLER_SHIFT] & 0x0f) == 0x0a) {  // 1024 byte block length indicates 2G card
                    utDisks[DISK_D].ulSD_sectors *= 2;
                }
            }
            fnMemoryDebugMsg("\r\n");
            if (utDisks[DISK_D].ptrSectorData == 0) {
                utDisks[DISK_D].ptrSectorData = SDCARD_MALLOC(512);      // allocate a buffer which will contain data read from the present sector (this should be long word aligned and possibly in a specific memory in case DMA operation is to be used)
            }
            #if defined SD_CONTROLLER_AVAILABLE
            iMemoryState[DISK_D] = SD_SELECT_CARD;
        case SD_SELECT_CARD:                                             // select the card before reading/writing
            if ((iActionResult = fnSendSD_command(ucSELECT_CARD_CMD7, &ucResult, ucData)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            iMemoryState[DISK_D] = SD_SET_INTERFACE_PREPARE;
        case SD_SET_INTERFACE_PREPARE:                                   // set the interface so that 4 bit data mode is used - first set command mode
            if ((iActionResult = fnSendSD_command(ucAPP_CMD_CMD55, &ucResult, 0)) != UTFAT_SUCCESS) {
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            if (ucResult > R1_IN_IDLE_STATE) {
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // try again
                break;
            }
            iMemoryState[DISK_D] = SD_SET_INTERFACE;                     // fall through to send the application command
        case SD_SET_INTERFACE:
            if ((iActionResult = fnSendSD_command(ucSET_BUS_WIDTH_CMD6, &ucResult, ucData)) != UTFAT_SUCCESS) { // set relative address
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            iMemoryState[DISK_D] = SD_SET_BLOCK_LENGTH;
            SET_SPI_SD_INTERFACE_FULL_SPEED();                           // speed up the SPI interface since initialisation is complete (as well as data bus width)
        case SD_SET_BLOCK_LENGTH:
            if ((iActionResult = fnSendSD_command(ucSET_BLOCK_LENGTH_CMD16, &ucResult, ucData)) != UTFAT_SUCCESS) { // set relative address
                if (iActionResult == CARD_BUSY_WAIT) {
                    _return;                                             // read is taking time to complete so quit for the moment
                }
                fnInitialisationError(DISK_D, 0);                        // the card is behaving incorrectly - stop and try again later
                break;
            }
            iMemoryState[DISK_D] = DISK_MOUNTING_1;
            uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);              // schedule again to attempt mounting
            utDisks[DISK_D].ucDriveNumber = 0;
            #else
            iMemoryState[DISK_D] = DISK_MOUNTING_1;
            uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);              // schedule again to attempt mounting
            utDisks[DISK_D].ucDriveNumber = 0;
            SET_SD_CS_HIGH();
            SET_SPI_SD_INTERFACE_FULL_SPEED();                           // speed up the SPI interface since initialisation is complete
            #endif
            break;
        #endif

            // Disk mounting begins here
            //
        case DISK_MOUNTING_1:                                            // start mounting SD card or other medium
            utDisks[iDiskNumber].ulPresentSector = 0;                    // the first sector that will be accessed
            if ((iActionResult = ut_read_disk(&utDisks[iDiskNumber])) == CARD_BUSY_WAIT) { // read boot sector from disk
                _return;                                                 // read is taking time to complete so quit for the moment
            }
            iMemoryState[iDiskNumber] = DISK_MOUNTING_2;
        case DISK_MOUNTING_2:
            if (ERROR_SECTOR_INVALID == iActionResult) {
                fnCardNotFormatted(iDiskNumber);
                break;
            }
            else {
                iMemoryState[iDiskNumber] = DISK_MOUNTING_3;
                if (iActionResult != 0) {
                    fnInitialisationError(iDiskNumber, 0);               // the card is behaving incorrectly - stop and try again later
                    break;
                }
                else {
                    EXTENDED_BOOT_RECORD *ptrExtendedBootSector = (EXTENDED_BOOT_RECORD *)utDisks[iDiskNumber].ptrSectorData; // check to see whether partitions are defined rather than it being a boot record
                    BOOT_SECTOR_FAT32 *ptrBootSector = (BOOT_SECTOR_FAT32 *)utDisks[iDiskNumber].ptrSectorData;
                    if ((ptrExtendedBootSector->EBR_partition_table[0].partition_type != 0) && (ptrBootSector->bs_common.BS_FilSysType[0] != 'F')) {
                        utDisks[iDiskNumber].ulPresentSector = ((ptrExtendedBootSector->EBR_partition_table[0].start_sector[3] << 24) + (ptrExtendedBootSector->EBR_partition_table[0].start_sector[2] << 16) + (ptrExtendedBootSector->EBR_partition_table[0].start_sector[1] << 8) + ptrExtendedBootSector->EBR_partition_table[0].start_sector[0]);
                        if ((iActionResult = ut_read_disk(&utDisks[iDiskNumber])) ==  CARD_BUSY_WAIT) { // read boot sector from disk
                            _return;                                     // read is taking time to complete so quit for the moment
                        }
                    }
                    else {
                        utDisks[iDiskNumber].ulPresentSector = 0;        // the boot sector is 0
                    }
                }
            }                                                            // fall-through
        case DISK_MOUNTING_3:
            if (ERROR_SECTOR_INVALID == iActionResult) {
                fnCardNotFormatted(iDiskNumber);
                break;
            }
            else {
                int iFAT32 = 0;
                BOOT_SECTOR_FAT32 *ptrBootSector;
                unsigned long  ulTotalSections;
                unsigned long  ulFatSize;
                unsigned long  ulFirstDataSector;
                unsigned long  ulCountofClusters;
                unsigned short BPB_RsvdSecCnt;
                unsigned short BPB_FSInfo;
                unsigned short BPB_BytesPerSec;
                unsigned char  BPB_SecPerClus;
                BOOT_SECT_COM  *ptr_common;

                ptrBootSector = (BOOT_SECTOR_FAT32 *)utDisks[iDiskNumber].ptrSectorData; // the buffer content is a boot sector
                BPB_BytesPerSec = (ptrBootSector->boot_sector_bpb.BPB_BytesPerSec[1] << 8);
                ulFatSize = ((ptrBootSector->boot_sector_bpb.BPB_FATSz16[1] << 8) + ptrBootSector->boot_sector_bpb.BPB_FATSz16[0]);
                if (ulFatSize == 0) {                                    // FAT32 will indicate the size in the BPB_FATSz32 field instead
                    ulFatSize = ((ptrBootSector->BPB_FATSz32[3] << 24) + (ptrBootSector->BPB_FATSz32[2] << 16) + (ptrBootSector->BPB_FATSz32[1] << 8) + ptrBootSector->BPB_FATSz32[0]);
                    iFAT32 = 1;                                          // FAT32, irresepctive of the disk's size
                }
                ulTotalSections = ((ptrBootSector->boot_sector_bpb.BPB_TotSec16[1] << 8) + ptrBootSector->boot_sector_bpb.BPB_TotSec16[0]);
                if (ulTotalSections == 0) {                              // FAT32 will indicate the size in the BPB_FATSz32 field instead
                    ulTotalSections = ((ptrBootSector->boot_sector_bpb.BPB_TotSec32[3] << 24) + (ptrBootSector->boot_sector_bpb.BPB_TotSec32[2] << 16) + (ptrBootSector->boot_sector_bpb.BPB_TotSec32[1] << 8) + ptrBootSector->boot_sector_bpb.BPB_TotSec32[0]);
                }
                utDisks[iDiskNumber].utFAT.ulFatSize = ulFatSize;         // the sectors in a single FAT
                utDisks[iDiskNumber].utFAT.ucNumberOfFATs = ptrBootSector->boot_sector_bpb.BPB_NumFATs; // the number of FAT copies
                ulFatSize *= ptrBootSector->boot_sector_bpb.BPB_NumFATs; // the complete number of sections occupied by all FATs
                BPB_SecPerClus = ptrBootSector->boot_sector_bpb.BPB_SecPerClus;
                utDisks[iDiskNumber].utFAT.ucSectorsPerCluster = BPB_SecPerClus;
                BPB_RsvdSecCnt = ((ptrBootSector->boot_sector_bpb.BPB_RsvdSecCnt[1] << 8) + ptrBootSector->boot_sector_bpb.BPB_RsvdSecCnt[0]);
                utDisks[iDiskNumber].utFAT.ulFAT_start = (utDisks[iDiskNumber].ulPresentSector + BPB_RsvdSecCnt); // the boot sector plus reserved sectors
            #if defined USB_MSD_HOST
                if ((BPB_BytesPerSec < 512) || (BPB_SecPerClus == 0) || ((BPB_SecPerClus * BPB_BytesPerSec) > (64 * 1024))) // memory stick uses up to 128 BPB_SecPerClus
            #else
                if ((BPB_BytesPerSec < 512) || (BPB_SecPerClus == 0) || ((BPB_SecPerClus * BPB_BytesPerSec) > (32 * 1024)))
            #endif
                {
                    fnMemoryDebugMsg("Malformed - ");
                    fnCardNotFormatted(iDiskNumber);
                    break;
                }
                utDisks[iDiskNumber].usDiskFlags &= ~DISK_UNFORMATTED;
                utDisks[iDiskNumber].usDiskFlags |= DISK_FORMATTED;
                utDisks[iDiskNumber].utFAT.usBytesPerSector = BPB_BytesPerSec;
                ulFirstDataSector = (BPB_RsvdSecCnt + ulFatSize);
                ulCountofClusters = ((ulTotalSections - ulFirstDataSector)/ptrBootSector->boot_sector_bpb.BPB_SecPerClus);
                if ((ulCountofClusters < 65525) && (iFAT32 == 0)) {      // not FAT32
            #if defined UTFAT16
                    utDisks[iDiskNumber].usDiskFlags |= DISK_FORMAT_FAT16;
                    utDisks[iDiskNumber].ulDirectoryBase = 1;
                    ptr_common = &((BOOT_SECTOR_FAT12_FAT16 *)utDisks[iDiskNumber].ptrSectorData)->bs_common;
            #else
                    fnMemoryDebugMsg("NOT FAT32- ");
                    fnCardNotFormatted(iDiskNumber);
                    break;
            #endif
                }
                else {
                    BPB_FSInfo = ((ptrBootSector->BPB_FSInfo[1] << 8) + ptrBootSector->BPB_FSInfo[0]);
                    utDisks[iDiskNumber].ulDirectoryBase = ((ptrBootSector->BPB_RootClus[3] << 24) + (ptrBootSector->BPB_RootClus[2] << 16) + (ptrBootSector->BPB_RootClus[1] << 8) + ptrBootSector->BPB_RootClus[0]); // root directory start cluster
                    utDisks[iDiskNumber].ulPresentSector = utDisks[iDiskNumber].utFileInfo.ulInfoSector = (utDisks[iDiskNumber].ulPresentSector + BPB_FSInfo); // sector location of structure
                    ptr_common = &ptrBootSector->bs_common;
                }
                utDisks[iDiskNumber].utFAT.ulClusterCount = ((ulTotalSections - BPB_RsvdSecCnt - ulFatSize)/BPB_SecPerClus); // total cluster count
                utDisks[iDiskNumber].ulLogicalBaseAddress = (utDisks[iDiskNumber].utFAT.ulFAT_start + ulFatSize); // data start sector (logical block address)
                utDisks[iDiskNumber].ulVirtualBaseAddress = (utDisks[iDiskNumber].ulLogicalBaseAddress - (utDisks[iDiskNumber].ulDirectoryBase * utDisks[iDiskNumber].utFAT.ucSectorsPerCluster));
            #if defined UTFAT16
                if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                    utDisks[iDiskNumber].ulVirtualBaseAddress += (32 - 1); // fixed 16 kbyte root folder
                }
            #endif
                uMemcpy(utDisks[iDiskNumber].cVolumeLabel, ptr_common->BS_VolLab, sizeof(utDisks[iDiskNumber].cVolumeLabel));
                iMemoryState[iDiskNumber] = DISK_MOUNTING_4;
                if ((iActionResult = ut_read_disk(&utDisks[iDiskNumber])) ==  CARD_BUSY_WAIT) { // read boot sector from disk
                    _return;                                             // read is taking time to complete so quit for the moment
                }
            }                                                            // fall through
        case DISK_MOUNTING_4:                                            // optional step for FAT32 disks
            if ((iActionResult == 0) && (!(utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16))) { // if the information sector was valid
                INFO_SECTOR_FAT32 *ptrInfo = (INFO_SECTOR_FAT32 *)utDisks[iDiskNumber].ptrSectorData;
                if ((ptrInfo->FSI_TrailSig[3] == 0xaa) && (ptrInfo->FSI_TrailSig[2] == 0x55) &&  // check that the FSInfo sector is valid
                  (ptrInfo->FSI_LeadSig[3] == 0x41) && (ptrInfo->FSI_LeadSig[2] == 0x61) && (ptrInfo->FSI_LeadSig[1] == 0x52) && (ptrInfo->FSI_LeadSig[0] == 0x52) &&
                  (ptrInfo->FSI_StrucSig[3] == 0x61) && (ptrInfo->FSI_StrucSig[2] == 0x41) && (ptrInfo->FSI_StrucSig[1] == 0x72) && (ptrInfo->FSI_StrucSig[0] == 0x72)) {
                    utDisks[iDiskNumber].utFileInfo.ulFreeClusterCount = ((ptrInfo->FSI_Free_Count[3] << 24) + (ptrInfo->FSI_Free_Count[2] << 16) + (ptrInfo->FSI_Free_Count[1] << 8) + ptrInfo->FSI_Free_Count[0]);
                    utDisks[iDiskNumber].utFileInfo.ulNextFreeCluster = ((ptrInfo->FSI_Nxt_Free[3] << 24) + (ptrInfo->FSI_Nxt_Free[2] << 16) + (ptrInfo->FSI_Nxt_Free[1] << 8) + ptrInfo->FSI_Nxt_Free[0]);
                    if (utDisks[iDiskNumber].utFileInfo.ulNextFreeCluster <= (utDisks[iDiskNumber].utFileInfo.ulFreeClusterCount - 2)) { // check that the information is valid
                        utDisks[iDiskNumber].usDiskFlags |= FSINFO_VALID;
                    }
                }
            }
        #if DISK_COUNT > 1
            if (iDiskNumber == DISK_E) {
                fnMemoryDebugMsg("Disk E mounted");
            }
            else {
                fnMemoryDebugMsg("Disk D mounted");
            }
        #elif defined USB_MSD_HOST
            fnMemoryDebugMsg("Disk E mounted");
        #else
            fnMemoryDebugMsg("Disk D mounted");
        #endif
            if (GET_SDCARD_WP_STATE()) {
                utDisks[iDiskNumber].usDiskFlags |= (DISK_MOUNTED | WRITE_PROTECTED_SD_CARD); // the write protected disk is now ready for use
                fnMemoryDebugMsg(" (WP)");
            }
            else {
                utDisks[iDiskNumber].usDiskFlags |= DISK_MOUNTED;        // the disk is now ready for use (not write protected)
            }
            fnMemoryDebugMsg("\r\n");
            iMemoryOperation[iDiskNumber] |= _INITIALISED_MEMORY;
            iMemoryState[iDiskNumber] = DISK_STATE_READY;
        #if defined T_CHECK_CARD_REMOVAL
            uTaskerMonoTimer(OWN_TASK, T_CHECK_CARD_REMOVAL, E_CHECK_CARD_REMOVAL); // poll the SD card to detect removal
        #endif
        #if defined _WINDOWS
            if (utDisks[iDiskNumber].usDiskFlags & WRITE_PROTECTED_SD_CARD) {
                SD_card_state((SDCARD_MOUNTED | SDCARD_WR_PROTECTED | SDCARD_INSERTED), SDCARD_REMOVED);
            }
            else {
                SD_card_state((SDCARD_MOUNTED | SDCARD_INSERTED), SDCARD_REMOVED);
            }
            if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                SD_card_state((SDCARD_FORMATTED_16), (SDCARD_FORMATTED_32));
            }
            else {
                SD_card_state((SDCARD_FORMATTED_32), (SDCARD_FORMATTED_16));
            }
        #endif
            break;
        #if defined T_CHECK_CARD_REMOVAL
        case STATE_CHECKING_DISK_REMOVAL:
            #if defined SDCARD_DETECT_INPUT_POLL
            if (!(SDCARD_DETECTION())) {                                 // if card has been removed start remounting process
                fnInitialisationError(iDiskNumber, 0);                   // try to remount the card
            }
            else {
                iMemoryState[iDiskNumber] = DISK_STATE_READY;
                iMemoryOperation[iDiskNumber] |= (_INITIALISED_MEMORY);
                uTaskerMonoTimer(OWN_TASK, T_CHECK_CARD_REMOVAL, E_CHECK_CARD_REMOVAL); // poll the SD card to detect removal
            }
            #else
            if (UTFAT_SUCCESS != _utReadDisk[iDiskNumber](&utDisks[iDiskNumber], utDisks[iDiskNumber].ulPresentSector)) { // {53} read presently selected buffer to verify that the card is still responding
                fnInitialisationError(iDiskNumber, 0);                   // try to remount the card
            }
            else {                                                       // card polling was successful
                iMemoryState[iDiskNumber] = DISK_STATE_READY;
                iMemoryOperation[iDiskNumber] |= (_INITIALISED_MEMORY);
                uTaskerMonoTimer(OWN_TASK, T_CHECK_CARD_REMOVAL, E_CHECK_CARD_REMOVAL); // poll the SD card to detect removal
            }
            #endif
            break;
        #endif
        #if defined UTFAT_WRITE && defined UTFAT_FORMATTING
        case STATE_FORMATTING_DISK_1:                                    // start by creating a partition
            {
                EXTENDED_BOOT_RECORD *ptrExtendedBootRecord = (EXTENDED_BOOT_RECORD *)utDisks[iDiskNumber].ptrSectorData;
                unsigned long ulPartitionSize = (utDisks[iDiskNumber].ulSD_sectors - BOOT_SECTOR_LOCATION);
                uMemset(ptrExtendedBootRecord, 0, 512);                  // ensure content is blank
                ptrExtendedBootRecord->EBR_partition_table[0].starting_cylinder = 2; // fixed values
                ptrExtendedBootRecord->EBR_partition_table[0].starting_head     = 0x0c;
                ptrExtendedBootRecord->EBR_partition_table[0].partition_type    = 0x0b;
                ptrExtendedBootRecord->EBR_partition_table[0].ending_cylinder   = 0x38;
                ptrExtendedBootRecord->EBR_partition_table[0].ending_head       = 0xf8;
                ptrExtendedBootRecord->EBR_partition_table[0].ending_sector     = 0xb8;
                ptrExtendedBootRecord->EBR_partition_table[0].start_sector[0]   = BOOT_SECTOR_LOCATION;
                ptrExtendedBootRecord->EBR_partition_table[0].partition_size[0] = (unsigned char)(ulPartitionSize);
                ptrExtendedBootRecord->EBR_partition_table[0].partition_size[1] = (unsigned char)(ulPartitionSize >> 8);
                ptrExtendedBootRecord->EBR_partition_table[0].partition_size[2] = (unsigned char)(ulPartitionSize >> 16);
                ptrExtendedBootRecord->EBR_partition_table[0].partition_size[3] = (unsigned char)(ulPartitionSize >> 24);
                ptrExtendedBootRecord->ucCheck55 = 0x55;
                ptrExtendedBootRecord->ucCheckAA = 0xaa;
                if ((iActionResult = utCommitSector(&utDisks[iDiskNumber], 0)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        _return;                                         // write is taking time to complete so quit for the moment
                    }
                    iMemoryState[iDiskNumber] = DISK_NOT_FORMATTED;
                    break;
                }
            #if defined _WINDOWS
                iFormatCount = 0;
            #endif
                iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_2;     // fall through
            }
        case STATE_FORMATTING_DISK_2:                                    // add the boot sector
            {
                BOOT_SECTOR_FAT32 *ptrBootSector = (BOOT_SECTOR_FAT32 *)utDisks[iDiskNumber].ptrSectorData;
            #if defined UTFAT16
                BOOT_SECTOR_FAT12_FAT16 *ptrBootSector_16 = (BOOT_SECTOR_FAT12_FAT16 *)utDisks[iDiskNumber].ptrSectorData;
            #endif
                BOOT_SECT_COM *ptr_common;
                unsigned long ulPartitionSize;
                uMemset(ptrBootSector, 0, 508);                          // ensure content is blank (without overwriting 0xaa550000)
                ptrBootSector->boot_sector_bpb.BS_jmpBoot[0]     = 0xeb; // fixed values
                ptrBootSector->boot_sector_bpb.BS_jmpBoot[1]     = 0x58;
                ptrBootSector->boot_sector_bpb.BS_jmpBoot[2]     = 0x90;
                uStrcpy(ptrBootSector->boot_sector_bpb.BS_OEMName, "MSDOS5.0");
                utDisks[iDiskNumber].utFAT.usBytesPerSector      = BYTES_PER_SECTOR;
                ptrBootSector->boot_sector_bpb.BPB_BytesPerSec[0]= (unsigned char)(BYTES_PER_SECTOR);
                ptrBootSector->boot_sector_bpb.BPB_BytesPerSec[1]= (unsigned char)(BYTES_PER_SECTOR >> 8);
                ptrBootSector->boot_sector_bpb.BPB_SecPerTrk[0]  = 63;
                ptrBootSector->boot_sector_bpb.BPB_NumHeads[0]   = 0xff;                
                ptrBootSector->boot_sector_bpb.BPB_Media         = FIXED_MEDIA;                
            #if defined UTFAT16
                if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                    unsigned long ulSectorCnt = (unsigned long)(utDisks[iDiskNumber].ulSD_sectors); // total count of sectors on volume
                    ptrBootSector->boot_sector_bpb.BS_jmpBoot[1] = 0x3c;
                    ptrBootSector->boot_sector_bpb.BPB_RsvdSecCnt[0] = 8;
                    ptrBootSector->boot_sector_bpb.BPB_NumFATs   = 1;
                    ptrBootSector->boot_sector_bpb.BPB_RootEntCnt[1] = (unsigned char)(512 >> 8);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec16[0]   = (unsigned char)(ulSectorCnt);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec16[1]   = (unsigned char)(ulSectorCnt >> 8);
                    ulSectorCnt = ((utDisks[iDiskNumber].ulSD_sectors - 8)/257); // size of FAT16 in sectors
                    if (ulSectorCnt > 0xfe) {
                        ulSectorCnt = (0xfe + 8 + ((0xfe * 512)/sizeof(unsigned short)));
                        ptrBootSector->boot_sector_bpb.BPB_TotSec16[0]   = (unsigned char)(ulSectorCnt);
                        ptrBootSector->boot_sector_bpb.BPB_TotSec16[1]   = (unsigned char)(ulSectorCnt >> 8);
                        ulSectorCnt = 0xfe;                              // limit size of FAT16
                    }
                    ptrBootSector->boot_sector_bpb.BPB_FATSz16[0]    = (unsigned char)(ulSectorCnt);
                    ptrBootSector->boot_sector_bpb.BPB_FATSz16[1]    = (unsigned char)(ulSectorCnt >> 8);
                    utDisks[iDiskNumber].utFAT.ucSectorsPerCluster = 1;  // one sector per cluster assumed since only small systems expected with FAT16
                    ptrBootSector->ucCheck55 = 0x55;                     // mark that the sector is valid
                    ptrBootSector->ucCheckAA = 0xaa;
                    utDisks[iDiskNumber].ulLogicalBaseAddress = (8 + ulSectorCnt);
                    ptr_common = &ptrBootSector_16->bs_common;
                    uMemcpy(ptr_common->BS_FilSysType, "FAT16   ", 8);
                    ulFAT32size = ulSectorCnt;
                    utDisks[iDiskNumber].ulVirtualBaseAddress = (utDisks[iDiskNumber].ulLogicalBaseAddress + (32 - 1));
                }
                else {
            #endif
                    ptrBootSector->boot_sector_bpb.BPB_HiddSec[0]    = BOOT_SECTOR_LOCATION;
                    ulPartitionSize = (utDisks[iDiskNumber].ulSD_sectors - BOOT_SECTOR_LOCATION);
                    ptrBootSector->boot_sector_bpb.BPB_NumFATs       = NUMBER_OF_FATS;
                    ptrBootSector->boot_sector_bpb.BPB_RsvdSecCnt[0] = RESERVED_SECTION_COUNT;
                    ptrBootSector->boot_sector_bpb.BPB_TotSec32[0]   = (unsigned char)(ulPartitionSize);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec32[1]   = (unsigned char)(ulPartitionSize >> 8);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec32[2]   = (unsigned char)(ulPartitionSize >> 16);
                    ptrBootSector->boot_sector_bpb.BPB_TotSec32[3]   = (unsigned char)(ulPartitionSize >> 24);
                    if (ulPartitionSize <= 532480) {                     // disks up to 260MB
                        utDisks[iDiskNumber].utFAT.ucSectorsPerCluster = 1;
                    }
                    else if (ulPartitionSize <= 16777216) {              // disks up to 8GB
                        utDisks[iDiskNumber].utFAT.ucSectorsPerCluster = 8;
                    }
                    else if (ulPartitionSize <= 33554432) {              // disks up to 16GB
                        utDisks[iDiskNumber].utFAT.ucSectorsPerCluster = 16;
                    }
                    else if (ulPartitionSize <= 67108864) {              // disks up to 32GB
                        utDisks[iDiskNumber].utFAT.ucSectorsPerCluster = 32;
                    }
                    else {                                               // greater than 32GB
                        utDisks[iDiskNumber].utFAT.ucSectorsPerCluster = 64;
                    }                    
                    ulFAT32size = (((256 * utDisks[iDiskNumber].utFAT.ucSectorsPerCluster) + NUMBER_OF_FATS)/2);
                    ulFAT32size = (((ulPartitionSize - RESERVED_SECTION_COUNT) + (ulFAT32size - 1)) / ulFAT32size);
                    ptrBootSector->BPB_FATSz32[0]   = (unsigned char)(ulFAT32size);
                    ptrBootSector->BPB_FATSz32[1]   = (unsigned char)(ulFAT32size >> 8);
                    ptrBootSector->BPB_FATSz32[2]   = (unsigned char)(ulFAT32size >> 16);
                    ptrBootSector->BPB_FATSz32[3]   = (unsigned char)(ulFAT32size >> 24);
                    ptrBootSector->BPB_RootClus[0]  = 2;
                    ptrBootSector->BPB_FSInfo[0]    = 1;
                    ptrBootSector->BPB_BkBootSec[0] = BACKUP_ROOT_SECTOR;
                    utDisks[iDiskNumber].ulLogicalBaseAddress = (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT + (ulFAT32size * NUMBER_OF_FATS));
                    ptr_common = &ptrBootSector->bs_common;
                    uMemcpy(ptr_common->BS_FilSysType, "FAT32   ", 8);
                    utDisks[iDiskNumber].ulVirtualBaseAddress = utDisks[iDiskNumber].ulLogicalBaseAddress - (2 * utDisks[iDiskNumber].utFAT.ucSectorsPerCluster);
            #if defined UTFAT16
                }
            #endif
                ptrBootSector->boot_sector_bpb.BPB_SecPerClus = utDisks[iDiskNumber].utFAT.ucSectorsPerCluster;
                ptr_common->BS_DrvNum  = 0x80;
                ptr_common->BS_BootSig = 0x29;
            #if defined RANDOM_NUMBER_GENERATOR
                ptr_common->BS_VolID[0] = (unsigned char)fnRandom();     // can also be generated by combining present data and time
                ptr_common->BS_VolID[1] = (unsigned char)fnRandom();
                ptr_common->BS_VolID[2] = (unsigned char)fnRandom();
                ptr_common->BS_VolID[3] = (unsigned char)fnRandom();
            #else
                ptr_common->BS_VolID[0] = 1;
                ptr_common->BS_VolID[1] = 2;
                ptr_common->BS_VolID[2] = 3;
                ptr_common->BS_VolID[3] = 4;
            #endif
                if (utDisks[iDiskNumber].cVolumeLabel[0] == 0) {
                    uMemcpy(ptr_common->BS_VolLab, "NO NAME    ", 11);
                }
                else {
                    int i = 0;
                    while (i < 11) {
                        if ((ptr_common->BS_VolLab[i] = utDisks[iDiskNumber].cVolumeLabel[i]) == 0) {
                            while (i < 11) {
                                ptr_common->BS_VolLab[i++] = ' ';
                            }
                        }
                        i++;
                    }
                }
              //ptrBootSector->ucCheck55 = 0x55;                         // already exists in buffer
              //ptrBootSector->ucCheckAA = 0xaa;
                iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_3;
            }
            // Fall through intentional
            //
        case STATE_FORMATTING_DISK_3:
            #if defined UTFAT16
            if (!(utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16)) {
            #endif
                if ((iActionResult = utCommitSector(&utDisks[iDiskNumber], BOOT_SECTOR_LOCATION)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        _return;                                         // write is taking time to complete so quit for the moment
                    }
                    iMemoryState[iDiskNumber] = DISK_NOT_FORMATTED;
                    break;
                }
            #if defined UTFAT16
            }
            #endif
            iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_4;
            // Fall through intentional
            //
        case STATE_FORMATTING_DISK_4:
            {
                unsigned long ulLocation = (BOOT_SECTOR_LOCATION + BACKUP_ROOT_SECTOR);
            #if defined UTFAT16
                if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                    ulLocation = 0;                                      // FAT16 has a single boot sector at the start of the memory area
                }
            #endif
                if ((iActionResult = utCommitSector(&utDisks[iDiskNumber], ulLocation)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        _return;                                         // write is taking time to complete so quit for the moment
                    }
                    iMemoryState[iDiskNumber] = DISK_NOT_FORMATTED;
                    break;
                }
                else {
                    DIR_ENTRY_STRUCTURE_FAT32 *ptrVolumeEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)utDisks[iDiskNumber].ptrSectorData;
                    uMemset(utDisks[iDiskNumber].ptrSectorData, 0, 512);
                    uMemcpy(ptrVolumeEntry->DIR_Name, utDisks[iDiskNumber].cVolumeLabel, 11); //add the volume ID
                    ptrVolumeEntry->DIR_Attr = DIR_ATTR_VOLUME_ID;
                    fnSetTimeDate(ptrVolumeEntry, 1);                    // add creation time and date
                    ulFatSectors = 0;
                    iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_5;
                }
            }
            // Fall through intentional
            //
        case STATE_FORMATTING_DISK_5:                                    // ensure that the first boot cluster is blank (with volume ID)
            if (ulFatSectors < utDisks[iDiskNumber].utFAT.ucSectorsPerCluster) {
                if ((iActionResult = utCommitSector(&utDisks[iDiskNumber], (ulFatSectors + utDisks[iDiskNumber].ulLogicalBaseAddress))) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        _return;                                         // write is taking time to complete so quit for the moment
                    }
                    iMemoryState[iDiskNumber] = DISK_NOT_FORMATTED;
                    break;
                }
                ulFatSectors++;
                uMemset(utDisks[iDiskNumber].ptrSectorData, 0, 512);
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // schedule again to continue with next sector
                break;
            }
            iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_6;
            ulFatSectors = 0;
            ucFatCopies = 1;
            // Fall through intentional
            //
        case STATE_FORMATTING_DISK_6:
            #if defined UTFAT16
            if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                uMemcpy(utDisks[iDiskNumber].ptrSectorData, ucEmptyFAT16, sizeof(ucEmptyFAT16));
            }
            else {
                uMemcpy(utDisks[iDiskNumber].ptrSectorData, ucEmptyFAT32, sizeof(ucEmptyFAT32));
            }
            #else
            uMemcpy(utDisks[iDiskNumber].ptrSectorData, ucEmptyFAT32, sizeof(ucEmptyFAT32));
            #endif
            iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_7;
            // Fall through intentional
            //
        case STATE_FORMATTING_DISK_7:
            {
                unsigned long ulLocation = (ulFatSectors + (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT));
            #if defined UTFAT16
                if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                    ulLocation = 8;
                }
            #endif
                if ((iActionResult = utCommitSector(&utDisks[iDiskNumber], ulLocation)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        _return;                                         // write is taking time to complete so quit for the moment
                    }
                    iMemoryState[iDiskNumber] = DISK_NOT_FORMATTED;
                    break;
                }
                uMemset(utDisks[iDiskNumber].ptrSectorData, 0, sizeof(ucEmptyFAT32));
                ulFatSectors++;
                iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_8;
            }
            // Fall through intentional
            //
        case STATE_FORMATTING_DISK_8:
            if (ulFatSectors >= (ulFAT32size * ucFatCopies)) {
                if (ucFatCopies < NUMBER_OF_FATS) {
                    ucFatCopies++;
                    iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_6; // next FAT copy
                    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);      // schedule again to continue with next sector
                    break;
                }
            #if defined UTFAT_FULL_FORMATTING
                if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FULL) {
                    utDisks[iDiskNumber].usDiskFlags &= ~DISK_FORMAT_FULL;
                    ulFatSectors += (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT);
                #if defined UTFAT16
                    if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                        ulFatSectors -= (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT - 8);
                    }
                #endif
                    iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_8A;// the FAT has been cleared but now clear out all data sectors too
                }
                else {
                    iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_9; // all FAT copies have been initialised
                }
            #else
                iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_9;     // all FAT copies have been initialised
            #endif
            }
            else {
                unsigned long ulLocation = (ulFatSectors + (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT));
            #if defined UTFAT16
                if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                    ulLocation -= (BOOT_SECTOR_LOCATION + RESERVED_SECTION_COUNT - 8);
                }
            #endif
                if ((iActionResult = utCommitSector(&utDisks[iDiskNumber], ulLocation)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        _return;                                         // write is taking time to complete so quit for the moment
                    }
                    iMemoryState[iDiskNumber] = DISK_NOT_FORMATTED;
                    break;
                }
                ulFatSectors++;
                if ((ulFatSectors % 256) == 0) {
                    fnMemoryDebugMsg("*");
                }
            #if defined _WINDOWS
                if (++iFormatCount == 0x100) {                           // accelerate simulation
                    ulFatSectors = (ulFAT32size * ucFatCopies);
                    iFormatCount = 0;
                }
            #endif
            }
            uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);              // schedule again to continue with next sector
            break;
            #if defined UTFAT_FULL_FORMATTING
        case STATE_FORMATTING_DISK_8A:                                   // delete disk data
            if (ulFatSectors >= utDisks[iDiskNumber].ulSD_sectors) {
                iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_9;     // all disk data has been cleaned
            }
            else {
                if ((iActionResult = utCommitSector(&utDisks[iDiskNumber], ulFatSectors)) != 0) {
                    if (iActionResult == CARD_BUSY_WAIT) {
                        _return;                                         // write is taking time to complete so quit for the moment
                    }
                    iMemoryState[iDiskNumber] = DISK_NOT_FORMATTED;
                    break;
                }
                ulFatSectors++;
                if ((ulFatSectors % 256) == 0) {
                    fnMemoryDebugMsg("X");
                }
                #if defined _WINDOWS
                if (++iFormatCount == 0x200) {                           // accelerate simulation
                    ulFatSectors = (utDisks[iDiskNumber].utFAT.ulFAT_start + utDisks[iDiskNumber].utFAT.ulFatSize + (utDisks[iDiskNumber].utFAT.ulClusterCount * (utDisks[iDiskNumber].utFAT.usBytesPerSector/512)));
                    iFormatCount = 0;
                }
                #endif
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // schedule again to continue with next sector
                break;
            }
            #endif
        case STATE_FORMATTING_DISK_9:                                    // finally create the FAT32 info record
            {
                INFO_SECTOR_FAT32 *ptrInfoSector;
                unsigned long ulFreeCount;
            #if defined UTFAT16
                if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                    fnMemoryDebugMsg("Disk D formatted FAT16\r\n");
                    iMemoryState[iDiskNumber] = DISK_MOUNTING_1;
                    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);      // schedule again to mount the disk which has just been formatted
                    break;
                }
            #endif
                ptrInfoSector = (INFO_SECTOR_FAT32 *)utDisks[iDiskNumber].ptrSectorData;
                ulFreeCount = ((utDisks[iDiskNumber].ulSD_sectors - RESERVED_SECTION_COUNT - BOOT_SECTOR_LOCATION - (NUMBER_OF_FATS * ulFAT32size))/utDisks[iDiskNumber].utFAT.ucSectorsPerCluster);
                ptrInfoSector->FSI_LeadSig[3] = 0x41;
                ptrInfoSector->FSI_LeadSig[2] = 0x61;
                ptrInfoSector->FSI_LeadSig[1] = 0x52;
                ptrInfoSector->FSI_LeadSig[0] = 0x52;
                ptrInfoSector->FSI_StrucSig[3] = 0x61;
                ptrInfoSector->FSI_StrucSig[2] = 0x41;
                ptrInfoSector->FSI_StrucSig[1] = 0x72;
                ptrInfoSector->FSI_StrucSig[0] = 0x72;                                 
                fnAddInfoSect(ptrInfoSector, (ulFreeCount - 1), 3);      // one cluster occupied by root directory by default and first useable cluster number is 3
                ptrInfoSector->FSI_StrucSig[3] = 0x61;
                ptrInfoSector->FSI_StrucSig[2] = 0x41;
                ptrInfoSector->FSI_StrucSig[1] = 0x72;
                ptrInfoSector->FSI_StrucSig[0] = 0x72;
                ptrInfoSector->FSI_TrailSig[3] = 0xaa;
                ptrInfoSector->FSI_TrailSig[2] = 0x55;
                if ((iActionResult = utCommitSector(&utDisks[iDiskNumber], (BOOT_SECTOR_LOCATION + 1))) != 0) { // write info sector
                    if (iActionResult == CARD_BUSY_WAIT) {
                        _return;                                         // write is taking time to complete so quit for the moment
                    }
                    iMemoryState[iDiskNumber] = DISK_NOT_FORMATTED;
                    break;
                }
                iMemoryState[iDiskNumber] = STATE_FORMATTING_DISK_10;
                utDisks[iDiskNumber].usDiskFlags |= FSINFO_VALID;
        case STATE_FORMATTING_DISK_10:                                   // backup
                if ((iActionResult = utCommitSector(&utDisks[iDiskNumber], (BOOT_SECTOR_LOCATION + 1 + BACKUP_ROOT_SECTOR))) != 0) { // write info sector backup
                    if (iActionResult == CARD_BUSY_WAIT) {
                        _return;                                         // write is taking time to complete so quit for the moment
                    }
                    iMemoryState[iDiskNumber] = DISK_NOT_FORMATTED;
                    break;
                }
                fnMemoryDebugMsg("Disk D formatted\r\n");
                iMemoryState[iDiskNumber] = DISK_MOUNTING_1;
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // schedule again to mount the disk which has just been formatted
            }
            break;
        #endif                                                           // end #if defined UTFAT_WRITE && defined UTFAT_FORMATTING
        }
    }
    else if (iMemoryOperation[iDiskNumber] & _COUNTING_CLUSTERS) {       // cluster counting in progress
        if (ulClusterSectorCheck[iDiskNumber] >= utDisks[iDiskNumber].utFAT.ulFAT_start) {
            int iRepetitions = 10;                                       // read maximum 10 sectors at a time
            unsigned long ulSectorContent[512/sizeof(signed long)];      // long word aligned
            if (ulClusterSectorCheck[iDiskNumber] < (iRepetitions + utDisks[iDiskNumber].utFAT.ulFAT_start)) {
                iRepetitions = (ulClusterSectorCheck[iDiskNumber] - utDisks[iDiskNumber].utFAT.ulFAT_start);
            }
            while (iRepetitions-- >= 0) {
                if ((utReadDiskSector(&utDisks[iDiskNumber], ulClusterSectorCheck[iDiskNumber]--, ulSectorContent)) == UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
        #if defined UTFAT16
                    if (utDisks[iDiskNumber].usDiskFlags & DISK_FORMAT_FAT16) {
                        int i = 0;                                       // count free FAT16 clusters
                        while (i < (512/sizeof(signed long))) {
                            if (ulSectorContent[i] == 0) {
                                ulActiveFreeClusterCount[iDiskNumber] +=2;
                            }
                            else if (((unsigned short)(ulSectorContent[i]) == 0) || ((unsigned short)(ulSectorContent[i] >> 16) == 0)) {
                                ulActiveFreeClusterCount[iDiskNumber]++;
                            }
                            i++;
                        }
                    }
                    else {
         #endif
                        int i = 0;                                       // count free FAT32 clusters
                        while (i < (512/sizeof(signed long))) {
                            if (ulSectorContent[i++] == 0) {
                                ulActiveFreeClusterCount[iDiskNumber]++;
                            }
                        }
        #if defined UTFAT16
                    }
        #endif
                }
            }
            uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);              // schedule to run again
        }
        else {
            iMemoryOperation[iDiskNumber] &= ~_COUNTING_CLUSTERS;
            utDisks[iDiskNumber].utFileInfo.ulFreeClusterCount = ulActiveFreeClusterCount[iDiskNumber];
            fnInterruptMessage(cluster_task[iDiskNumber], UTFAT_OPERATION_COMPLETED); // inform that the cluster count has completed
            cluster_task[iDiskNumber] = 0;
        }
    }
    #if DISK_COUNT > 1
    }                                                                    // end of while for each disk
    iDiskNumber = DISK_D;
    #endif
    #endif                                                               // end #if defined SDCARD_SUPPORT || defined USB_MSD_HOST

    while (fnRead(ptrTaskTable->TaskID, ucInputMessage, HEADER_LENGTH)) {// check task input queue
        switch (ucInputMessage[MSG_SOURCE_TASK]) {
        case TIMER_EVENT:
    #if defined NAND_FLASH_FAT
            if (E_CLEAN_SPARE == ucInputMessage[MSG_TIMER_EVENT]) {
                fnCleanNAND();                                           // if there are area of the NAND Flash marked to be cleaned do the work as a background operation
                break;
            }
    #endif
    #if defined T_CHECK_CARD_REMOVAL
            if (E_CHECK_CARD_REMOVAL == ucInputMessage[MSG_TIMER_EVENT]) { // time to check whether the card is responding
                if (DISK_STATE_READY == iMemoryState[iDiskNumber]) {     // only perform removal check when the card is in the ready state - don't disturb operation when formatting or mounting
                    iMemoryState[iDiskNumber] = STATE_CHECKING_DISK_REMOVAL; // set state to cause next check
                    iMemoryOperation[iDiskNumber] &= ~(_INITIALISED_MEMORY);
                }
                else {
                    uTaskerMonoTimer(OWN_TASK, T_CHECK_CARD_REMOVAL, E_CHECK_CARD_REMOVAL); // poll the SD card to detect removal
                }
                uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);          // schedule the task to perform a check to see whether the card is responding (or has been removed)
            }
    #endif
    #if defined MANAGED_FILES && defined TIME_SLICED_FILE_OPERATION
			if (ucInputMessage[MSG_TIMER_EVENT] & _DELAYED_DELETE) {
                uFileManagedDelete(ucInputMessage[MSG_TIMER_EVENT] & ~(_DELAYED_DELETE));
                break;
			}
        #if defined MANAGED_FILE_WRITE
            else if (ucInputMessage[MSG_TIMER_EVENT] & _DELAYED_WRITE) {
                uFileManagedWrite(ucInputMessage[MSG_TIMER_EVENT] & ~(_DELAYED_WRITE));
            }
        #endif
        #if defined MANAGED_FILE_READ
            else if (ucInputMessage[MSG_TIMER_EVENT] & _DELAYED_READ) {
                uFileManagedRead(ucInputMessage[MSG_TIMER_EVENT] & ~(_DELAYED_READ));
            }
        #endif
    #endif
            break;

    #if defined SDCARD_DETECT_INPUT_INTERRUPT
        case INTERRUPT_EVENT:                                            // new state of SD card detection
            if (E_SDCARD_DETECTION_CHANGE == ucInputMessage[MSG_INTERRUPT_EVENT]) {
                if (SDCARD_DETECTION()) {                                // card has been inserted so try to mount it
                                                                         // do nothing in this case since the interrupt event will have already started mounting attempt
        #if defined _WINDOWS
                    SD_card_state(SDCARD_INSERTED, SDCARD_REMOVED);
        #endif
                }
                else {                                                   // card has been removed so stop operation
                    fnInitialisationError(iDiskNumber, 0);
                }
        #if defined SDCARD_SINGLE_EDGE_INTERRUPT
                fnPrepareDetectInterrupt();                              // devices that support interrupt one just one edge need to change the interrupt polarity now
        #endif
            }
            break;
    #endif

    #if defined USB_MSD_HOST
        case TASK_USB:
            fnRead(ptrTaskTable->TaskID, ucInputMessage, ucInputMessage[MSG_CONTENT_LENGTH]); // read the message content
            if (ucInputMessage[0] == MOUNT_USB_MSD) {                    // we should start mounting a memory stick that is ready to be worked with (DISK_E)
                if (iMemoryState[DISK_E] <= DISK_MOUNTING_1) {           // don't disturb if already mounting or being mounted
                    if (utDisks[DISK_E].ptrSectorData == 0) {            // if sector memoyr is not yet allocated
                        utDisks[DISK_E].ptrSectorData = SDCARD_MALLOC(512); // allocate a buffer which will contain data read from the present sector (this should be long word aligned and possibly in a specific memory in case DMA operation is to be used)
                    }
                    utDisks[DISK_E].ucDriveNumber = DISK_E;
                    iMemoryState[DISK_E] = DISK_MOUNTING_1;              // set to state initiating the mounting sequence
                    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);      // schedule the task to start mounting
                    fnDebugMsg("Memory stick mounting...\r\n");
                }
            }
            break;
    #endif
        }
    }
}
#endif                                                                   // end #if defined SDCARD_SUPPORT || defined MANAGED_FILES


#if defined SDCARD_SUPPORT || defined USB_MSD_HOST
static void fnCardNotFormatted(int iDisk)
{
    utDisks[iDisk].usDiskFlags |= DISK_UNFORMATTED;
    iMemoryState[iDisk] = DISK_NOT_FORMATTED;
#if DISK_COUNT > 1
    fnMemoryDebugMsg("Non-formatted ");
    if (iDisk == DISK_E) {
        fnMemoryDebugMsg("Memory stick\r\n");
    }
    else {
        fnMemoryDebugMsg("SD-card\r\n");
    }
#elif defined SDCARD_INTERFACE
    fnMemoryDebugMsg("Non-formatted SD-card\r\n");
#elif defined USB_MSD_HOST
    fnMemoryDebugMsg("Non-formatted Memory stick\r\n");
#endif
    #if defined _WINDOWS
    SD_card_state(SDCARD_MOUNTED, (SDCARD_FORMATTED_32 | SDCARD_FORMATTED_16));
    #endif
}

// Interrupt initisalisation sequence on error, remove power and start next try after a delay
//
static void fnInitialisationError(int iDisk, int iNotSupported)
{
    SET_SD_CS_HIGH();
    POWER_DOWN_SD_CARD();
    iMemoryState[iDisk] = SD_STATE_WAIT_CARD;                            // wait for next check of valid card
    uTaskerMonoTimer(OWN_TASK, T_NEXT_CHECK, E_POLL_SD_CARD);            // try again later
    if (iNotSupported != 0) {
        utDisks[iDisk].usDiskFlags = DISK_TYPE_NOT_SUPPORTED;
    #if defined _WINDOWS
        SD_card_state(0, (SDCARD_FORMATTED_32 | SDCARD_FORMATTED_16));
    #endif
    }
    else {
        fnMemoryDebugMsg("SD-card not responding\r\n");
        utDisks[iDisk].usDiskFlags = DISK_NOT_PRESENT;
        iMemoryOperation[iDisk] = _IDLE_MEMORY;
    #if defined _WINDOWS
        SD_card_state(0, (SDCARD_INSERTED | SDCARD_MOUNTED | SDCARD_FORMATTED_32 | SDCARD_FORMATTED_16 | SDCARD_WR_PROTECTED));
    #endif
    }
}

#if !defined NAND_FLASH_FAT
static const unsigned char *fnCreateCommand(unsigned char ucCommand, unsigned long ulValue)
{
    static unsigned char ucCommandBuffer[6];                             // space for a permanent command
    ucCommandBuffer[0] = ucCommand;
    ucCommandBuffer[1] = (unsigned char)(ulValue >> 24);
    ucCommandBuffer[2] = (unsigned char)(ulValue >> 16);
    ucCommandBuffer[3] = (unsigned char)(ulValue >> 8);
    ucCommandBuffer[4] = (unsigned char)(ulValue);
    ucCommandBuffer[5] = 0;                                              // dummy CRC
    return (const unsigned char *)ucCommandBuffer;
}
#endif

#if !defined SD_CONTROLLER_AVAILABLE && !defined NAND_FLASH_FAT          // SPI interface is local - SD card interface uses HW specific external routines
// Read a sector from SD card into the specified data buffer
//
static int fnGetSector(unsigned char *ptrBuf)
{
    unsigned char ucResponse;
    int iLength = 512;
    do {
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
    } while ((ucResponse = READ_SPI_DATA()) == 0xff);
    if (ucResponse != 0xfe) {
        return UTFAT_DISK_READ_ERROR;                                    // error
    }
    do {
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        *ptrBuf++ = READ_SPI_DATA();
    } while (--iLength);
    WRITE_SPI_CMD(0xff);                                                 // read and discard two CRC bytes
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    WRITE_SPI_CMD(0xff);
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    return UTFAT_SUCCESS;                                                // read successfully terminated
}

    #if defined UTFAT_WRITE
// Write present sector with buffer data
//
static int fnPutSector(unsigned char *ptrBuf, int iMultiBlock)
{
    #define DATA_TOKEN 0xfe
    unsigned char ucResponse;
    int iLength = 512;
    do {
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
    } while ((ucResponse = READ_SPI_DATA()) != 0xff);
    WRITE_SPI_CMD(DATA_TOKEN);                                           // transmit data byte
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    READ_SPI_DATA();                                                     // clear the receiver
    do {
        WRITE_SPI_CMD(*ptrBuf);                                          // transmit data byte
        ptrBuf++;
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        READ_SPI_DATA();                                                 // clear the receiver
    } while (--iLength);
    WRITE_SPI_CMD(0xff);                                                 // send two dummy CRC bytes
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    READ_SPI_DATA();
    WRITE_SPI_CMD(0xff);
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    READ_SPI_DATA();
    WRITE_SPI_CMD(0xff);                                                 // send two dummy CRC bytes
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    if ((ucResponse & 0x1f) != 0x05) {
        return UTFAT_DISK_WRITE_ERROR;
    }
    return UTFAT_SUCCESS;                                                // write successfully completed
}
    #endif

// Read a specified amount of data from present SD card sector into the specified data buffer (usStart and usStop are offset from start of sector and avoid other data outside of this range being overwritted)
//
static int fnReadPartialSector(unsigned char *ptrBuf, unsigned short usStart, unsigned short usStop)
{
    unsigned char ucResponse;
    unsigned short usLoopCounter = 0;
    // this may block so a timer may be required to protect it (100ms?)
    do {
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
    } while ((ucResponse = READ_SPI_DATA()) == 0xff);
    if (ucResponse != 0xfe) {
        fnMemoryDebugMsg("ERR-1!!!\r\n");
        return 0;                                                        // error
    }
    while (usLoopCounter < usStart) {
        WRITE_SPI_CMD(0xff);
        usLoopCounter++;
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        READ_SPI_DATA();                                                 // discard unwanted data at the start of the sector
    }
    while (usLoopCounter < usStop) {
        WRITE_SPI_CMD(0xff);
        usLoopCounter++;
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        *ptrBuf++ = READ_SPI_DATA();
    }
    while (usLoopCounter < 512) {
        WRITE_SPI_CMD(0xff);
        usLoopCounter++;
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
        READ_SPI_DATA();                                                 // discard any additional data, but always read a full 512 byte block
    }
    WRITE_SPI_CMD(0xff);                                                 // read and discard two CRC bytes
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    WRITE_SPI_CMD(0xff);
    WAIT_TRANSMISSON_END();                                              // wait until transmission complete
    ucResponse = READ_SPI_DATA();
    return UTFAT_SUCCESS;                                                // read terminated
}

// Send a command to the SD-card and return a result, plus optional returned arguments
//
static int fnSendSD_command(const unsigned char ucCommand[6], unsigned char *ucResult, unsigned char *ptrReturnData)
{
    static int iCommandYieldCount = 0;
    static int iCommandState = 0;
    int iRtn = UTFAT_SUCCESS;

    if (iCommandYieldCount > CommandTimeout) {                           // the present access is taking too long - quit with SD card error
        fnMemoryDebugMsg("TIMEOUT!!!\r\n");
        iRtn = ERROR_CARD_TIMEOUT;
    }
    else {
        switch (iCommandState) {
        case 0:
            if (fnWaitSD_ready(20) != 0) {                               // poll up to 20 times before yielding
                iCommandYieldCount++;                                    // monitor the maximum delay
                uTaskerStateChange(OWN_TASK, UTASKER_GO);                // switch to polling mode of operation
                return CARD_BUSY_WAIT;
            }  
            for ( ; iCommandState < 6; iCommandState++) {                // command content is always 6 bytes in length
                WRITE_SPI_CMD(ucCommand[iCommandState]);                 // send the command and arguments
                WAIT_TRANSMISSON_END();                                  // wait until transmission complete
                READ_SPI_DATA();                                         // read to clear the flag
            }                                                            // fall through intentional after sending the command
            if (iCommandYieldCount != 0) {
                uTaskerStateChange(OWN_TASK, UTASKER_STOP);              // switch to event mode if operation is continuing
                iCommandYieldCount = 0;
            }
        case 6:
            do {
                if (iCommandState++ > 26) {
                    iCommandYieldCount++;                                // monitor the maximum delay
                    iCommandState = 6;
                    uTaskerStateChange(OWN_TASK, UTASKER_GO);            // switch to polling mode of operation
                    return CARD_BUSY_WAIT;                               // poll up to 20 times before yielding
                }
                WRITE_SPI_CMD(0xff);                                     // send idle line
                WAIT_TRANSMISSON_END();                                  // wait until transmission complete
                *ucResult = READ_SPI_DATA();                             // read result byte
            } while (*ucResult & SD_CARD_BUSY);                          // poll the card until it is no longer indicating busy and returns the value
            if (ptrReturnData != 0) {                                    // if the caller requests data, read it here
                int iReturnLength;
                if (ucCommand[0] == SEND_CSD_CMD9) {                     // exception requiring 18 bytes
                    iReturnLength = 18;
                }
                else {
                    iReturnLength = 4;
                }
                for (iCommandState = 0; iCommandState < iReturnLength; iCommandState++) {
                    WRITE_SPI_CMD(0xff);                                 // send the command and arguments
                    WAIT_TRANSMISSON_END();                              // wait until transmission complete
                    *ptrReturnData++ = READ_SPI_DATA();
                }
            }
            break;
        }
    }
    if (iCommandYieldCount != 0) {
        uTaskerStateChange(OWN_TASK, UTASKER_STOP);                      // switch to event mode of operation since write has completed
        iCommandYieldCount = 0;
    }
    iCommandState = 0;
    return iRtn;
}

// Wait until the SD card is ready by reading until 0xff is returned
//
static int fnWaitSD_ready(int iMaxWait)
{
    do {
        if (iMaxWait-- == 0) {
            return 1;                                                    // maximum wait attempts executed
        }
        WRITE_SPI_CMD(0xff);
        WAIT_TRANSMISSON_END();                                          // wait until transmission complete
    } while (READ_SPI_DATA() != 0xff);                                   // 0xff expected when the device is ready
    return 0;                                                            // the device is now ready
}
#endif

#if defined NAND_FLASH_FAT
// Read a single sector from the disk - the sector number is specified by ulSectorNumber
//
static int utReadDisk(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    switch (iMemoryOperation[DISK_D] & _READING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation[DISK_D] |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            if (fnReadNANDsector(ulSectorNumber, 0, ptr_utDisk->ptrSectorData, 512) != 0) {
                fnMemoryDebugMsg(" - ECC check failed\r\n");             // this message can be ignored if the page is blank since it will not have a valid ECC
            }
            ptr_utDisk->ulPresentSector = ulSector;                      // update the sector which is presently valid

            iMemoryOperation[DISK_D] &= ~_READING_MEMORY;                // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Read a part of the specified sector to the buffer (avoiding overwriting all buffer content)
//
static int utReadPartialDiskData(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf, unsigned short usOffset, unsigned short usLength)
{
    static unsigned long ulSector;
    switch (iMemoryOperation[DISK_D] & _READING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation[DISK_D] |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            unsigned char ucTemp[512];
            if (fnReadNANDsector(ulSectorNumber, 0, ucTemp, 512) != 0) {
                fnMemoryDebugMsg(" - ECC check failed\r\n");             // this message can be ignored if the page is blank since it will not have a valid ECC
            }
            uMemcpy(ptrBuf, &ucTemp[usOffset], usLength);
            iMemoryOperation[DISK_D] &= ~_READING_MEMORY;                // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Read a single, complete sector from the disk to the specified buffer
//
static int utReadDiskSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf)
{
    static unsigned long ulSector;
    switch (iMemoryOperation[DISK_D] & _READING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation[DISK_D] |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            if (ptr_utDisk->usDiskFlags & DISK_TEST_MODE) {
                _fnReadNANDdata((unsigned short)(ulSectorNumber/NAND_PAGES_IN_BLOCK), (unsigned char)(ulSectorNumber%NAND_PAGES_IN_BLOCK), 0, ptrBuf, 528);
            }
            else {
                if (fnReadNANDsector(ulSectorNumber, 0, ptrBuf, 512) != 0) {
                    fnMemoryDebugMsg(" - ECC check failed\r\n");         // this message can be ignored if the page is blank since it will not have a valid ECC
                }
            }
            iMemoryOperation[DISK_D] &= ~_READING_MEMORY;                // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}
#else

    #if defined SDCARD_SUPPORT
// Read a single sector from the disk (SD-card) - the sector number is specified by ulSectorNumber
//
static int utReadSDsector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation[DISK_D] & _READING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation[DISK_D] |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            unsigned char ucResult;
            while ((iActionResult = fnSendSD_command(fnCreateCommand(READ_SINGLE_BLOCK_CMD17, ulSector), &ucResult, 0)) == CARD_BUSY_WAIT) {}
            if (iActionResult < 0) {
                SET_SD_CS_HIGH();
                iMemoryOperation[DISK_D] &= ~_READING_MEMORY;            // read operation has failed
                return iActionResult;
            }
            if (ucResult == 0) {
                if ((iActionResult = fnGetSector(ptr_utDisk->ptrSectorData)) != UTFAT_SUCCESS) { // {110a} read a single sector to disk buffer
                    SET_SD_CS_HIGH();
                    iMemoryOperation[DISK_D] &= ~_READING_MEMORY;        // read operation has failed
                    return iActionResult;
                }
                if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
                    ulSector /= 512;                                     // convert back to sector number from byte offset
                }
                ptr_utDisk->ulPresentSector = ulSector;                  // update the sector which is presently valid
            }
            SET_SD_CS_HIGH();
            iMemoryOperation[DISK_D] &= ~_READING_MEMORY;                // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Read a part of the specified sector to the buffer (avoiding overwriting all buffer content)
//
static int utReadPartialDiskData(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf, unsigned short usOffset, unsigned short usLength)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation[DISK_D] & _READING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation[DISK_D] |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            unsigned char ucResult;
            while ((iActionResult = fnSendSD_command(fnCreateCommand(READ_SINGLE_BLOCK_CMD17, ulSector), &ucResult, 0)) == CARD_BUSY_WAIT) {}
            if (iActionResult < 0) {
                SET_SD_CS_HIGH();
                iMemoryOperation[DISK_D] &= ~_READING_MEMORY;            // read operation has completed
                return iActionResult;
            }
            if (ucResult == 0) {
                fnReadPartialSector(ptrBuf, usOffset, (unsigned short)(usOffset + usLength)); // start reading a sector direct to buffer
            }
            SET_SD_CS_HIGH();
            iMemoryOperation[DISK_D] &= ~_READING_MEMORY;                // read operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Read a single, complete sector from the disk to the specified buffer
//
static int utReadDiskSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber, void *ptrBuf)
{
    static unsigned long ulSector;
    static int iActionResult = UTFAT_SUCCESS;
    switch (iMemoryOperation[DISK_D] & _READING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation[DISK_D] |= _READING_MEMORY;
        ulSector = ulSectorNumber;
    case _READING_MEMORY:
        {
            unsigned char ucResult;
            while ((iActionResult = fnSendSD_command(fnCreateCommand(READ_SINGLE_BLOCK_CMD17, ulSector), &ucResult, 0)) == CARD_BUSY_WAIT) {}
            if (iActionResult < 0) {
                SET_SD_CS_HIGH();
                iMemoryOperation[DISK_D] &= ~_READING_MEMORY;            // read operation has completed
                return iActionResult;
            }
            if (ucResult == 0) {
                iActionResult = fnGetSector(ptrBuf);                     // read a single sector to the buffer
            }
            SET_SD_CS_HIGH();
            iMemoryOperation[DISK_D] &= ~_READING_MEMORY;                // read operation has completed
        }
        break;
    }
    return iActionResult;
}
    #endif
#endif

// Read one sector from the disk specified by the drive number
//
static int ut_read_disk(UTDISK *ptr_utDisk)
{
    int iRtn = _utReadDisk[ptr_utDisk->ucDriveNumber](ptr_utDisk, ptr_utDisk->ulPresentSector); // read specified sector to buffer
    if (iRtn == UTFAT_SUCCESS) {                                         // read from SD media was successful and we expect to have the boot sector in the buffer
        BOOT_SECTOR_FAT32 *ptrBootSector = (BOOT_SECTOR_FAT32 *)ptr_utDisk->ptrSectorData;
        if ((ptrBootSector->ucCheck55 != 0x55) || (ptrBootSector->ucCheckAA != 0xaa)) {
            return ERROR_SECTOR_INVALID;                                 // the sector data is no valid
        }
    }
    return iRtn;
}

#if defined UTFAT_LFN_READ
// This routine checks through a chain of long file name entries to see whether there is a match with the reference file or directory name
// It is called multiple times, once for each possible entry
//
static int fnMatchLongName(OPEN_FILE_BLOCK *ptrOpenBlock, LFN_ENTRY_STRUCTURE_FAT32 *ptrLFN_entry, int iDeletedEntry, int *ptrSkip)
{
    int iReturn = MATCH_CONTINUE;
    if ((iDeletedEntry == 0) && (ptrLFN_entry->LFN_EntryNumber == DIR_NAME_FREE)) { // deleted entry when searching for valid entry
        iReturn = ENTRY_DELETED;
    }
    else {
        if (iDeletedEntry != 0) {                                        // searching for deleted entry
            if (ptrLFN_entry->LFN_EntryNumber == 0) {                    // ensure end of directory is detected
                iDeletedEntry = 0;
            }
            else if ((ptrLFN_entry->LFN_Zero0 != 0) || (ptrLFN_entry->LFN_Zero1 != 0) || (ptrLFN_entry->LFN_Zero2 != 0)) { // check whether it is clearly not a LFN
                iDeletedEntry = 0;
            }
        }
        if ((iDeletedEntry == 0) && ((ptrLFN_entry->LFN_Attribute & DIR_ATTR_MASK) != DIR_ATTR_LONG_NAME)) { // check whether this is a long file name entry
            if (ptrLFN_entry->LFN_EntryNumber == 0) {                    // end of directory entries reached
                if (DIR_ATTR_VOLUME_ID == (ptrLFN_entry->LFN_Attribute & DIR_ATTR_MASK)) {
                    iReturn = ENTRY_VOLUME_ID;
                }
                else {
                    iReturn = END_DIRECTORY_ENTRIES;
                }
            }
            else {
    #if defined UTFAT_EXPERT_FUNCTIONS
                if (ptrLFN_entry->LFN_EntryNumber == DIR_NAME_FREE) {    // possibly the deleted SFN alias after a deleted LFN
                    // It is not possible to check the short frame alias checksum since it has been deleted but if the name strings match we accept it
                    //
                    if (ptrOpenBlock->ptrFileNameMatch == ptrOpenBlock->ptrFileNameStart) {
                        ptrOpenBlock->ptrFileNameMatch = (CHAR *)ptrOpenBlock->ptrFileNameEnd; // reset in case it needs to be used again
                        return DELETED_LFN_MATCH_SUCCESSFUL;             // complete match has been verified
                    }
                }
    #endif
                iReturn = MATCH_NOT_LFN;                                 // this is not a long file name entry so ignore it
            }
            if (ptrLFN_entry->LFN_Attribute != DIR_ATTR_VOLUME_ID) {     // as long as the short file name is for a directory or file we add it to the SFN cache
                if (ptrOpenBlock->ulSFN_found < SFN_ENTRY_CACHE_SIZE) {  // as long as the cache has not been filled
                    uMemcpy(ptrOpenBlock->cSFN_entry[ptrOpenBlock->ulSFN_found++], ptrLFN_entry, 11); // copy SNF to cache and increment cache entry count
                }
            }
        }
        else {                                                           // LFN entry found
            int iLFN_start;
            int i = 13;                                                  // there are maximum 13 characters in an entry
            unsigned char ucCharacter = 0;
            unsigned char ucMatchCharacter;
#if defined UTFAT_EXPERT_FUNCTIONS
            if ((ptrLFN_entry->LFN_EntryNumber & 0x40) && ((ptrLFN_entry->LFN_EntryNumber != DIR_NAME_FREE) || (ptrOpenBlock->ptrFileNameMatch == (CHAR *)ptrOpenBlock->ptrFileNameEnd)))
#else
            if (ptrLFN_entry->LFN_EntryNumber & 0x40)                    // start of long file name (in case detected in the middle of a long file name)
#endif
            {
                iLFN_start = 1;
                ptrOpenBlock->ptrFileNameMatch = (CHAR *)ptrOpenBlock->ptrFileNameEnd; // reset the search string
            }
            else {
                iLFN_start = 0;
            }

            while (--i >= 0) {                                           // this assumes only English character set
                switch (i) {
                case 12:
                    ucCharacter = ptrLFN_entry->LFN_Name_12;
                    break;
                case 11:
                    ucCharacter = ptrLFN_entry->LFN_Name_11;
                    break;
                case 10:
                    ucCharacter = ptrLFN_entry->LFN_Name_10;
                    break;
                case 9:
                    ucCharacter = ptrLFN_entry->LFN_Name_9;
                    break;
                case 8:
                    ucCharacter = ptrLFN_entry->LFN_Name_8;
                    break;
                case 7:
                    ucCharacter = ptrLFN_entry->LFN_Name_7;
                    break;
                case 6:
                    ucCharacter = ptrLFN_entry->LFN_Name_6;
                    break;
                case 5:
                    ucCharacter = ptrLFN_entry->LFN_Name_5;
                    break;
                case 4:
                    ucCharacter = ptrLFN_entry->LFN_Name_4;
                    break;
                case 3:
                    ucCharacter = ptrLFN_entry->LFN_Name_3;
                    break;
                case 2:
                    ucCharacter = ptrLFN_entry->LFN_Name_2;
                    break;
                case 1:
                    ucCharacter = ptrLFN_entry->LFN_Name_1;
                    break;
                case 0:
                    ucCharacter = ptrLFN_entry->LFN_Name_0;
                    break;
                }
                if (ucCharacter == 0xff) {                               // past end of entry
                    continue;
                }
                if (ucCharacter == 0x00) {                               // null-terminator
                    continue;
                }
                ucMatchCharacter = *(--(ptrOpenBlock->ptrFileNameMatch));// the character that is to be matched
                if (ucCharacter != ucMatchCharacter) {                   // check the match, working backward, with pre-decrement
                    if ((ucCharacter >= 'A') && (ucCharacter <= 'Z')) {
                        ucCharacter += ('a' - 'A');
                    }
                    else if ((ucCharacter >= 'a') && (ucCharacter <= 'z')) {
                        ucMatchCharacter += ('a' - 'A');
                    }
                    else {
                        iReturn = MATCH_FALSE;
                        break;
                    }
                    if (ucCharacter != ucMatchCharacter) {               // last chance for case-insensitive match
                        iReturn = MATCH_FALSE;
                        break;
                    }
                }
            }

            if (iReturn != MATCH_FALSE) {                                // if no errors (yet)
                if (iLFN_start != 0) {                                   // start of long file name
    #if defined UTFAT_LFN_READ && ((defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE) || defined UTFAT_EXPERT_FUNCTIONS)
                    uMemcpy(&ptrOpenBlock->lfn_file_location, &ptrOpenBlock->present_location, sizeof(ptrOpenBlock->lfn_file_location)); // the directory location where the long file name entry chain begins
                    ptrOpenBlock->ucLFN_entries = 0;                     // reset LFN entry counter
    #endif
                    ptrOpenBlock->ucSFN_alias_checksum = ptrLFN_entry->LFN_Checksum; // the same checksum is expected in each entry in the long file name part
                }
                else if (ptrOpenBlock->ucSFN_alias_checksum != ptrLFN_entry->LFN_Checksum) { // if the checksum is not correct in the long file name chain reject it
                    iReturn = MATCH_FALSE;
                }
                if (iReturn != MATCH_FALSE) {
    #if defined UTFAT_LFN_READ && ((defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE) || defined UTFAT_EXPERT_FUNCTIONS)
                    ptrOpenBlock->ucLFN_entries++;                       // one more LFN entry counted belonging to this file name
    #endif
                    if ((ptrLFN_entry->LFN_EntryNumber & 0x3f) == 1) {   // the end of the long file name reached (this is never true when matching a deleted LFN))
                        if (ptrOpenBlock->ptrFileNameMatch == ptrOpenBlock->ptrFileNameStart) { // check whether complete name has been verified
                            ptrOpenBlock->ptrFileNameMatch = (CHAR *)ptrOpenBlock->ptrFileNameEnd; // reset in case it needs to be used again
                            if ((iDeletedEntry != 0) && (ptrLFN_entry->LFN_EntryNumber != DIR_NAME_FREE)) { // matching deleted files
                                return MATCH_FALSE;
                            }
                            return MATCH_SUCCESSFUL;                     // complete match has been verified
                        }
                        iReturn =  MATCH_FALSE;
                    }
                    else if (ptrOpenBlock->ptrFileNameMatch < ptrOpenBlock->ptrFileNameStart) {
                        iReturn =  MATCH_FALSE;
                    }
                }
            }
        }
    }

    if (iReturn < MATCH_CONTINUE) {                                      // if an error was detected
        ptrOpenBlock->ptrFileNameMatch = (CHAR *)ptrOpenBlock->ptrFileNameEnd; // reset on mis-matches or errors
    #if defined UTFAT_LFN_READ && ((defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE) || defined UTFAT_EXPERT_FUNCTIONS)
        ptrOpenBlock->lfn_file_location.directory_location.ulSector = 0; // mark as not LFN
    #endif
        if ((iDeletedEntry == 0) && (MATCH_FALSE == iReturn)) {          // non-matching LFN when matching valid entres
            *ptrSkip = (ptrLFN_entry->LFN_EntryNumber & 0x3f);           // the number of LFN entries that can be skipped
        }
    }
    return iReturn;
}
#endif


// Load a sector if the present sector content is not alread valid
//
static int fnLoadSector(UTDISK *ptr_utDisk, unsigned long ulSector)
{
    if (ptr_utDisk->ulPresentSector != ulSector) {                       // if the requested sector is not the already loaded one
#if defined UTFAT_WRITE
        if (ptr_utDisk->usDiskFlags & WRITEBACK_BUFFER_FLAG) {           // if changes have been made to the sector since its read it must be written back
            while (utCommitSector(ptr_utDisk, ptr_utDisk->ulPresentSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
            ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;
        }
#endif
        ptr_utDisk->ulPresentSector = ulSector;
        return (_utReadDisk[ptr_utDisk->ucDriveNumber](ptr_utDisk, ulSector)); // read the new sector
    }
    return UTFAT_SUCCESS;
}

// Read a part of a sector directly to a buffer (length is maximum 512 bytes)
//
static int fnLoadPartialData(UTDISK *ptr_utDisk, unsigned long ulSector, void *ptrBuf, unsigned short usOffset, unsigned short usLength)
{
    if (ptr_utDisk->ulPresentSector == ulSector) {                       // if the requested sector is already loaded (also if it has been modified)
        uMemcpy(ptrBuf, &ptr_utDisk->ptrSectorData[usOffset], usLength); // the data is already in the local sector copy so copy it directly to the caller's buffer
    }
    else {                                                               // the normal case is to load directly
        return (utReadPartialDiskData(ptr_utDisk, ulSector, ptrBuf, usOffset, usLength)); // read data from disk without changing the present sector
    }
    return UTFAT_SUCCESS;
}

// Increment the sector within a cluster - this is a simple increment of ulSector unless the end of a cluster is encountered, in which case it causes a move to the first sector in the next cluster
//
static int fnNextSector(UTDISK *ptr_utDisk, FILE_LOCATION *ptr_location)
{
    ptr_location->ulSector++;                                            // next sector
    if (((ptr_location->ulSector - ptr_utDisk->ulLogicalBaseAddress) % ptr_utDisk->utFAT.ucSectorsPerCluster) == 0) { // check whether the present cluster end has been reached
        unsigned long ulClusterSector;
        unsigned long ulCluster;
#if defined UTFAT16
        unsigned short usCluster;
        if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
            if (ptr_location->ulSector == (ptr_utDisk->ulVirtualBaseAddress + 2)) { // root folder end reached
                return UTFAT_FAT16_ROOT_FOLDER_EXHAUSTED;                // FAT16 root folder exhausted
            }
            else if (ptr_location->ulSector < (ptr_utDisk->ulVirtualBaseAddress + 2)) { // in the FAT16 root folder
                return UTFAT_SUCCESS;                                    // this never grows so its FAT16 entry always remains the same
            }
            ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 8)); // section where the FAT16 responsible for this cluster resides
            if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&usCluster, (unsigned short)((ptr_location->ulCluster & 0xff) * sizeof(unsigned short)), sizeof(unsigned short)) != UTFAT_SUCCESS) { // read directly to cluster value
                return UTFAT_DISK_READ_ERROR;
            }
            ulCluster = LITTLE_SHORT_WORD(usCluster);                    // ensure endien is correct
        }
        else {
            ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT32 responsible for this cluster resides
            if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
                return UTFAT_DISK_READ_ERROR;
            }
            ulCluster = LITTLE_LONG_WORD(ulCluster);                     // ensure endien is correct
        }
#else
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT responsible for this cluster resides
        if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
            return UTFAT_DISK_READ_ERROR;
        }
        ulCluster = LITTLE_LONG_WORD(ulCluster);                         // ensure endien is correct
#endif
        if ((ulCluster <= ptr_utDisk->ulDirectoryBase) || (ulCluster > ptr_utDisk->utFAT.ulClusterCount)) {
            return UTFAT_DIRECTORY_AREA_EXHAUSTED;
        }
        ptr_location->ulCluster = ulCluster;                             // set next cluster
        ptr_location->ulSector = ((ptr_location->ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);
    }
    return UTFAT_SUCCESS;
}

static int fnNextSectorCreate(UTDISK *ptr_utDisk, FILE_LOCATION *ptr_location, int iSeek) // added iSeek parameter
{
    ptr_location->ulSector++;                                            // next sector
    if (((ptr_location->ulSector - ptr_utDisk->ulLogicalBaseAddress) % ptr_utDisk->utFAT.ucSectorsPerCluster) == 0) { // check whether the present clust end has been reached
        unsigned long ulClusterSector;
        unsigned long ulCluster;
        unsigned long ulClusterMask;
    #if defined UTFAT16
        unsigned short usCluster;
        if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
            ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 8)); // section where the FAT16 responsible for this cluster resides
            if (iSeek != 0) {
                if (fnLoadSector(ptr_utDisk, ulClusterSector) != UTFAT_SUCCESS) { // when seeking we prefer to load a sector locally to avoid multiple reads when seeking through them
                    return UTFAT_DISK_READ_ERROR;
                }
            }
            if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&usCluster, (unsigned short)((ptr_location->ulCluster & 0xff) * sizeof(unsigned short)), sizeof(unsigned short)) != UTFAT_SUCCESS) { // read directly to cluster value
                return UTFAT_DISK_READ_ERROR;
            }
            ulCluster = LITTLE_SHORT_WORD(usCluster);                    // ensure endien is correct
            ulClusterMask = FAT16_CLUSTER_MASK;
        }
        else {
            ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT32 responsible for this cluster resides
            if (iSeek != 0) {
                if (fnLoadSector(ptr_utDisk, ulClusterSector) != UTFAT_SUCCESS) { // when seeking we prefer to load a sector locally to avoid multiple reads when seeking through them
                    return UTFAT_DISK_READ_ERROR;
                }
            }
            if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
                return UTFAT_DISK_READ_ERROR;
            }
            ulCluster = LITTLE_LONG_WORD(ulCluster);                     // ensure endien is correct
            ulClusterMask = CLUSTER_MASK;
        }
    #else
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT32 responsible for this cluster resides
        if (iSeek != 0) {
            if (fnLoadSector(ptr_utDisk, ulClusterSector) != UTFAT_SUCCESS) { // when seeking we prefer to load a sector locally to avoid multiple reads when seeking through them
                return UTFAT_DISK_READ_ERROR;
            }
        }
        if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
            return UTFAT_DISK_READ_ERROR;
        }
        ulCluster = LITTLE_LONG_WORD(ulCluster);                         // ensure endien is correct
        ulClusterMask = CLUSTER_MASK;
    #endif
        if ((ulCluster <= ptr_utDisk->ulDirectoryBase) || (ulCluster >= ulClusterMask)) { // invalid or end of current cluster - add a cluster so that the file can grow
    #if defined UTFAT_WRITE                                              // allow operation without write support
            ulCluster = fnAllocateCluster(ptr_utDisk, ptr_location->ulCluster, NEW_RELATIVE_CLUSTER); // create a new cluster so that it can be suednext cluster
    #endif
        }
        if (ptr_location->ulCluster > ptr_utDisk->utFAT.ulClusterCount) {
            return UTFAT_DIRECTORY_AREA_EXHAUSTED;
        }
        ptr_location->ulCluster = ulCluster;
        ptr_location->ulSector = ((ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);
    }
    return UTFAT_SUCCESS;
}

#if defined UTFAT_LFN_WRITE
static int fnDirectorySectorCreate(UTDISK *ptr_utDisk, FILE_LOCATION *ptr_location)
{
    unsigned long ulClusterSector;
    unsigned long ulCluster;
    unsigned long ulClusterMask;
    #if defined UTFAT16
    unsigned short usCluster;
    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 8)); // section where the FAT16 responsible for this cluster resides
        if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&usCluster, (unsigned short)((ptr_location->ulCluster & 0xff) * sizeof(unsigned short)), sizeof(unsigned short)) != UTFAT_SUCCESS) { // read directly to cluster value
            return UTFAT_DISK_READ_ERROR;
        }
        ulCluster = LITTLE_SHORT_WORD(usCluster);                        // ensure endien is correct
        ulClusterMask = FAT16_CLUSTER_MASK;
    }
    else {
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT32 responsible for this cluster resides
        if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
            return UTFAT_DISK_READ_ERROR;
        }
        ulCluster = LITTLE_LONG_WORD(ulCluster);                         // ensure endien is correct
        ulClusterMask = CLUSTER_MASK;
    }
    #else
    ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ptr_location->ulCluster >> 7)); // section where the FAT responsible for this cluster resides
    if (fnLoadPartialData(ptr_utDisk, ulClusterSector, (unsigned char *)&ulCluster, (unsigned short)((ptr_location->ulCluster & 0x7f) * sizeof(unsigned long)), sizeof(unsigned long)) != UTFAT_SUCCESS) { // read directly to cluster value
        return UTFAT_DISK_READ_ERROR;
    }
    ulCluster = LITTLE_LONG_WORD(ulCluster);                             // ensure endien is correct
    ulClusterMask = CLUSTER_MASK;
    #endif
    if ((ulCluster <= ptr_utDisk->ulDirectoryBase) || (ulCluster >= ulClusterMask)) { // invalid or end of current cluster - add a cluster so that the file can grow
        ulCluster = fnAllocateCluster(ptr_utDisk, ptr_location->ulCluster, (INITIALISE_DIR_EXTENSION | NEW_RELATIVE_CLUSTER)); // next cluster
    }
    if (ptr_location->ulCluster > ptr_utDisk->utFAT.ulClusterCount) {
        return UTFAT_DIRECTORY_AREA_EXHAUSTED;
    }
    ptr_location->ulCluster = ulCluster;
    ptr_location->ulSector = ((ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);
    return UTFAT_SUCCESS;
}

    #if defined UTFAT_LFN_WRITE
        #if defined UTFAT_LFN_WRITE_PATCH
// Create a short file name alisas that does not represent a valid file name and minimises the risk
// of causing problems with older systems supporting long file names and causing collision in a directory
// This implementation is based on the original algorithm  used by VFAT which can be found at https://lkml.org/lkml/2009/6/26/313.
// All credits go to the original author, whos comments are retained
//
static void fnCreateInvalidSFN_alias(CHAR cInvalidSFN_alias[12])         // the length includes the NT byte
{       
    unsigned long ulRandom;
    uMemset(cInvalidSFN_alias, 0, 12);                                   // set content to zero including NT byte

   /* we start with a space followed by nul as spaces at the
    * start of an entry are trimmed in FAT, which means that
    * starting the 11 bytes with 0x20 0x00 gives us a value which
    * cannot be used to access the file. It also means that the
    * value as seen from all Windows and Linux APIs is a constant
    */
    cInvalidSFN_alias[0]  = ' ';

   /* we use / and 2 nul bytes for the extension. These are
    * invalid in FAT and mean that utilities that show the
    * directory show no extension, but still work via the long
    * name for old Linux kernels
    */
    cInvalidSFN_alias[8]  = '/';

   /*
    * fill the remaining 6 bytes with random invalid values
    * This gives us a low collision rate, which means a low
    * chance of problems with chkdsk.exe and WindowsXP
    */
    do {
        ulRandom = (fnRandom() | (fnRandom() << 16));
        ulRandom &= 0x1f1f1f1f;
    } while (ulRandom == 0);                                             // avoid 0 as explained below
    uMemcpy(&cInvalidSFN_alias[2], &ulRandom, 4);                        // fill out [2..5] with random values
    ulRandom = (fnRandom() & 0x1f1f);
    uMemcpy(&cInvalidSFN_alias[6], &ulRandom, 2);                        // fill out [6..7] with random values

   /* a value of zero would leave us with only nul and spaces,
    * which would not work with older linux systems
    */
}
#else
// Create short file name according to true FAT32 LFN specification
//
static void fnCreateSFN_alias(CHAR cFileName[12])
{
    _EXCEPTION("TO DO!!");                                               // due to various advantages of the workaround in modern embedded systems it is possible that this will not be added
}
    #endif
#endif

#if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
// This function moves to a deleted short file name location and changes its name from undeleted to a fixed short file name which can then be renamed if needed
//
extern int utUndeleteFile(UTLISTDIRECTORY *ptrListDirectory)
{
    UTDISK *ptr_utDisk = &utDisks[ptrListDirectory->ptr_utDirObject->ucDrive];
    DIR_ENTRY_STRUCTURE_FAT32 *ptrDirectoryEntry;

    if (fnLoadSector(ptr_utDisk, ptrListDirectory->undelete_disk_location.directory_location.ulSector) != UTFAT_SUCCESS) { // move to the disk sector containing the directory and read its content
        return UTFAT_DISK_READ_ERROR;
    }
    ptrDirectoryEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData;
    ptrDirectoryEntry += ptrListDirectory->undelete_disk_location.ucDirectoryEntry; // move to the present entry
    if (ptrDirectoryEntry->DIR_Name[0] != DIR_NAME_FREE) {
        return UTFAT_FILE_NOT_WRITEABLE;                                 // if we have a reference to a non-deleted file we ignore the request
    }
    uMemcpy(ptrDirectoryEntry->DIR_Name, "UNDELETETXT", 11);             // give a valid short file name to the file so that it can be accessed again
    while (utCommitSector(ptr_utDisk, ptrListDirectory->undelete_disk_location.directory_location.ulSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
    ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;                   // the disk is up to date with the buffer
    return UTFAT_SUCCESS;
}
#endif
#endif


// Move to the next entry in a directory - this is a simple increment of ucDirectoryEntry as long as the end of the sector is not reached, in which case
// the next sector will need to be loaded 
//
static int fnNextDirectoryEntry(UTDISK *ptr_utDisk, DISK_LOCATION *ptr_location)
{
    if (++(ptr_location->ucDirectoryEntry) >= (ptr_utDisk->utFAT.usBytesPerSector/sizeof(DIR_ENTRY_STRUCTURE_FAT32))) { // end of present sector reached
        ptr_location->ucDirectoryEntry = 0;                              // start at first entry in next sector
        return (fnNextSector(ptr_utDisk, &ptr_location->directory_location)); // move to next sector associated with the directory
    }
    return UTFAT_SUCCESS;
}


static void fnLoadShortFileInfo(UTFILEINFO *ptr_utFileInfo, const DIR_ENTRY_STRUCTURE_FAT32 *ptrDirectoryEntry)
{
    int i;
    unsigned char c;
    CHAR *ptrShortName = ptr_utFileInfo->cFileName;
    unsigned char ucNT_info = ptrDirectoryEntry->DIR_NTRes;

    for (i = 0; i < 8; i++) {                                            // name
        c = (unsigned char)ptrDirectoryEntry->DIR_Name[i];
        if (c == ' ') {
            break;
        }
        if (c == 0x05) {
            c = DIR_NAME_FREE;
        }
        if ((ucNT_info & 0x08) && (c >= 'A') && (c <= 'Z')) {
            c += ('a' - 'A');                                            // convert to small letters
        }
        *ptrShortName++ = c;
    }
    if (ptrDirectoryEntry->DIR_Name[8] != ' ') {                         // extension
        *ptrShortName++ = '.';
        for (i = 8; i < 11; i++) {
            c = ptrDirectoryEntry->DIR_Name[i];
            if (c == ' ') {
                break;
            }
            if ((ucNT_info & 0x10) && (c >= 'A') && (c <= 'Z')) {
                c += ('a' - 'A');                                        // convert to small letters
            }
            *ptrShortName++ = c;
        }
    }
    *ptrShortName = 0;                                                   // terminate

    ptr_utFileInfo->ucFileAttributes = ptrDirectoryEntry->DIR_Attr;
    ptr_utFileInfo->ulFileSize = ((ptrDirectoryEntry->DIR_FileSize[3] << 24) + (ptrDirectoryEntry->DIR_FileSize[2] << 16) + (ptrDirectoryEntry->DIR_FileSize[1] << 8) + ptrDirectoryEntry->DIR_FileSize[0]);
    ptr_utFileInfo->usFileData = ((ptrDirectoryEntry->DIR_WrtDate[1] << 8) + ptrDirectoryEntry->DIR_WrtDate[0]);
    ptr_utFileInfo->usFileTime = ((ptrDirectoryEntry->DIR_WrtTime[1] << 8) + ptrDirectoryEntry->DIR_WrtTime[0]);
}

#if defined UTFAT_LFN_READ
// Calculate the checkum of a short file name alias
//
static unsigned char fnLFN_checksum(CHAR *cPtrSFN_alias)
{
   int i;
   unsigned char ucCheckSum = 0;
 
   for (i = 11; i; i--) {
      ucCheckSum = ((ucCheckSum & 1) << 7) + (ucCheckSum >> 1) + *cPtrSFN_alias++;
   }
   return ucCheckSum;
}

static int fnExtractLongFileName(CHAR cLongFileName[MAX_UTFAT_FILE_NAME], LFN_ENTRY_STRUCTURE_FAT32 *ptrLFN_Entry, unsigned char *ucLongFileNameLength, unsigned char *ucLFN_checksum)
{
    int i = *ucLongFileNameLength;
    if ((ptrLFN_Entry->LFN_Attribute & DIR_ATTR_MASK) != (DIR_ATTR_LONG_NAME)) {
        return -1;                                                       // not long file name attribute so don't test further
    }
    if (i == 0) {
        return -1;                                                       // protect against invalid length
    }
    #if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
    if (((ptrLFN_Entry->LFN_EntryNumber & 0x40)) && ((ptrLFN_Entry->LFN_EntryNumber != DIR_NAME_FREE) || (i == MAX_UTFAT_FILE_NAME))) {
    #else
    if (ptrLFN_Entry->LFN_EntryNumber & 0x40)                            // first LFN entry
    #endif
    {
        if ((ptrLFN_Entry->LFN_Name_12 != 0xff) && (ptrLFN_Entry->LFN_Name_12 != 0x00)) { // string ending without correct termination (padding or null terminator)
            cLongFileName[--i] = 0;                                      // force a null terminator
        }
        *ucLFN_checksum = ptrLFN_Entry->LFN_Checksum;                    // the checksum to be used to verify that the SFN alias is valid
        }
    }
    else if (*ucLFN_checksum != ptrLFN_Entry->LFN_Checksum) {            // it is expected that the SNF alias in all individual LFN entries are the same
        return -1;
    }
    if (i < 14) {                                                        // protect against over-long names
        uMemset(cLongFileName, 0, i);                                    // pad with zeros
        i = 0;
    }
    else {
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_12;                  // enter the characters backwards in the LFN buffer
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_11;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_10;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_9;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_8;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_7;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_6;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_5;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_4;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_3;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_2;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_1;
        cLongFileName[--i] = ptrLFN_Entry->LFN_Name_0;
    }
    *ucLongFileNameLength = i;                                           // update the length position
    return UTFAT_SUCCESS;                                                // long file name has been extracted
}
#endif

#if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
    static int fnExtractFileDetails(UTDISK *ptr_utDisk, UTFILEINFO *ptr_ut_fileInfo, DISK_LOCATION *ptr_disk_location, DISK_LOCATION *ptr_undelete_disk_location, unsigned char ucListFlags)
#else
    static int fnExtractFileDetails(UTDISK *ptr_utDisk, UTFILEINFO *ptr_ut_fileInfo, DISK_LOCATION *ptr_disk_location)
#endif
{
#if defined UTFAT_LFN_READ
    CHAR cLongFileName[MAX_UTFAT_FILE_NAME];                             // temporary buffer to hold LFN
    unsigned char ucLongFileNameLength = MAX_UTFAT_FILE_NAME;            // maximum LFN length supported
    unsigned char ucLFN_checksum = 0;
#endif
    DIR_ENTRY_STRUCTURE_FAT32 *ptrDirectoryEntry;
    int iFound = 0;                                                      // entry not yet found

    while (iFound == 0) {                                                // repeat until the complete entry has been retrieved
        if (fnLoadSector(ptr_utDisk, ptr_disk_location->directory_location.ulSector) != UTFAT_SUCCESS) { // move to the disk sector containing the directory and read its content
            return UTFAT_DISK_READ_ERROR;
        }
        ptrDirectoryEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData;
        ptrDirectoryEntry += ptr_disk_location->ucDirectoryEntry;        // move to the present entry
        switch (ptrDirectoryEntry->DIR_Name[0]) {
        case 0:                                                          // end of directory has been reached
            if (ptrDirectoryEntry->DIR_Attr & DIR_ATTR_VOLUME_ID) {      // skip volume ID
                break;
            }
            return UTFAT_END_OF_DIRECTORY;
        case DIR_NAME_FREE:                                              // 0xe5 means that a deleted entry has been found
#if defined UTFAT_LFN_READ
    #if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
            if (!(ucListFlags & DELETED_TYPE_LISTING)) {
                ucLongFileNameLength = MAX_UTFAT_FILE_NAME;              // reset the LFN index
            }
    #else
            ucLongFileNameLength = MAX_UTFAT_FILE_NAME;                  // reset the LFN index
    #endif
#endif
#if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
            if (ucListFlags & DELETED_TYPE_LISTING) {                    // don't skip non-deleted entries
                goto _entry_found;                                       // list a deleted entry
            }
#endif
            break;                                                       // skip the entry
        default:                                                         // non-deleted entry
#if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
            if (ucListFlags & DELETED_TYPE_LISTING) {                    // skip non-deleted entries if deleted items are to be listed
//              if (ucLongFileNameLength == MAX_UTFAT_FILE_NAME) {       // not in the process of collecting a possible long file name
                    break;
//              }
            }
#endif
#if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
_entry_found:                                                            // entry is to be listed
#endif
#if defined UTFAT_LFN_READ                                               // if long file name read support is enabled
            if (fnExtractLongFileName(cLongFileName, (LFN_ENTRY_STRUCTURE_FAT32 *)ptrDirectoryEntry, &ucLongFileNameLength, &ucLFN_checksum) != UTFAT_SUCCESS) {
                if (!(ptrDirectoryEntry->DIR_Attr & DIR_ATTR_VOLUME_ID)) { // if not a volume entry it is a short file name or short file name alias
                    fnLoadShortFileInfo(ptr_ut_fileInfo, ptrDirectoryEntry); // extract the short file information
                    if (ucLongFileNameLength != MAX_UTFAT_FILE_NAME) {   // long file name has been collected
    #if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
                        if ((ucListFlags & DELETED_TYPE_LISTING) || (fnLFN_checksum((CHAR *)(ptrDirectoryEntry->DIR_Name)) == ucLFN_checksum)) 
    #else
                        if (fnLFN_checksum((CHAR *)(ptrDirectoryEntry->DIR_Name)) == ucLFN_checksum) 
    #endif
                        {                                                // as long as the short file name alias checksum matches
                            uStrcpy(ptr_ut_fileInfo->cFileName, &cLongFileName[ucLongFileNameLength]); // copy the collected long file name to the file name string
                            iFound = 1;
                        }
                        else {
                            // If the checksum of the SFN alias doesn't match with the LFN entry it may mean that the SFN location has been deleted
                            // and reused by a system that doesn't understand LFN - it is therefore ignored
                            //
                            _EXCEPTION("Debug if interested");
                        }
                    }
                    else {
                        iFound = 1;                                      // mark that a file has been found and we can terminate after incrementing the directory entry
    #if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
                        if (ucListFlags & DELETED_TYPE_LISTING) {        // if listing deleted entries
                            int i;
                            ptr_ut_fileInfo->cFileName[0] = '~';         // deleted files/directories start with ~
                            for (i = 1; i < 10; i++) {                   // display short file name
                                if (ptr_ut_fileInfo->cFileName[i] < 0) {
                                    ptr_ut_fileInfo->cFileName[i] = (ptr_ut_fileInfo->cFileName[i] - 0x80);
                                }
                                if (ptr_ut_fileInfo->cFileName[i] == 0) {
                                    break;
                                }
                                if (ptr_ut_fileInfo->cFileName[i] <= ' ') {
                                    ptr_ut_fileInfo->cFileName[i] += 0x21; // replace deleted character for display purposes
                                }
                            }
        #if defined UTFAT_UNDELETE
                          uMemcpy(ptr_undelete_disk_location, ptr_disk_location, sizeof(DISK_LOCATION)); // copy the present location in case it is to be used to undelete this entry
        #endif
                        }
    #endif
                    }
                    ucLongFileNameLength = MAX_UTFAT_FILE_NAME;          // reset long file name counter
                }
            }
#else                                                                    // no LFN read supported
            if (!(ptrDirectoryEntry->DIR_Attr & DIR_ATTR_VOLUME_ID)) {   // if not a volume entry
                fnLoadShortFileInfo(ptr_ut_fileInfo, ptrDirectoryEntry); // extract the file information
                iFound = 1;                                              // mark that a file has been found and we can terminate after incrementing the directory entry
            }
#endif
            break;
        }
        if (fnNextDirectoryEntry(ptr_utDisk, ptr_disk_location) == UTFAT_DIRECTORY_AREA_EXHAUSTED) { // move to the next entry
            if (iFound == 1) {
                return UTFAT_FINAL_LISTING_ITEM_FOUND;
            }
            return UTFAT_DIRECTORY_AREA_EXHAUSTED;                       // quit since the complete directory space has been exhausted
        }
    }
    return UTFAT_SUCCESS;
}

// Read directory or file entry and fill out details for display purposes
//
extern int utReadDirectory(UTLISTDIRECTORY *ptr_utListDirectory, UTFILEINFO *ptr_ut_fileInfo)
{
    UTDISK *ptr_utDisk = &utDisks[ptr_utListDirectory->ptr_utDirObject->ucDrive]; // the disk that the directory is associated with
#if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
    return (fnExtractFileDetails(ptr_utDisk, ptr_ut_fileInfo, &ptr_utListDirectory->private_disk_location, &ptr_utListDirectory->undelete_disk_location, ptr_utListDirectory->ucListFlags));
#else
    return (fnExtractFileDetails(ptr_utDisk, ptr_ut_fileInfo, &ptr_utListDirectory->private_disk_location));
#endif
}


#if !defined _REMOVE_FORMATTED_OUTPUT                                    // no directory listing possible without formatted output
// Perform a directory listing to a buffer (with options for DOS and FTP styles)
//
extern int utListDir(UTLISTDIRECTORY *ptr_utListDirectory, FILE_LISTING *ptrFileListingInfo)
{
    unsigned char ucLineLength;
    int iResult;
    CHAR *ptrBuf = ptrFileListingInfo->ptrBuffer;
    CHAR *ptrName;
    UTFILEINFO utFileInfo;                                               // temporary file info object
    DISK_LOCATION previous_disk_loc;
    unsigned short usYear;
    unsigned char  ucDay, ucMonth, ucHour, ucMinutes;
    ptrFileListingInfo->usStringLength = 0;
    if (ptrFileListingInfo->ucStyle & FTP_TYPE_LISTING) {
        ucLineLength = FTP_STYLE_LIST_ENTRY_LENGTH;                      // the maximum length of a single entry line - the user's buffer must be adequate to accept this
    }
    else {
        ucLineLength = DOS_STYLE_LIST_ENTRY_LENGTH;                      // the maximum length of a single entry line - the user's buffer must be adequate to accept this
    }
    #if defined UTFAT_UNDELETE || defined UTFAT_EXPERT_FUNCTIONS
    ptr_utListDirectory->ucListFlags = ptrFileListingInfo->ucStyle;
    #endif
    ptrFileListingInfo->usItemsReturned = 0;
    while (ptrFileListingInfo->usItemsReturned < ptrFileListingInfo->usMaxItems) { // for maximum amount of items
        uMemcpy(&previous_disk_loc, &ptr_utListDirectory->private_disk_location, sizeof (ptr_utListDirectory->private_disk_location)); // backup in case we need to restore due to lack of buffer space
        iResult = utReadDirectory(ptr_utListDirectory, &utFileInfo);
        if ((iResult != UTFAT_SUCCESS) && (iResult != UTFAT_FINAL_LISTING_ITEM_FOUND)) { // read next directory or file in the present directory
            return UTFAT_NO_MORE_LISTING_ITEMS_FOUND;                    // no more items found
        }
        if ((ptrFileListingInfo->usStringLength + ucLineLength + uStrlen(utFileInfo.cFileName)) > ptrFileListingInfo->usBufferLength) {
            uMemcpy(&ptr_utListDirectory->private_disk_location, &previous_disk_loc, sizeof (ptr_utListDirectory->private_disk_location)); // restore location
            return UTFAT_NO_MORE_LISING_SPACE;                           // no more file listings fit in the present buffer
        }
        ucDay = (utFileInfo.usFileData & 0x1f);
        ucMonth = ((utFileInfo.usFileData >> 5) & 0xf);
        usYear = ((utFileInfo.usFileData >> 9) + 1980);
        ucMinutes = ((utFileInfo.usFileTime >> 5) & 0x3f);
        ucHour = (utFileInfo.usFileTime >> 11);
        ptrFileListingInfo->ucFileAttributes = utFileInfo.ucFileAttributes;
        if (ptrFileListingInfo->ucStyle & FTP_TYPE_LISTING) {
            int i = 10;
            unsigned short usRights;
            CHAR cAccess = 'd';
            if (utFileInfo.ucFileAttributes & DIR_ATTR_DIRECTORY) {
                usRights = 0x3ed;                                        // directory
            }
            else {                                                       // file
                if (utFileInfo.ucFileAttributes & DIR_ATTR_READ_ONLY) {
                    usRights = 0x124;
                }
                else {
                    usRights = 0x1a4;
                }
            }
            while (i--) {
                if (usRights & 0x200) {
                    *ptrBuf++ = cAccess;                                 // rights
                }
                else {
                    *ptrBuf++ = '-';                                     // no rights
                }
                switch (cAccess) {                                       // set next flag
                case 'r':
                    cAccess = 'w';
                    break;
                case 'w':
                    cAccess = 'x';
                    break;
                default:
                    cAccess = 'r';
                    break;
                }
                usRights <<= 1;
            }
            ptrBuf = uStrcpy(ptrBuf, " 1 502 502 ");
            if (utFileInfo.ucFileAttributes & DIR_ATTR_DIRECTORY) {      // when listing a directory display its minimum cluster size rather than a size of 0
                UTFAT *ptrFAT = &utDisks[ptr_utListDirectory->ptr_utDirObject->ucDrive].utFAT; // the FAT that the directory is associated with
                ptrBuf = fnBufferDec((ptrFAT->ucSectorsPerCluster * ptrFAT->usBytesPerSector), 0, ptrBuf); // file size
            }
            else {
                ptrBuf = fnBufferDec(utFileInfo.ulFileSize, 0, ptrBuf); // file size
            }
            *ptrBuf++ = ' ';
            switch (ucMonth) {
            default:
                ptrBuf = uStrcpy(ptrBuf, "Jan");
                break;
            case 2:
                ptrBuf = uStrcpy(ptrBuf, "Feb");
                break;
            case 3:
                ptrBuf = uStrcpy(ptrBuf, "Mar");
                break;
            case 4:
                ptrBuf = uStrcpy(ptrBuf, "Apr");
                break;
            case 5:
                ptrBuf = uStrcpy(ptrBuf, "May");
                break;
            case 6:
                ptrBuf = uStrcpy(ptrBuf, "Jun");
                break;
            case 7:
                ptrBuf = uStrcpy(ptrBuf, "Jul");
                break;
            case 8:
                ptrBuf = uStrcpy(ptrBuf, "Aug");
                break;
            case 9:
                ptrBuf = uStrcpy(ptrBuf, "Sep");
                break;
            case 10:
                ptrBuf = uStrcpy(ptrBuf, "Oct");
                break;
            case 11:
                ptrBuf = uStrcpy(ptrBuf, "Nov");
                break;
            case 12:
                ptrBuf = uStrcpy(ptrBuf, "Dec");
                break;
            }
            ptrBuf = fnBufferDec(ucDay, (LEADING_ZERO | WITH_SPACE), ptrBuf); // day of month
            ptrBuf = fnBufferDec(usYear, WITH_SPACE, ptrBuf);
        }
        else {                                                           // DOS style listing
            if (utFileInfo.ucFileAttributes & DIR_ATTR_READ_ONLY) {
                *ptrBuf++ = 'R';
            }
            else {
                *ptrBuf++ = '-';
            }
            if (utFileInfo.ucFileAttributes & DIR_ATTR_HIDDEN) {
                *ptrBuf++ = 'H';
            }
            else {
                *ptrBuf++ = '-';
            }
            if (utFileInfo.ucFileAttributes & DIR_ATTR_SYSTEM) {
                *ptrBuf++ = 'S';
            }
            else {
                *ptrBuf++ = '-';
            }
            if (utFileInfo.ucFileAttributes & DIR_ATTR_ARCHIVE) {
                *ptrBuf++ = 'A';
            }
            else {
                *ptrBuf++ = '-';
            }
            ptrBuf = fnBufferDec(ucDay, (LEADING_ZERO | WITH_SPACE), ptrBuf);
            *ptrBuf++ = '.';
            ptrBuf = fnBufferDec(ucMonth, (LEADING_ZERO), ptrBuf);
            *ptrBuf++ = '.';
            ptrBuf = fnBufferDec(usYear, 0, ptrBuf);
            *ptrBuf++ = ' ';
            ptrBuf = fnBufferDec(ucHour, (LEADING_ZERO | WITH_SPACE), ptrBuf);
            *ptrBuf++ = ':';
            ptrBuf = fnBufferDec(ucMinutes, (LEADING_ZERO), ptrBuf);
            if (utFileInfo.ucFileAttributes & DIR_ATTR_DIRECTORY) {
                if ((ptrFileListingInfo->ucStyle & FTP_TYPE_LISTING) == DOS_TYPE_LISTING) {
                    ptrBuf = uStrcpy(ptrBuf, " <DIR>           ");
                }
                ptrFileListingInfo->usDirectoryCount++;                  // count the directories in this listing
            }
            else {
                int iLen;
                CHAR cLenBuf[18];                                        // maximum decimal file length plus null terminator plus " <DIR> "fill
                CHAR *ptrLen = fnBufferDec(utFileInfo.ulFileSize, WITH_SPACE, cLenBuf);
                iLen = (17 - (ptrLen - cLenBuf));
                while (iLen--) {
                    *ptrBuf++ = ' ';
                }
                ptrBuf = uStrcpy(ptrBuf, cLenBuf);
                ptrFileListingInfo->usFileCount++;                       // count the files in this listing
                ptrFileListingInfo->ulFileSizes += utFileInfo.ulFileSize;// sum of the total file sizes in this listing
            }
        }
        *ptrBuf++ = ' ';
        ptrName = ptrBuf;                                                // pointer to start of the file name in the output string
        ptrBuf = uStrcpy(ptrBuf, utFileInfo.cFileName);                  // add the file name
        ptrFileListingInfo->ucNameLength = (unsigned char)(ptrBuf - ptrName); // length of file name for easy recognition in the output string
        if ((ptrFileListingInfo->ucStyle & NO_CR_LF_LISTING) == 0) {     // don't add CR and LF if not desired
            *ptrBuf++ = '\r';
            *ptrBuf++ = '\n';
        }
        ptrFileListingInfo->usStringLength = (ptrBuf - ptrFileListingInfo->ptrBuffer); // the complete string length in output buffer
        ptrFileListingInfo->usItemsReturned++;                           // additional item counted
        if (ptrFileListingInfo->ucStyle & NO_CR_LF_LISTING) {
            *ptrBuf = 0;
        }
        if (iResult == UTFAT_FINAL_LISTING_ITEM_FOUND) {
            return UTFAT_NO_MORE_LISTING_ITEMS_FOUND;                    // the content is the final content
        }
    }
    return UTFAT_SUCCESS;
}
#endif

static int fnCreateNameParagraph(const CHAR **pptrDirectoryPath, CHAR cDirectoryName[12])
{
#if defined UTFAT_LFN_WRITE
    int iLFN_force = FULLY_QUALIFIED_SHORT_NAME;
#endif
    int iLength = 0;
    int iEvenByte = 0;
    int iSeparatorFound = 0;
    CHAR cInputCharacter;
    unsigned char ucCharacterCharacteristics;

    uMemset(cDirectoryName, ' ', 11);                                    // start with blank short file name
    cDirectoryName[11] = NT_FLAG;
    while (1) {
        cInputCharacter = *(*pptrDirectoryPath)++;
        switch (cInputCharacter) {
        case '.':
            if ((iLength != 0) && (iLength <= 8)) {                      // separator found
                iSeparatorFound = 1;
                iLength = 8;
                continue;
            }
            break;
        case 0x0a:                                                       // for FTP compatibility
        case 0x0d:
            if (iLength == 0) {
                return INVALID_PARAGRAPH;
            }
        case 0:                                                          // end of path string
        case FORWARD_SLASH:                                              // directory level change or terminator
        case BACK_SLASH:
            if (iLength != 0) {                                          // if characters have been collected
                (*pptrDirectoryPath)--;                                  // set back to terminator of file name
#if defined UTFAT_LFN_WRITE
                return iLFN_force;
#else
                return ((cInputCharacter == 0) || (**pptrDirectoryPath == 0)); // return true if the end of the path has been found
#endif
            }                                                            // fall through to ignore
            continue;                                                    // ignore at start of string
        default:
            if ((unsigned char)cInputCharacter <= ' ') {                 // reject all invisible characters
#if defined UTFAT_LFN_WRITE
                if ((iLength != 0) && (cInputCharacter == ' ')) {        // spaces in the name force LFNs
                    iLFN_force = FULLY_QUALIFIED_LONG_NAME;
                    cInputCharacter = '_';                               // replace long file name accepted characters with '_';
                    break;
                }
#endif
                continue;
            }
            else {
                if ((unsigned char)cInputCharacter <= '~') {
                    ucCharacterCharacteristics = ucCharacterTable[cInputCharacter - '!'];
                    if (ucCharacterCharacteristics & (_CHAR_REJECT | _CHAR_REJECT_NON_JAP)) {
                        if ((iEvenByte == 0) || (!(ucCharacterCharacteristics & _CHAR_REJECT_NON_JAP))) { // don't apply these to second byte of Japanese characters
#if defined UTFAT_LFN_WRITE
                            if (ucCharacterCharacteristics & _CHAR_ACCEPT_LFN) {
                                cInputCharacter = '_';                   // replace long file name accepted characters with '_';
                                iLFN_force = FULLY_QUALIFIED_LONG_NAME;  // if this name is to be created it must be a long file name type due to this character
                            }
                            else {
                                continue;                                // reject
                            }
#else
                            continue;                                    // reject
#endif
                        }
                    }
                    else if (iEvenByte == 0) {                           // don't apply to second byte of Japanese characters
                        if (ucCharacterCharacteristics & _CHAR_CAPITAL) {
                            if (iSeparatorFound != 1) {
                                cDirectoryName[11] &= ~0x08;
                            }
                            else {
                                cDirectoryName[11] &= ~0x10;
                            }
                        }
                        else if (ucCharacterCharacteristics & _CHAR_SMALL) {
#if defined UTFAT_LFN_WRITE
                            iLFN_force = FULLY_QUALIFIED_LONG_NAME_SFNM; // if this name is to be created it must be a long file name type due the fact that it has small letters in the name (if it is to be searched for it may still mathc with a SFN)
#endif
                            cInputCharacter -= ('a' - 'A');              // convert to upper case
                            if (iSeparatorFound != 1) {
                                cDirectoryName[11] |= 0x08;
                            }
                            else {
                                cDirectoryName[11] |= 0x10;
                            }
                        }
                    }
                }
                else if ((((unsigned char)cInputCharacter >= 0x81) && ((unsigned char)cInputCharacter <= 0x9f)) || (((unsigned char)cInputCharacter >= 0xe0) && ((unsigned char)cInputCharacter <= 0xfc))) { // Japanese
                    iEvenByte = 1;
                    if ((iLength == 0) && (cInputCharacter == (CHAR)DIR_NAME_FREE)) {
                        cDirectoryName[iLength++] = 0x05;                // substitute
                        continue;                                        // continue with even byte
                    }
                }
                else {
                    continue;                                            // reject
                }
            }
            break;                                                       // accept
        }
        if (iLength >= 8) {
            if ((iSeparatorFound == 0) || (iLength >= 11)) {             // name part or extension is too long
#if defined UTFAT_LFN_READ
                while (1) {
                    switch (*(*pptrDirectoryPath)++) {                   // search the end of the long file name paragraph
                    case 0x0d:                                           // end of line considered fully qualified terminator (for FTP compatibility)
                    case 0x0a:
                    case 0:
                        (*pptrDirectoryPath)--;                          // set back to null-terminator of long file name
                        return FULLY_QUALIFIED_LONG_NAME;
                    case FORWARD_SLASH:
                    case BACK_SLASH:
                        *pptrDirectoryPath = (*pptrDirectoryPath - 1);   // set back to end of long file name
                        return LONG_NAME_PARAGRAPH;
                    }
                }
#else
                break;
#endif
            }
        }
        iEvenByte = 0;
        cDirectoryName[iLength++] = cInputCharacter;                     // collect short file name
    }
    return INVALID_PARAGRAPH;                                            // invalid
}


static UTDIRECTORY utDirectoryObjects[UT_DIRECTORIES_AVAILABLE] = {{0}};

extern UTDIRECTORY *utAllocateDirectory(unsigned char ucDisk, unsigned short usPathLength)
{
    int i = 0;
    while (i < UT_DIRECTORIES_AVAILABLE) {
        if (utDirectoryObjects[i].usDirectoryFlags == 0) {               // directory not allocated
            utDirectoryObjects[i].usDirectoryFlags = UTDIR_ALLOCATED;
            utDirectoryObjects[i].ucDrive = ucDisk;
            if (usPathLength != 0) {
                utDirectoryObjects[i].ptrDirectoryPath = SDCARD_MALLOC((MAX_MALLOC)(usPathLength + 1)); // reserve space for holding the directory path string (plus space for null terminator)
            }
            else {
                utDirectoryObjects[i].ptrDirectoryPath = 0;
            }
            utDirectoryObjects[i].usDirectoryPathLength = usPathLength;  // enter the maximum length that can be stored
            return &utDirectoryObjects[i];
        }
        i++;
    }
    return 0;                                                            // no directory object available for allocation
}

extern UTDIRECTORY *utFreeDirectory(UTDIRECTORY *ptrDirectory)
{
#if defined _WINDOWS
    if (ptrDirectory->ptrDirectoryPath != 0) {                           // if memory was allocated for use by a path string warn that it is not freed
        _EXCEPTION("Path string not freed!!!");
    }
#endif
    uMemset(ptrDirectory, 0, sizeof(UTDIRECTORY));                       // release the directory object so that it could be reused
    return 0;
}

static int _utOpenDirectory(OPEN_FILE_BLOCK *ptrOpenBlock, UTDIRECTORY *ptrDirObject, int iDeletedEntry)
{
#if defined UTFAT_LFN_READ
    int iSkip = 0;
    int iLFN_status = 0;
    int iMatchStatus;
#endif
    DISK_LOCATION *ptrDiskLocation = ptrOpenBlock->ptrDiskLocation;      // the start of the present directory
    const CHAR *ptrLocalDirPath = ptrOpenBlock->ptrLocalDirPath;
    UTDISK *ptr_utDisk = &utDisks[ptrDirObject->ucDrive];
    DIR_ENTRY_STRUCTURE_FAT32 *ptrFoundEntry = 0;
    int iMatchedParagraph;
    DISK_LOCATION DirStart;                                              // backup of original disk location

    uMemcpy(&DirStart, ptrDiskLocation, sizeof(DirStart));               // backup the original starting directory location

#if defined UTFAT_LFN_READ
    if ((*ptrLocalDirPath == BACK_SLASH) || (*ptrLocalDirPath == FORWARD_SLASH)) {
        ptrLocalDirPath++;                                               // ignore leading slashes
    }
#endif

    while (1) {                                                          // search through the directory to find the file/directory entry
#if defined UTFAT_LFN_READ
        ptrOpenBlock->ptrFileNameStart = ptrLocalDirPath;                // set a pointer to the name being searched for
#endif
        if ((ptrOpenBlock->iQualifiedPathType = fnCreateNameParagraph(&ptrLocalDirPath, ptrOpenBlock->cShortFileName)) < 0) { // fill the file name with a single paragraph from the file path
            return UTFAT_PATH_NOT_FOUND;                                 // error in the requested path
        }
#if defined UTFAT_LFN_READ
        ptrOpenBlock->ptrFileNameEnd = ptrLocalDirPath;                  // set to last character of name to match
        ptrOpenBlock->ptrFileNameMatch = (CHAR *)ptrOpenBlock->ptrFileNameEnd;
        if ((*ptrLocalDirPath == BACK_SLASH) || (*ptrLocalDirPath == FORWARD_SLASH)) {
            ptrLocalDirPath++;                                           // ignore leading slashes
        }
#endif
        iMatchedParagraph = 0;
        do {                                                             // handle entry, which could consist of multiple objects if LFN
            if (fnLoadSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) != UTFAT_SUCCESS) { // load new sector (this will only read from physical disk when the sector is not already in local cache)
                return UTFAT_DISK_READ_ERROR;
            }
            ptrFoundEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
            ptrFoundEntry += ptrDiskLocation->ucDirectoryEntry;          // move to the present entry
#if defined UTFAT_LFN_READ                                               // when LFN support is enabled
            if (iLFN_status == 0) {                                      // if a potential long file name match has been found
                if (iSkip == 0) {                                        // if not skipping non-matching LFN entries
                    uMemcpy(&ptrOpenBlock->present_location, ptrDiskLocation, sizeof(ptrOpenBlock->present_location)); // note the present location being checked
                    iMatchStatus = fnMatchLongName(ptrOpenBlock, (LFN_ENTRY_STRUCTURE_FAT32 *)ptrFoundEntry, iDeletedEntry, &iSkip); // try to match the file/directory with LFN entry
    #if defined UTFAT_LFN_WRITE
                    if (ENTRY_DELETED == iMatchStatus) {                 // if a deleted entry is found
                        if (ptrOpenBlock->ucDeleteCount == 0) {          // first deleted entry in possible contiguous deleted entries
                            uMemcpy(&ptrOpenBlock->DeleteLocationRef, ptrDiskLocation, sizeof(ptrOpenBlock->DeleteLocationRef)); // mark the location of first deleted entry
                        }
                        if (ptrOpenBlock->ucDeleteCount < DELETED_ENTRY_COUNT) { // limit the deleted entry count to the entries available
                            ptrOpenBlock->ucDeleteCount++;               // count contiguous deleted entries
                        }
                    }
                    else {                                               // not a deleted entry
                        if (ptrOpenBlock->ucDeleteCount != 0) {          // end of deleted space
                                                                         // convert ptrOpenBlock->ucDeleteCount to an index (index 0 is a single-hole location, 1 a double-hold-location, up to 20 for a 21-hole-location)
                            uMemcpy(&ptrOpenBlock->DeleteLocation[ptrOpenBlock->ucDeleteCount - 1], &ptrOpenBlock->DeleteLocationRef, sizeof(ptrOpenBlock->DeleteLocationRef)); // save details about the location of this hole type
                            ptrOpenBlock->DeleteLocationRef.directory_location.ulSector = 0; // reset for next use
                            ptrOpenBlock->ucDeleteCount = 0;             // reset for next use
                        }
                    }
    #endif
                }
                else {
                    iMatchStatus = ENTRY_DELETED;                        // ignore as if deleted
                    if (--iSkip == 0) {                                  // SFN alias being skipped
                        if (ptrOpenBlock->ulSFN_found < SFN_ENTRY_CACHE_SIZE) { // as long as the cache has not been filled
                            uMemcpy(ptrOpenBlock->cSFN_entry[ptrOpenBlock->ulSFN_found++], &ptrFoundEntry->DIR_Name, 11); // copy SNF alias to cache and increment cach entry count
                        }
                    }
                }
            }
            else {                                                       // match has been found in LFN part of name
                if (fnLFN_checksum((CHAR *)ptrFoundEntry->DIR_Name) == ptrOpenBlock->ucSFN_alias_checksum) { // check the checksum of the short file name alias associated with the matched long file name [note that compilers may suggest that lfn_match_block.ucSFN_alias_checksum could be used uninitialised but this is not the case because we only arrive here after it has been set on previous loop]
                    ptrOpenBlock->iQualifiedPathType = FULLY_QUALIFIED_SHORT_NAME;
                }
                else {
                    iLFN_status = 0;                                     // consider the match as failed
                }
                iMatchStatus = MATCH_NOT_LFN;
            }
            switch (iMatchStatus) {
            case MATCH_SUCCESSFUL:                                       // complete match has been found in the long file name part
                iLFN_status = 1;                                         // for an unconditional match on next directory entry
                break;
            case MATCH_CONTINUE:                                         // match is still correct but not completed
                break;
            case MATCH_FALSE:                                            // LFN entry found but not matching
              //iSkip = (ptrFoundEntry->DIR_Name[0] & 0x3f);             // the entries to skip (LFN parts plus one SFN alias)
                break;
            case END_DIRECTORY_ENTRIES:                                  // end if end found
                if (*ptrLocalDirPath == 0) {
                    return UTFAT_FILE_NOT_FOUND;                         // the directory path was OK but the file was not found
                }
                return UTFAT_PATH_NOT_FOUND;                             // the directory path was not OK
    #if defined UTFAT_EXPERT_FUNCTIONS
            case DELETED_LFN_MATCH_SUCCESSFUL:
                ptrOpenBlock->iQualifiedPathType = FULLY_QUALIFIED_LONG_NAME_SFNM;
                iLFN_status = 1;
                // Fall through intentionally
                //
    #endif
            case MATCH_NOT_LFN:
                if ((FULLY_QUALIFIED_SHORT_NAME == ptrOpenBlock->iQualifiedPathType) || (FULLY_QUALIFIED_LONG_NAME_SFNM == ptrOpenBlock->iQualifiedPathType)) { // possibly short file name
                    switch (ptrFoundEntry->DIR_Name[0]) {
                    case 0:                                              // end of directory reached
                        if (ROOT_DIRECTORY_SETTING & ptrOpenBlock->iRootDirectory) { // if a virtual root path is being set it has failed so reset
                            ptrDirObject->usDirectoryFlags &= (UTDIR_ALLOCATED);
                            ptrDirObject->usRelativePathLocation = 0;
                            if (ptrDirObject->ptrDirectoryPath != 0) {
                                *ptrDirObject->ptrDirectoryPath = 0;
                            }
                        }
                        if (ptrOpenBlock->iQualifiedPathType != 0) {
                            if (ptrOpenBlock->usDirFlags & UTDIR_SET_START) {
                                uMemcpy(ptrDiskLocation, &DirStart, sizeof(DirStart)); // set the location to the start of the lowest directory
                            }
                            return UTFAT_FILE_NOT_FOUND;
                        }
                        else {
                            return UTFAT_PATH_NOT_FOUND;
                        }
                    case DIR_NAME_FREE:                                  // deleted entry (free)
    #if defined UTFAT_EXPERT_FUNCTIONS
                        if ((iDeletedEntry != 0) && (iMatchStatus == MATCH_NOT_LFN)) {
                            iLFN_status = 1;                             // force match
                        }
                        else if (iMatchStatus != DELETED_LFN_MATCH_SUCCESSFUL) {
                            break;
                        }
    #else
                        break;
    #endif
                    default:
#endif
                        if (
#if defined UTFAT_LFN_READ
                            (iLFN_status != 0) ||                        // this short file name entry unconditionally belongs to a previous long file name block
#endif
                            ((!(ptrFoundEntry->DIR_Attr & DIR_ATTR_VOLUME_ID)) && (!uMemcmp(ptrFoundEntry->DIR_Name, ptrOpenBlock->cShortFileName, sizeof(ptrFoundEntry->DIR_Name))))) { // if entry has been found
                            ptrOpenBlock->ulCluster = ((ptrFoundEntry->DIR_FstClusHI[1] << 24) + (ptrFoundEntry->DIR_FstClusHI[0] << 16) + (ptrFoundEntry->DIR_FstClusLO[1] << 8) + ptrFoundEntry->DIR_FstClusLO[0]);
                            ptrDirObject->ptrEntryStructure = ptrFoundEntry; // set pointer to the entry so that the caller can extract additional information if required
                            if (!(ptrFoundEntry->DIR_Attr & DIR_ATTR_DIRECTORY)) {
                                ptrDirObject->public_file_location.ulCluster = ptrOpenBlock->ulCluster; // file's start cluster
                                ptrDirObject->public_file_location.ulSector = ((ptrOpenBlock->ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);// section referenced to logical base address
                                return UTFAT_PATH_IS_FILE;               // not a directory so can not be traced further
                            }
                            if ((ptrOpenBlock->iQualifiedPathType != 0) && (ptrOpenBlock->usDirFlags & UTDIR_DIR_AS_FILE)) { // if a directory is to be treated as a file, its location is preserved rather than moving to its content
                                return UTFAT_SUCCESS;                    // file matched
                            }
                            ptrDiskLocation->directory_location.ulCluster = ptrOpenBlock->ulCluster; // move to the new directory
                            ptrDiskLocation->directory_location.ulSector = ((ptrOpenBlock->ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);// section referenced to logical base address
                            ptrDiskLocation->ucDirectoryEntry = 2;       // skip "." and ".." entries
                            if (*ptrLocalDirPath == 0) {                 // check whether the end of the path has been reached
                                if (ROOT_DIRECTORY_RELOCATE == ptrOpenBlock->iRootDirectory) {
                                    return UTFAT_PATH_IS_ROOT_REF;       // directory path successfully found but relative to root
                                }
                                if (ROOT_DIRECTORY_SETTING & ptrOpenBlock->iRootDirectory) { // successfully found (virtual) root set set the location
                                    uMemcpy(&ptrDirObject->root_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->root_disk_location)); // this is the first open, which is setting the virtual root as seen by the user
                                    uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->public_disk_location));
                                }
                                return UTFAT_SUCCESS;                    // file matched
                            }
                            uMemcpy(&DirStart, ptrDiskLocation, sizeof(DirStart)); // set the directory location to the start of this directory
                            ptrOpenBlock->ptrLocalDirPath = ptrLocalDirPath;
#if defined UTFAT_LFN_READ
                            if (iLFN_status != 0) {                      // if forced match due to LFN
                                ptrOpenBlock->iQualifiedPathType = FULLY_QUALIFIED_LONG_NAME; // correct qualifier
                                iLFN_status = 0;                         // reset the flag
                            }
#endif
                            iMatchedParagraph = 1;                       // this paragraph matched, but not last one
                        }
#if defined UTFAT_LFN_READ
                        break;
                    }
                }
                break;
            }
#endif
            if (iMatchedParagraph != 0) {
                break;
            }
            if (fnNextDirectoryEntry(ptr_utDisk, ptrDiskLocation) != UTFAT_SUCCESS) { // move to next directory entry
                if (iMatchedParagraph != 0) {
                    return UTFAT_SUCCESS;
                }
                if (ptrOpenBlock->iQualifiedPathType != 0) {
                    if (ptrOpenBlock->usDirFlags & UTDIR_SET_START) {
                        uMemcpy(ptrDiskLocation, &DirStart, sizeof(DirStart)); // set the location to the start of the lowest directory
                    }
                    return UTFAT_FILE_NOT_FOUND;
                }
                else {
                    return UTFAT_PATH_NOT_FOUND;
                }
            }
        } while (1);
    }
}

static int _fnHandlePath(OPEN_FILE_BLOCK *ptrOpenBlock, const CHAR *ptrDirPath, UTDIRECTORY *ptrDirObject)
{
    UTDISK *ptr_utDisk = &utDisks[ptrDirObject->ucDrive];

    if (!(ptr_utDisk->usDiskFlags & DISK_MOUNTED)) {                     // if the disk is not ready for use quit immediately
        return UTFAT_DISK_NOT_READY;
    }

    ptrOpenBlock->iRootDirectory = 0;

    if (ptrDirPath != 0) {                                               // if there is a path string
        int iMoveUp = 0;
        unsigned short usPathTerminator = ptrDirObject->usRelativePathLocation; // present location path terminator
        while ((*ptrDirPath == '.') && (*(ptrDirPath + 1) == '.')) {     // if input ".."
            if (ptrDirObject->ptrDirectoryPath == 0) {                   // if no directory string path in use only referenced paths possible so always move to root
                ptrDirObject->usDirectoryFlags |= UTDIR_TEST_FULL_PATH;
                ptrDirPath = 0;
                break;
            }

            if (usPathTerminator <= 3) {                                 // can't move up any further
                return UTFAT_PATH_NOT_FOUND;
            }
            if (iMoveUp != 0) {
                usPathTerminator--;
            }
            while ((ptrDirObject->ptrDirectoryPath[usPathTerminator] != BACK_SLASH) && (ptrDirObject->ptrDirectoryPath[usPathTerminator] != FORWARD_SLASH)) {
                usPathTerminator--;
            }
            ptrDirPath += 2;
            iMoveUp++;
            if (*ptrDirPath == 0) {
                break;
            }
            ptrDirPath++;
        }
        if (iMoveUp != 0) {                                              // if we are to move up in the path ("../..")
            ptrDirObject->usDirectoryFlags |= UTDIR_TEST_FULL_PATH;      // use full path to set directory but don't move original (this flag is automatically reset)
            ptrOpenBlock->usDirFlags = ptrDirObject->usDirectoryFlags;   // backup original flags
            if (usPathTerminator > 3) {                                  // not root directory
                int iReturn;
                CHAR cTemp = ptrDirObject->ptrDirectoryPath[usPathTerminator]; // backup the location
                ptrDirObject->ptrDirectoryPath[usPathTerminator] = 0;    // temporary termination
                ptrDirObject->usDirectoryFlags &= ~UTDIR_DIR_AS_FILE;    // this may not be set when moving to next directory location
                if ((iReturn = utOpenDirectory(&ptrDirObject->ptrDirectoryPath[3], ptrDirObject)) != UTFAT_SUCCESS) { // try to locate the new directory
                    ptrDirObject->ptrDirectoryPath[usPathTerminator] = cTemp;// correct the original path
                    return UTFAT_PATH_NOT_FOUND;
                }
                if (*ptrDirPath != 0) {                                   // if the path should go downwards after locating the upward path location
                    ptrDirObject->usDirectoryFlags = ((ptrOpenBlock->usDirFlags & ~UTDIR_TEST_FULL_PATH) | UTDIR_TEST_FULL_PATH_TEMP);
                    if ((iReturn = utOpenDirectory(ptrDirPath, ptrDirObject)) != UTFAT_SUCCESS) { // continue downward search from the new location
                        ptrOpenBlock->usDirFlags = 0;
                    }
                    else if (ptrOpenBlock->usDirFlags & UTDIR_ALLOW_MODIFY_PATH) {
                        CHAR *ptrStart = &ptrDirObject->ptrDirectoryPath[usPathTerminator];                        
                        usPathTerminator += (uStrcpy(ptrStart, (ptrDirPath - 1)) - ptrStart);
                        return UTFAT_SUCCESS_PATH_MODIFIED;
                    }
                }
                if (ptrOpenBlock->usDirFlags & UTDIR_ALLOW_MODIFY_PATH) {// if path modification allowed
                    ptrDirObject->usRelativePathLocation = usPathTerminator;
                    return UTFAT_SUCCESS_PATH_MODIFIED;
                }
                ptrDirObject->ptrDirectoryPath[usPathTerminator] = cTemp;// correct the original path
                return iReturn;
            }
            else {
                if (ptrOpenBlock->usDirFlags & UTDIR_ALLOW_MODIFY_PATH) {// if path modification allowed
                    ptrDirObject->ptrDirectoryPath[3] = 0;
                    ptrDirObject->usRelativePathLocation = 3;
                    uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->root_disk_location, sizeof(ptrDirObject->private_disk_location)); // synchronise to root location
                    return UTFAT_SUCCESS_PATH_MODIFIED;
                }
            }
        }
    }
    ptrOpenBlock->ptrLocalDirPath = ptrDirPath;
    ptrOpenBlock->usDirFlags = ptrDirObject->usDirectoryFlags;           // flags on entry

    ptrDirObject->usDirectoryFlags &= ~(UTDIR_TEST_FULL_PATH | UTDIR_TEST_FULL_PATH_TEMP | UTDIR_TEST_REL_PATH | UTDIR_DIR_AS_FILE | UTDIR_SET_START | UTDIR_ALLOW_MODIFY_PATH); // these flags are automatically cleared

    if ((ptrOpenBlock->ptrLocalDirPath == 0) || (*ptrOpenBlock->ptrLocalDirPath == 0) || ((*ptrOpenBlock->ptrLocalDirPath == LINE_FEED) || (*ptrOpenBlock->ptrLocalDirPath == CARRIAGE_RETURN))) { // if root directory
        ptrOpenBlock->iRootDirectory = ROOT_DIRECTORY_REFERENCE;         // ready to work with root directory or referenced directory
    }
    else if (((*ptrOpenBlock->ptrLocalDirPath == BACK_SLASH) || (*ptrOpenBlock->ptrLocalDirPath == FORWARD_SLASH))) {
        ptrOpenBlock->usDirFlags |= UTDIR_TEST_REL_PATH;
        if (*(ptrOpenBlock->ptrLocalDirPath + 1) != 0) {                 // root directory referenced
            if ((*(ptrOpenBlock->ptrLocalDirPath + 1) == BACK_SLASH) || (*(ptrOpenBlock->ptrLocalDirPath + 1) == FORWARD_SLASH)) { // this allows "/\" to be recognised as root
                ptrOpenBlock->iRootDirectory = ROOT_DIRECTORY_SET;       // ready to work with root directory explicitly
            }
            else {
                ptrOpenBlock->iRootDirectory = ROOT_DIRECTORY_RELOCATE;  // ready to work with root directory or referenced directory
            }
        }
        else {
            ptrOpenBlock->iRootDirectory = ROOT_DIRECTORY_SET;           // ready to work with root directory explicitly
        }
    }

    if (!(ptrOpenBlock->usDirFlags & UTDIR_VALID)) {
        ptrOpenBlock->ulCluster = ptr_utDisk->ulDirectoryBase;           // the first cluster in the root directory
        ptrDirObject->private_disk_location.directory_location.ulCluster = ptrOpenBlock->ulCluster;
#if defined UTFAT16
        if ((ptrOpenBlock->ulCluster <= 1) && (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16)) { // if FAT16 root folder
            ptrDirObject->private_disk_location.directory_location.ulSector = (ptr_utDisk->utFAT.ucSectorsPerCluster + ptr_utDisk->ulVirtualBaseAddress - (32 - 1));// the sector in which the directory entries begin
        }
        else {
            ptrDirObject->private_disk_location.directory_location.ulSector = (ptrOpenBlock->ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster);// section referenced to logical base address
            ptrDirObject->private_disk_location.directory_location.ulSector += ptr_utDisk->ulVirtualBaseAddress;// the sector in which the directory entries begin
        }
#else
        ptrDirObject->private_disk_location.directory_location.ulSector = (ptrOpenBlock->ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster);// section referenced to logical base address
        ptrDirObject->private_disk_location.directory_location.ulSector += ptr_utDisk->ulVirtualBaseAddress;// the sector in which the directory entries begin
#endif
        ptrDirObject->private_disk_location.ucDirectoryEntry = 0;        // reset the entry index
        uMemcpy(&ptrDirObject->root_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->root_disk_location)); // enter the fixed root location
        uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->root_disk_location)); // ensure that the public entry can not be used with invalid value
        ptrDirObject->usDirectoryFlags = (UTDIR_ALLOCATED | UTDIR_VALID | UTDIR_REFERENCED);// {100a} the entry is now valid
        ptrOpenBlock->iRootDirectory |= ROOT_DIRECTORY_SETTING;          // mark that the (virtual) root has been set
        if (ptrDirObject->ptrDirectoryPath != 0) {                       // if a root path is being used and root is to be set, set also for virtual root location 
            uStrcpy(ptrDirObject->ptrDirectoryPath, "D:\\");             // set root path
            ptrDirObject->usRelativePathLocation = 3;                    // the relative path is equal to the root path
        }
    }

    if (ptrOpenBlock->usDirFlags & (UTDIR_TEST_FULL_PATH | UTDIR_TEST_FULL_PATH_TEMP | UTDIR_TEST_REL_PATH | UTDIR_REFERENCED)) {
        if (ptrOpenBlock->usDirFlags & UTDIR_TEST_FULL_PATH) {           // set temporary root directory
            uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->root_disk_location, sizeof(ptrDirObject->private_disk_location)); // {102a} synchronise to root location
        }
        else if (!(ptrOpenBlock->usDirFlags & UTDIR_TEST_FULL_PATH_TEMP)) {
            if (ptrOpenBlock->iRootDirectory & (ROOT_DIRECTORY_RELOCATE | ROOT_DIRECTORY_SET)) {
                uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->root_disk_location, sizeof(ptrDirObject->public_disk_location)); // synchronise to root
                if (ptrOpenBlock->iRootDirectory == ROOT_DIRECTORY_SET) {
                    return UTFAT_PATH_IS_ROOT;                           // inform that the location is root
                }
            }
            else {
                uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->public_disk_location)); // synchronise to present location
            }
        }                                                                // UTDIR_TEST_FULL_PATH_TEMP continues using present public values to continue a search
        ptrOpenBlock->ptrDiskLocation = &ptrDirObject->public_disk_location;        
    }
    else {
        ptrOpenBlock->ptrDiskLocation = &ptrDirObject->private_disk_location; // work with the private disk location pointer so that its adsolute base location is set
        uMemcpy(&ptrDirObject->public_disk_location, &ptrDirObject->private_disk_location, sizeof(ptrDirObject->private_disk_location)); // synchronise to present location
    }
    if (ptrOpenBlock->iRootDirectory & (ROOT_DIRECTORY_REFERENCE | ROOT_DIRECTORY_SET)) { // if root directory reference
        return UTFAT_SUCCESS;                                            // ready to work with root directory or referenced directory
    }
    ptrOpenBlock->iContinue = 1;                                         // the function has done its work and requires that the caller function continues to complete
    return UTFAT_SUCCESS;
}

// Open or modify a directory object
//
extern int utOpenDirectory(const CHAR *ptrDirPath, UTDIRECTORY *ptrDirObject)
{
    int iReturn;
    OPEN_FILE_BLOCK openBlock;
    uMemset(&openBlock, 0, sizeof(openBlock));                           // initialise open file block

    iReturn = _fnHandlePath(&openBlock, ptrDirPath, ptrDirObject);       // handle the input path string
    if (openBlock.iContinue == 0) {                                      // if the path handling had an error or completed all work
        return iReturn;                                                  // return with code
    }

    return _utOpenDirectory(&openBlock, ptrDirObject, 0);
}

    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
static FILE_DATA_CACHE *fnGetDataCache(void)
{
    int i;
    for (i = 0; i < UTFAT_FILE_CACHE_POOL; i++) {                        // attempt to find a free buffer in the file data cache pool
        if (FileDataCache[i].ucFileCacheState == FILE_BUFFER_FREE) {     // if a free cache is found
            FileDataCache[i].ucFileCacheState = (FILE_BUFFER_IN_USE);    // initial cache state
            return &FileDataCache[i];                                    // allocate a data cache for use by the file
        }
    }
    return 0;                                                            // presently no free cache available in the pool
}

static int fnHandleFileDataCache(UTDISK *ptr_utDisk, UTFILE *ptr_utFile, FILE_DATA_CACHE *ptrDataCache, int iNoLoad)
{
    if (ptrDataCache->ucFileCacheState & FILE_BUFFER_VALID) {            // the cache has valid data in it
        if (ptrDataCache->ulFileCacheSector != ptr_utFile->public_file_location.ulSector) { // the sector doesn't match
    #if defined UTFAT_WRITE
            if (ptrDataCache->ucFileCacheState & FILE_BUFFER_MODIFIED) { // if modified data waiting to be saved
                while (utCommitSectorData(ptr_utDisk, ptrDataCache->ucFileDataCache, ptrDataCache->ulFileCacheSector) == CARD_BUSY_WAIT) {} // commit the data to the disk
                ptrDataCache->ucFileCacheState &= ~(FILE_BUFFER_MODIFIED); // no longer needs to be saved
            }
    #endif
            if (iNoLoad == 0) {                                          // if the load operation is not blocked
                if (utReadDiskSector(ptr_utDisk, ptr_utFile->public_file_location.ulSector, ptrDataCache->ucFileDataCache) != UTFAT_SUCCESS) { // read complete sector directly to buffer
                    return UTFAT_DISK_READ_ERROR;
                }
                ptrDataCache->ulFileCacheSector = ptr_utFile->public_file_location.ulSector; // new valid cached sector
            }
            else {
                ptrDataCache->ucFileCacheState &= ~(FILE_BUFFER_VALID);
            }
        }
    }
    else if (iNoLoad == 0) {                                             // if we have no valid data in the cache we load it now if the load operation is not blocked
        if (utReadDiskSector(ptr_utDisk, ptr_utFile->public_file_location.ulSector, ptrDataCache->ucFileDataCache) != UTFAT_SUCCESS) { // read complete sector directly to buffer
            return UTFAT_DISK_READ_ERROR;
        }
        ptrDataCache->ulFileCacheSector = ptr_utFile->public_file_location.ulSector; // save the sector that the cached data reflects
        ptrDataCache->ucFileCacheState |= FILE_BUFFER_VALID;             // mark that the cache now holds valid data from this sector
    }
    return UTFAT_SUCCESS;
}
#endif

// Read linear file data content directly to an external buffer
//
extern int utReadFile(UTFILE *ptr_utFile, void *ptrBuffer, unsigned short usReadLength)
{
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
    unsigned long ulContentRemaining;
    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
    FILE_DATA_CACHE *ptrDataCache;
    #endif
    unsigned short usAccessLength;
    unsigned short usAccessOffset;

    ptr_utFile->usLastReadWriteLength = 0;
    if (!(ptr_utFile->ulFileMode & UTFAT_OPEN_FOR_READ)) {
        return UTFAT_FILE_NOT_READABLE;                                  // don't allow read of files not opened for read
    }
    ulContentRemaining = (ptr_utFile->ulFileSize - ptr_utFile->ulFilePosition); // the amount of data until the end of the file
    if (ulContentRemaining == 0) {                                       // can't read more data form the end of the file
        return UTFAT_SUCCESS;
    }
    if (usReadLength > ulContentRemaining) {                             // if the user is attempting to read more data than is in the file
        usReadLength = (unsigned short)ulContentRemaining;               // limit to remaining size in file
    }

    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
    if ((ptr_utFile->ulFileMode & UTFAT_WITH_DATA_CACHE) && (ptr_utFile->ptrFileDataCache == 0)) { // use a file data buffer if one is available in the pool
        ptr_utFile->ptrFileDataCache = fnGetDataCache();                 // try to get data cache to work with
    }
    ptrDataCache = ptr_utFile->ptrFileDataCache;
    #endif

    while (usReadLength != 0) {                                          // while requested length of data has not yet been returned
        usAccessOffset = (unsigned short)(ptr_utFile->ulFilePosition & 0x1ff);
        usAccessLength = (512 - usAccessOffset);
        if (usAccessLength > usReadLength) {
            usAccessLength = usReadLength;
        }
    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
        if (ptrDataCache != 0) {                                         // if a data cache is being used by this file
            int iReturnValue = fnHandleFileDataCache(ptr_utDisk, ptr_utFile, ptrDataCache, 0); // handle cache - load data to cache if necessary and save any cached data that may require saving
            if (iReturnValue != UTFAT_SUCCESS) {
                return iReturnValue;                                     // access error
            }
            uMemcpy(ptrBuffer, &ptrDataCache->ucFileDataCache[usAccessOffset], usAccessLength); // return the cached data
        }
        else {                                                           // the caller doesn't use a data cache
        #if defined UTFAT_FILE_CACHE_POOL && (UTFAT_FILE_CACHE_POOL > 0) && (UTMANAGED_FILE_COUNT > 0) // check whether another user has written file changes to its data cache which need to be used instead of the data stored on the disk
            if (fnGetManagedFileCache(ptr_utFile->public_file_location.ulSector, ptrBuffer, usAccessOffset, usAccessLength) == 0) { // attempt to get the data form a managed file's data cache
        #endif
    #endif
                if ((usAccessOffset != 0) || (usAccessLength != 512)) {  // only load partical data if a complete sector can not be read directly to the user's buffer
                    if (fnLoadPartialData(ptr_utDisk, ptr_utFile->public_file_location.ulSector, (unsigned char *)ptrBuffer, usAccessOffset, usAccessLength) != UTFAT_SUCCESS) { // read directly to buffer
                        return UTFAT_DISK_READ_ERROR;
                    }
                }
                else {                                                   // load a complete sector directly to the user's buffer
                    if (utReadDiskSector(ptr_utDisk, ptr_utFile->public_file_location.ulSector, (unsigned char *)ptrBuffer) != UTFAT_SUCCESS) { // read complete sector directly to buffer
                        return UTFAT_DISK_READ_ERROR;
                    }
                }
    #if defined UTFAT_FILE_CACHE_POOL && (UTFAT_FILE_CACHE_POOL > 0) && (UTMANAGED_FILE_COUNT > 0)
            }
    #endif
    #if defined UTFAT_FILE_CACHE_POOL && (UTFAT_FILE_CACHE_POOL > 0)
        }
    #endif
        usReadLength -= usAccessLength;
        ptr_utFile->ulFilePosition += usAccessLength;
        ptr_utFile->usLastReadWriteLength += usAccessLength;
        ptrBuffer = (void *)((CAST_POINTER_ARITHMETIC)ptrBuffer + usAccessLength);
        if ((ptr_utFile->ulFilePosition % ptr_utDisk->utFAT.usBytesPerSector) == 0) { // {13} on a sector boundary
            int iResult = fnNextSector(ptr_utDisk, &ptr_utFile->public_file_location);
            if (iResult != UTFAT_SUCCESS) {
                return iResult;
            }
        }
    }
    return UTFAT_SUCCESS;
}


// Open a list directory referenced to its main directory object, which must already have been configured
//
extern int utLocateDirectory(const CHAR *ptrDirPath, UTLISTDIRECTORY *ptrListDirectory)
{
    int iReturn = UTFAT_SUCCESS;
    if (ptrListDirectory->ptr_utDirObject == 0) {
        return UTFAT_DIRECTORY_OBJECT_MISSING;
    }
    if ((iReturn = utOpenDirectory(ptrDirPath, ptrListDirectory->ptr_utDirObject)) >= UTFAT_SUCCESS) {
        uMemcpy(&ptrListDirectory->private_disk_location, &ptrListDirectory->ptr_utDirObject->public_disk_location, sizeof(ptrListDirectory->private_disk_location)); // copy the referenced directory details to the list directory object
    }
    return iReturn;
}

// This routine sets a fixed time and data - it can be replaced by a routine to set the information from a RTC or similar
//
#define CREATION_HOURS   12
#define CREATION_MINUTES 00
#define CREATION_SECONDS 00

#define CREATION_DAY_OF_MONTH  11
#define CREATION_MONTH_OF_YEAR 7
#define CREATION_YEAR    (2014 - 1980)

#if defined UTFAT_WRITE
static void fnSetTimeDate(DIR_ENTRY_STRUCTURE_FAT32 *ptrEntry, int iCreation)
{
    unsigned short usCreationTime;
    unsigned short usCreationDate;
    #if defined SUPPORT_FILE_TIME_STAMP
    if (fnGetLocalFileTime(&usCreationTime, &usCreationDate) != 0) {
        usCreationTime = (CREATION_SECONDS | (CREATION_MINUTES << 5) | (CREATION_HOURS << 11));
        usCreationDate = (CREATION_DAY_OF_MONTH | (CREATION_MONTH_OF_YEAR << 5) | (CREATION_YEAR << 9));
    }
    #else
    usCreationTime = (CREATION_SECONDS | (CREATION_MINUTES << 5) | (CREATION_HOURS << 11));
    usCreationDate = (CREATION_DAY_OF_MONTH | (CREATION_MONTH_OF_YEAR << 5) | (CREATION_YEAR << 9));
    #endif
    ptrEntry->DIR_WrtTime[0] = (unsigned char)(usCreationTime);
    ptrEntry->DIR_WrtTime[1] = (unsigned char)(usCreationTime >> 8);
    ptrEntry->DIR_LstAccDate[0] = ptrEntry->DIR_WrtDate[0] = (unsigned char)(usCreationDate);
    ptrEntry->DIR_LstAccDate[1] = ptrEntry->DIR_WrtDate[1] = (unsigned char)(usCreationDate >> 8);
    if (iCreation != 0) {
        ptrEntry->DIR_CrtTime[0] = ptrEntry->DIR_WrtTime[0];
        ptrEntry->DIR_CrtTime[1] = ptrEntry->DIR_WrtTime[1];
        ptrEntry->DIR_CrtDate[0] = ptrEntry->DIR_LstAccDate[0];
        ptrEntry->DIR_CrtDate[1] = ptrEntry->DIR_LstAccDate[1];
    }
}

static void fnAddEntry(DIR_ENTRY_STRUCTURE_FAT32 *ptrEntry, unsigned long ulPresentCluster, unsigned char ucAttributes)
{
    ptrEntry->DIR_FstClusLO[0] = (unsigned char)ulPresentCluster;
    ptrEntry->DIR_FstClusLO[1] = (unsigned char)(ulPresentCluster >> 8);
    ptrEntry->DIR_FstClusHI[0] = (unsigned char)(ulPresentCluster >> 16);
    ptrEntry->DIR_FstClusHI[1] = (unsigned char)(ulPresentCluster >> 24);
    ptrEntry->DIR_Attr = ucAttributes;
    uMemset(ptrEntry->DIR_FileSize, 0, sizeof(ptrEntry->DIR_FileSize));  // set the file size to zero
    fnSetTimeDate(ptrEntry, 1);                                          // add creation time
}

static void fnSetFileInformation(DIR_ENTRY_STRUCTURE_FAT32 *ptrFileEntry, unsigned long ulFileSize)
{
    ptrFileEntry->DIR_FileSize[0] = (unsigned char)(ulFileSize);         // update the file size
    ptrFileEntry->DIR_FileSize[1] = (unsigned char)(ulFileSize >> 8);
    ptrFileEntry->DIR_FileSize[2] = (unsigned char)(ulFileSize >> 16);
    ptrFileEntry->DIR_FileSize[3] = (unsigned char)(ulFileSize >> 24);
    fnSetTimeDate(ptrFileEntry, 0);                                      // set the modification time
}

static int fnCommitFileInfo(UTFILE *ptr_utFile, UTDISK *ptr_utDisk)
{
    DIR_ENTRY_STRUCTURE_FAT32 *ptrFileEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
    ptrFileEntry += ptr_utFile->private_disk_location.ucDirectoryEntry;  // move to the file entry
    if (fnLoadSector(ptr_utDisk, ptr_utFile->private_disk_location.directory_location.ulSector) != UTFAT_SUCCESS) { // ensure that the directory sector is loaded
        return UTFAT_DISK_READ_ERROR;
    }
    fnSetFileInformation(ptrFileEntry, ptr_utFile->ulFileSize);
    while (utCommitSector(ptr_utDisk, ptr_utFile->private_disk_location.directory_location.ulSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
    ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;                   // the disk is up to date with the buffer
    return UTFAT_SUCCESS;
}

// Allocate a single new cluster
//
static unsigned long fnAllocateCluster(UTDISK *ptr_utDisk, unsigned long ulPresentCluster, unsigned char ucClusterType)
{
    unsigned long  ulFAT = ptr_utDisk->utFAT.ulFAT_start;                // sector in which the file's first cluster is located in
    unsigned long  ulFatOriginal;
    unsigned long  ulSectorContent[512/sizeof(signed long)];             // sector to read FAT32 sectors to
    unsigned long  ulAbsoluteCluster;
    unsigned char  ucClusterMax;
    unsigned char  ucClusterEntry;
    unsigned char  ucOriginalCluster;
    if (ucClusterType & NEW_RELATIVE_CLUSTER) {
        ulAbsoluteCluster = ulPresentCluster;
    }
    else {
        if ((ptr_utDisk->usDiskFlags & FSINFO_VALID) && (ptr_utDisk->utFileInfo.ulNextFreeCluster <= ptr_utDisk->utFileInfo.ulFreeClusterCount)) { // the info block is valid so reference to next free cluster
            ulAbsoluteCluster = ptr_utDisk->utFileInfo.ulNextFreeCluster;
        }
        else {
            ulAbsoluteCluster = ptr_utDisk->ulDirectoryBase;
        }
        ulPresentCluster = ulAbsoluteCluster;
    }
    ucClusterEntry = (unsigned char)ulPresentCluster;
    #if defined UTFAT16
    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
        ulFAT += (ulPresentCluster/(512/sizeof(signed short)));
        ucClusterMax = ((512/sizeof(unsigned short)) - 1);
    }
    else {
        ulFAT += (ulPresentCluster/(512/sizeof(signed long)));
        ucClusterEntry &= ((512/sizeof(signed long)) - 1);
        ucClusterMax = ((512/sizeof(unsigned long)) - 1);
    }
    #else
    ulFAT += (ulPresentCluster/(512/sizeof(signed long)));
    ucClusterEntry &= ((512/sizeof(signed long)) - 1);
    ucClusterMax = ((512/sizeof(unsigned long)) - 1);
    #endif
    ulFatOriginal = ulFAT;
    ucOriginalCluster = ucClusterEntry;
    while (1) {
        if ((utReadDiskSector(ptr_utDisk, ulFAT, ulSectorContent)) != UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
            return (unsigned long)UTFAT_DISK_READ_ERROR;
        }
        while (1) {
            unsigned long ulClusterEntryContent;
    #if defined UTFAT16
            if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                ulClusterEntryContent = ulSectorContent[ucClusterEntry/2];
                if (ucClusterEntry & 1) {
                    ulClusterEntryContent &= LITTLE_LONG_WORD(0xffff0000);
                }
                else {
                    ulClusterEntryContent &= LITTLE_LONG_WORD(0x0000ffff);
                }
            }
            else {
                ulClusterEntryContent = ulSectorContent[ucClusterEntry];
            }
    #else
            ulClusterEntryContent = ulSectorContent[ucClusterEntry];
    #endif
            if (ulClusterEntryContent == 0) {                            // next free cluster location found
                unsigned char ucFatCopies = 0;
                if (ucClusterType & (INITIALISE_DIR_CLUSTER | INITIALISE_DIR_EXTENSION)) { // the new cluster must be configured as empty directory
                    unsigned char ucSectors = ptr_utDisk->utFAT.ucSectorsPerCluster;
                    unsigned long ulSectorToDelete = ((ulAbsoluteCluster * ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulVirtualBaseAddress);
                    while (ucSectors != 0) {
                        if ((ucSectors-- == ptr_utDisk->utFAT.ucSectorsPerCluster) && (ucClusterType & INITIALISE_DIR_CLUSTER)) { // first sector - add "." and ".." entries
                            unsigned char ucDirectoryDefault[512];
                            DIR_ENTRY_STRUCTURE_FAT32 *ptrEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ucDirectoryDefault;
                            uMemset(ucDirectoryDefault, 0x00, sizeof(ucDirectoryDefault));
                            uMemset(ptrEntry->DIR_Name, ' ', sizeof(ptrEntry->DIR_Name));
                            ptrEntry->DIR_Name[0] = '.';
                            fnAddEntry(ptrEntry, ulPresentCluster, DIR_ATTR_DIRECTORY);
                            ptrEntry++;
                            uMemset(ptrEntry->DIR_Name, ' ', sizeof(ptrEntry->DIR_Name));
                            ptrEntry->DIR_Name[0] = '.';
                            ptrEntry->DIR_Name[1] = '.';
                            fnAddEntry(ptrEntry, 2, DIR_ATTR_DIRECTORY);
                            while (utCommitSectorData(ptr_utDisk, ucDirectoryDefault, ulSectorToDelete) == CARD_BUSY_WAIT) {}
                        }
                        else {
                            while (utDeleteSector(ptr_utDisk, ulSectorToDelete) == CARD_BUSY_WAIT) {} // delete sector
                        }
                        ulSectorToDelete++;
                    }
                }
    #if defined UTFAT16
                if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                    if (ucClusterEntry & 0x01) {
                        ulSectorContent[ucClusterEntry/2] |= (LITTLE_LONG_WORD(FAT16_CLUSTER_MASK << 16)); // mark last cluster in extension
                    }
                    else {
                        ulSectorContent[ucClusterEntry/2] |= (LITTLE_LONG_WORD(FAT16_CLUSTER_MASK)); // mark last cluster in extension
                    }
                }
                else {                                                   // else FAT32
                    ulSectorContent[ucClusterEntry] = LITTLE_LONG_WORD(CLUSTER_MASK); // mark last cluster in extension
                }
    #else
                ulSectorContent[ucClusterEntry] = LITTLE_LONG_WORD(CLUSTER_MASK); // mark last cluster in extension
    #endif
                if (ucClusterType & UPDATE_FAT_END) {
                    ucClusterType = 0;
    #if defined UTFAT16
                    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                        if (ucClusterEntry & 0x01) {
                            ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0x0000ffff);
                            ulSectorContent[ucClusterEntry/2] |= LITTLE_LONG_WORD(ulAbsoluteCluster); // mark last cluster in extension
                        }
                        else {
                            ucClusterEntry -= 2;                         // {38}
                            ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0xffff0000);
                            ulSectorContent[ucClusterEntry/2] |= LITTLE_LONG_WORD(ulAbsoluteCluster << 16); // mark last cluster in extension
                        }
                    }
                    else {
                        ulSectorContent[ucOriginalCluster] = LITTLE_LONG_WORD(ulAbsoluteCluster); // modify the original end marker to point to the additional cluster
                    }
    #else
                    ulSectorContent[ucOriginalCluster] = LITTLE_LONG_WORD(ulAbsoluteCluster); // modify the original end marker to point to the additional cluster
    #endif
                }
                while (ucFatCopies < ptr_utDisk->utFAT.ucNumberOfFATs) {
                    while (utCommitSectorData(ptr_utDisk, ulSectorContent, (ulFAT + (ucFatCopies * ptr_utDisk->utFAT.ulFatSize))) == CARD_BUSY_WAIT) {} // write the new FAT entry
                    ucFatCopies++;
                }
                if (ucClusterType & UPDATE_FAT_END_IN_DIFF_SECT) {       // the new cluster end has been marked but the original end must be modified to point to it (it is in a different sector so needs to be modified seperately)
                    ucFatCopies = 0;
                    if ((utReadDiskSector(ptr_utDisk, ulFatOriginal, ulSectorContent)) != UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
                        return (unsigned long)UTFAT_DISK_READ_ERROR;
                    }
    #if defined UTFAT16
                    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                        if (ucClusterEntry & 0x01) {
                            ulSectorContent[((unsigned char)ulPresentCluster/2)] &= ~LITTLE_LONG_WORD(0x0000ffff); // unsigned char casting before divide by 2!!
                            ulSectorContent[((unsigned char)ulPresentCluster/2)] |= LITTLE_LONG_WORD(ulAbsoluteCluster); // mark last cluster in extension
                        }
                        else {
                            ulSectorContent[((unsigned char)ulPresentCluster/2)] &= ~LITTLE_LONG_WORD(0xffff0000);
                            ulSectorContent[((unsigned char)ulPresentCluster/2)] |= LITTLE_LONG_WORD(ulAbsoluteCluster << 16); // mark last cluster in extension
                        }
                    }
                    else {
                        ulSectorContent[ulPresentCluster & ((512/sizeof(signed long)) - 1)] = LITTLE_LONG_WORD(ulAbsoluteCluster);
                    }
    #else
                    ulSectorContent[ulPresentCluster & ((512/sizeof(signed long)) - 1)] = LITTLE_LONG_WORD(ulAbsoluteCluster);
    #endif
                    while (ucFatCopies < ptr_utDisk->utFAT.ucNumberOfFATs) {
                        while (utCommitSectorData(ptr_utDisk, ulSectorContent, (ulFatOriginal + (ucFatCopies * ptr_utDisk->utFAT.ulFatSize))) == CARD_BUSY_WAIT) {} // write the new FAT entry
                        ucFatCopies++;
                    }
                }
                if (ptr_utDisk->usDiskFlags & FSINFO_VALID) {
                    if (ulAbsoluteCluster >= ptr_utDisk->utFileInfo.ulNextFreeCluster) {                        
                        ptr_utDisk->utFileInfo.ulNextFreeCluster = (ulAbsoluteCluster + 1);
                    }
                    ptr_utDisk->usDiskFlags |= WRITEBACK_INFO_FLAG;      // mark that the info block information has changed and should be committed at some point
                    ptr_utDisk->utFileInfo.ulFreeClusterCount--;         // since we have allocated a new cluster there is one less free
                }
                return (ulAbsoluteCluster);
            }
            ulAbsoluteCluster++;
            if (ucClusterEntry >= ucClusterMax) {
                break;
            }
            ucClusterEntry++;
        }
        ucClusterType &= ~UPDATE_FAT_END;                                // the new FAT entry is in a different FAT sector so needs to be modified later
        ucClusterEntry = 0;
        ulFAT++;
    }
}

static void fnAddInfoSect(INFO_SECTOR_FAT32 *ptrInfoSector, unsigned long ulFreeCount, unsigned long ulNextFree)
{
    ptrInfoSector->FSI_Free_Count[0] = (unsigned char)(ulFreeCount);
    ptrInfoSector->FSI_Free_Count[1] = (unsigned char)(ulFreeCount >> 8);
    ptrInfoSector->FSI_Free_Count[2] = (unsigned char)(ulFreeCount >> 16);
    ptrInfoSector->FSI_Free_Count[3] = (unsigned char)(ulFreeCount >> 24);
    ptrInfoSector->FSI_Nxt_Free[0]   = (unsigned char)(ulNextFree);
    ptrInfoSector->FSI_Nxt_Free[1]   = (unsigned char)(ulNextFree >> 8);
    ptrInfoSector->FSI_Nxt_Free[2]   = (unsigned char)(ulNextFree >> 16);
    ptrInfoSector->FSI_Nxt_Free[3]   = (unsigned char)(ulNextFree >> 24);
}

static int fnCommitInfoChanges(UTDISK *ptr_utDisk)
{
    if (ptr_utDisk->usDiskFlags & WRITEBACK_INFO_FLAG) {                 // info sector content has changed
        INFO_SECTOR_FAT32 info_Sector;
        unsigned long ulInfoSector_Location = ptr_utDisk->utFileInfo.ulInfoSector;
        if ((utReadDiskSector(ptr_utDisk, ulInfoSector_Location, &info_Sector)) != UTFAT_SUCCESS) { // read the information sector
            return UTFAT_DISK_READ_ERROR;
        }
        fnAddInfoSect(&info_Sector, ptr_utDisk->utFileInfo.ulFreeClusterCount, ptr_utDisk->utFileInfo.ulNextFreeCluster);
        while (utCommitSectorData(ptr_utDisk, &info_Sector, ulInfoSector_Location) == CARD_BUSY_WAIT) {}
        ptr_utDisk->usDiskFlags &= ~WRITEBACK_INFO_FLAG;
    }
    return UTFAT_SUCCESS;
}
#endif

#if defined UTFAT_LFN_READ && defined UTFAT_LFN_WRITE
// Verify that a long file name entry is possible or else move to a location that is possible, which may be the end of the present directory
//
static int fnInsertLFN_name(OPEN_FILE_BLOCK *ptr_openBlock, UTFILE *ptr_utFile, const CHAR *ptrLongFileName, int iRename)
{
    DIR_ENTRY_STRUCTURE_FAT32 original_file_object;
    DIR_ENTRY_STRUCTURE_FAT32 *ptrDirectoryEntry;
    LFN_ENTRY_STRUCTURE_FAT32 *ptrLongFileEntry = 0;
    UTDISK *ptr_utDisk = ptr_openBlock->ptr_utDisk;
    DISK_LOCATION *ptrDiskLocation = ptr_openBlock->ptrDiskLocation;
    const CHAR *ptrShortFileName = ptr_openBlock->ptrLocalDirPath;
    int i;
    int iFileObjectMoved = 0;
    int iNameLength = uStrlen(ptrLongFileName);                          // length of long file name that must be saved
    unsigned char ucEntryLength = 1;                                     // smallest size that can be required by a LFN (SFN will be required after it too)
    const CHAR *ptrReverseLFN = (ptrLongFileName + iNameLength);         // set the LFN name pointer to the end of the string (terminator) since it is going to be copied in reverse order
    unsigned char ucNextCharacter = 0xff;                                // default is pad character
    unsigned char ucNextCharacterExtension = 0xff;
    unsigned char ucSFN_alias_checksum;
    #if defined UTFAT_LFN_WRITE_PATCH
    fnCreateInvalidSFN_alias(ptr_openBlock->cShortFileName);             // create invalid SFN entry that avoids FAT32 LFN patent issues, doesn't allow compatibility with non-LFN systems and has lowest probability to cause Windows XP or chkdsk problems
    #else
    fnCreateSFN_alias(openBlock.cShortFileName);                         // create short file name according to true FAT32 LFN specification
    #endif
    ucSFN_alias_checksum = fnLFN_checksum(ptr_openBlock->cShortFileName);// calculate the short file name alias's checksum

    iNameLength++;                                                       // include the string terminator in the length
    while (iNameLength > 13) {                                           // calculate the number of entries required to store the long file name in
        ucEntryLength++;
        iNameLength -= 13;                                               // subtract the part of the name that can fit in each single entry
    }
    i = ucEntryLength;                                                   // we check whether there is a known hole in the directory for reuse
    if ((iRename != 0) && (ptr_utFile->lfn_file_location.directory_location.ulSector != 0) && (ptr_utFile->ucLFN_entries >= ucEntryLength)) { // renamed file is LFN and so its space can be used if large enough
        uMemcpy(ptrDiskLocation, &ptr_utFile->lfn_file_location, sizeof(DISK_LOCATION)); // set the start of the original long file name
        if (ptr_utFile->ucLFN_entries != ucEntryLength) {                // if a location past the original start is to be used move to it
            ptr_utFile->ucLFN_entries -= ucEntryLength;
            while (ptr_utFile->ucLFN_entries != 0) {
                fnNextDirectoryEntry(ptr_utDisk, ptrDiskLocation);
                ptr_utFile->ucLFN_entries--;
            }
            ptr_utFile->ucLFN_entries = ucEntryLength;
            uMemcpy(&ptr_utFile->lfn_file_location, ptrDiskLocation, sizeof(DISK_LOCATION));
        }
    }
    else {
        while (i < DELETED_ENTRY_COUNT) {                                // make use of existing deleted area in the directory (starting with a hole of exactly the size required)
            if (ptr_openBlock->DeleteLocation[i].directory_location.ulSector != 0) { // deleted space of adequate size found
                break;                                                   // we have a new reference to the location of the deleted entries
            }
            i++;                                                         // try larger holes
        }
        if (iRename != 0) {                                              // the renamed file must be relocated (original details transferred then destroyed)
            if (fnLoadSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) != UTFAT_SUCCESS) {
                return UTFAT_DISK_READ_ERROR;
            }
            ptrDirectoryEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
            ptrDirectoryEntry += ptrDiskLocation->ucDirectoryEntry;      // move to the present entry
            uMemcpy(&original_file_object, ptrDirectoryEntry, sizeof(original_file_object)); // backup the origional file object
            ptrDirectoryEntry->DIR_Name[0] = DIR_NAME_FREE;              // free the original object
            ptr_utDisk->usDiskFlags |= WRITEBACK_BUFFER_FLAG;            // mark that the original sector content must be committed
            iFileObjectMoved = 1;
        }
        if (i >= DELETED_ENTRY_COUNT) {                                  // no reuse is possible in the existing directory area so the renamed file must be moved to the end of the present directory
            uMemcpy(ptrDiskLocation, &ptr_openBlock->DirectoryEndLocation, sizeof(DISK_LOCATION)); // move to the new LFN location which is extending the directory
        }
        else {
            uMemcpy(ptrDiskLocation, &ptr_openBlock->DeleteLocation[i], sizeof(DISK_LOCATION)); // move to the new LFN location which is reusing a deleted area
        }
    }
    if (fnLoadSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) != UTFAT_SUCCESS) { // this is either reusing deleted entry space or creating a new name at the end of the directory
        return UTFAT_DISK_READ_ERROR;
    }
    ptrDirectoryEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
    ptrDirectoryEntry += ptrDiskLocation->ucDirectoryEntry;              // move to the present entry

    ucEntryLength |= 0x40;                                               // mark first entry
    i = -1;
    while (1) {                                                          // for each character in the LFN
        if (iNameLength >= 13) {                                         // pad end when the LFN doesn't fill an entry
            ucNextCharacter = *ptrReverseLFN;
            ucNextCharacterExtension = 0;                                // only english character set supported
        }
        switch (i--) {
        case 0:                                                          // move to next entry
            {
                int iResult = fnNextDirectoryEntry(ptr_utDisk, ptrDiskLocation);
                if (UTFAT_DIRECTORY_AREA_EXHAUSTED == iResult) {         // the present directory cluster end has been reached so a new one must be started
                    ptrDiskLocation->directory_location.ulSector--;      // set back to previous sector
                    iResult = fnDirectorySectorCreate(ptr_utDisk, &ptrDiskLocation->directory_location); // create additional directory cluster
                }
                if (iResult != UTFAT_SUCCESS) {
                    return iResult;                                      // return error
                }
                if (fnLoadSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) != UTFAT_SUCCESS) {
                    return UTFAT_DISK_READ_ERROR;
                }
                ptrDirectoryEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
                ptrDirectoryEntry += ptrDiskLocation->ucDirectoryEntry;  // move to the present entry
                if (ptrReverseLFN < ptrLongFileName) {                   // the complete LFN has been added
                    if (iFileObjectMoved != 0) {
                        uMemcpy(ptrDirectoryEntry, &original_file_object, sizeof(original_file_object)); // restore the original file object
                    }
                    return UTFAT_SUCCESS;
                }
            }
            // Fall through intentional
            //
        case -1:
            ptrLongFileEntry = (LFN_ENTRY_STRUCTURE_FAT32 *)ptrDirectoryEntry;
            uMemset(ptrLongFileEntry, 0x00, sizeof(LFN_ENTRY_STRUCTURE_FAT32)); // ensure initially zeroed
            ptrLongFileEntry->LFN_EntryNumber = ucEntryLength;           // entry number
            ucEntryLength &= ~0x40;
            ucEntryLength--;
            ptrLongFileEntry->LFN_Attribute = 0x0f;                      // mark as hidden system file so that SFN systems will ignore the entry
            ptrLongFileEntry->LFN_Checksum = ucSFN_alias_checksum;       // the SFN alias checksum is inserted in each entry
            i = 31;
            continue;
        case 1:
            ptrLongFileEntry->LFN_Name_0 = ucNextCharacter;
            ptr_utDisk->usDiskFlags |= WRITEBACK_BUFFER_FLAG;            // mark that the original sector content must be committed
            break;
        case 2:
            ptrLongFileEntry->LFN_Name_0_extension = ucNextCharacterExtension;
            continue;
        case 3:
            ptrLongFileEntry->LFN_Name_1 = ucNextCharacter;
            break;
        case 4:
            ptrLongFileEntry->LFN_Name_1_extension = ucNextCharacterExtension;
            continue;
        case 5:
            ptrLongFileEntry->LFN_Name_2 = ucNextCharacter;
            break;
        case 6:
            ptrLongFileEntry->LFN_Name_2_extension = ucNextCharacterExtension;
            continue;
        case 7:
            ptrLongFileEntry->LFN_Name_3 = ucNextCharacter;
            break;
        case 8:
            ptrLongFileEntry->LFN_Name_3_extension = ucNextCharacterExtension;
            continue;
        case 9:
            ptrLongFileEntry->LFN_Name_4 = ucNextCharacter;
            break;
        case 10:
            ptrLongFileEntry->LFN_Name_4_extension = ucNextCharacterExtension;
            continue;
      //case 11:                                                         // attribute (added during initialisation)
      //case 12:                                                         // zero (zeroed during initialisation)
      //case 13:                                                         // checksum (added during initialisation)
     //     continue;
        case 14:
            ptrLongFileEntry->LFN_Name_5 = ucNextCharacter;
            i -= 3;                                                      // skip 11, 12, and 13
            break;
        case 15:
            ptrLongFileEntry->LFN_Name_5_extension = ucNextCharacterExtension;
            continue;
        case 16:
            ptrLongFileEntry->LFN_Name_6 = ucNextCharacter;
            break;
        case 17:
            ptrLongFileEntry->LFN_Name_6_extension = ucNextCharacterExtension;
            continue;
        case 18:
            ptrLongFileEntry->LFN_Name_7 = ucNextCharacter;
            break;
        case 19:
            ptrLongFileEntry->LFN_Name_7_extension = ucNextCharacterExtension;
            continue;
        case 20:
            ptrLongFileEntry->LFN_Name_8 = ucNextCharacter;
            break;
        case 21:
            ptrLongFileEntry->LFN_Name_8_extension = ucNextCharacterExtension;
            continue;
        case 22:
            ptrLongFileEntry->LFN_Name_9 = ucNextCharacter;
            break;
        case 23:
            ptrLongFileEntry->LFN_Name_9_extension = ucNextCharacterExtension;
            continue;
        case 24:
            ptrLongFileEntry->LFN_Name_10 = ucNextCharacter;
            break;
        case 25:
            ptrLongFileEntry->LFN_Name_10_extension = ucNextCharacterExtension;
            continue;
      //case 26:                                                         // zeros (zeroed during initialisation)
      //case 27:
      //    continue;
        case 28:
            ptrLongFileEntry->LFN_Name_11 = ucNextCharacter;
            i -= 2;                                                      // skip 26 and 27
            break;
        case 29:
            ptrLongFileEntry->LFN_Name_11_extension = ucNextCharacterExtension;
            continue;
        case 30:
            ptrLongFileEntry->LFN_Name_12 = ucNextCharacter;
            break;
        case 31:
            ptrLongFileEntry->LFN_Name_12_extension = ucNextCharacterExtension;
            continue;
        }
        if (iNameLength >= 13) {                                         // name character was added
            ptrReverseLFN--;
        }
        else {
            iNameLength++;                                               // pad was added
        }
    }
    return UTFAT_SUCCESS;
}
#endif

#if defined UTFAT_WRITE

// The new file or its new name will be located either at the end of the present directory or else reuse deleted areas of adequate size
//
static int fnSetFileLocation(UTFILE *ptr_utFile, OPEN_FILE_BLOCK *ptr_openBlock, const CHAR *ptrFilePath, int iRename)
{
    int iReturn = UTFAT_SUCCESS;
    #if defined UTFAT_LFN_READ && (defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE)
    if (iRename != 0) {                                                  // if renaming
        iReturn = fnDeleteLFN_entry(ptr_utFile);                         // delete any original LFN directory entry (short file names can be simply overwritten)
        if (iReturn != UTFAT_SUCCESS) {
            return iReturn;
        }
    }
        #if defined UTFAT_LFN_WRITE
    if (ptr_openBlock->iQualifiedPathType >= FULLY_QUALIFIED_LONG_NAME_SFNM) { // the new file requires a long file name and cannot use a short one
        return fnInsertLFN_name(ptr_openBlock, ptr_utFile, ptrFilePath, iRename);
    }
        #endif
    #endif
    // Creating a short file name - this is set at the end of the present directory and doesn't reuse deleted space
    //
    return iReturn;
}

// Create or rename a file
//
static int fnCreateFile(OPEN_FILE_BLOCK *ptr_openBlock, UTFILE *ptr_utFile, const CHAR *ptrFilePath, unsigned long ulAccessMode)
{
    int iReturn;
    int iRename = ((ulAccessMode & _RENAME_EXISTING) != 0);              // check whether renaming existing file
    unsigned long ulFreeCluster = 0;
    DISK_LOCATION *ptrDiskLocation;                                      // this is the start of the directory in which the new file is to be placed - this allows reuse of deleted file locations in the directory
    DIR_ENTRY_STRUCTURE_FAT32 *ptrFoundEntry;
    UTDISK *ptr_utDisk = ptr_openBlock->ptr_utDisk;
    iReturn = fnSetFileLocation(ptr_utFile, ptr_openBlock, ptrFilePath, iRename); // get the location in the present directory where the (SFN or SFN alias of) new/renamed file will be located
    if (iReturn != UTFAT_SUCCESS) {
        return iReturn;                                                  // presumed error
    }
    ptrDiskLocation = ptr_openBlock->ptrDiskLocation;                    // this is location in the directory where the new/renamed file (SFN or SFN alias) is to be placed
    if (fnLoadSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) != UTFAT_SUCCESS) {
        return UTFAT_DISK_READ_ERROR;
    }
    ptrFoundEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
    ptrFoundEntry += ptrDiskLocation->ucDirectoryEntry;                  // move to the present directory entry

    uMemcpy(ptrFoundEntry->DIR_Name, ptr_openBlock->cShortFileName, 11); // add the short file name
    ptrFoundEntry->DIR_NTRes = ptr_openBlock->cShortFileName[11];
    if (iRename == 0) {                                                  // if creating and not renaming
        if (ptr_utFile == 0) {                                           // directory and not file
            ulFreeCluster = fnAllocateCluster(ptr_utDisk, ptrDiskLocation->directory_location.ulCluster, INITIALISE_DIR_CLUSTER);
            fnAddEntry(ptrFoundEntry, ulFreeCluster, DIR_ATTR_DIRECTORY);
        }
        else {
            ulFreeCluster = fnAllocateCluster(ptr_utDisk, 0, 0);         // allocate a cluster for the new file
            fnAddEntry(ptrFoundEntry, ulFreeCluster, DIR_ATTR_ARCHIVE);
        }
    }
    while (utCommitSector(ptr_utDisk, ptrDiskLocation->directory_location.ulSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
    ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;                   // the disk is up to date with the buffer
    if (iRename == 0) {                                                  // if creating and not renaming
        if (ptr_utFile != 0) {                                           // file and not directory
            uMemcpy(&ptr_utFile->private_disk_location, &ptr_utFile->ptr_utDirObject->public_disk_location, sizeof(ptr_utFile->private_disk_location)); // copy the referenced directory details
            ptr_utFile->private_file_location.ulCluster = ulFreeCluster;
            ptr_utFile->private_file_location.ulSector = (ulFreeCluster * ptr_utDisk->utFAT.ucSectorsPerCluster); // section referenced to logical base address
            ptr_utFile->private_file_location.ulSector += ptr_utDisk->ulVirtualBaseAddress; // the sector in which the file content begins
            uMemcpy(&ptr_utFile->public_file_location, &ptr_utFile->private_file_location, sizeof(ptr_utFile->private_file_location)); // copy the referenced file start details
            ptr_utFile->ulFileMode = ulAccessMode;
            ptr_utFile->ulFileSize = 0;
            ptr_utFile->ulFilePosition = 0;                              // set position to start of file on open
            ptr_utFile->ucDrive = ptr_utFile->ptr_utDirObject->ucDrive;
            iReturn = UTFAT_PATH_IS_FILE;
        }
    }
    fnCommitInfoChanges(ptr_utDisk);
    return iReturn;
}
#endif

#if defined UTFAT_EXPERT_FUNCTIONS
static int fnDisplayLFN(DISK_LOCATION *ptrFileLocation, UTDISK *ptr_utDisk)
{
    DISK_LOCATION dirLocation;
    LFN_ENTRY_STRUCTURE_FAT32 *ptrLFN_entry = 0;
    unsigned char *ptrData;
    int j;
    int iCheckedStart = 0;
    unsigned char ucEntry;
    unsigned char ucEntryCheck = 0;
    unsigned char ucSFN_alias_CS;

    uMemcpy(&dirLocation, ptrFileLocation, sizeof(dirLocation));         // backup the location
    
    do {                                                                 // work through the long file name entries
        if (fnLoadSector(ptr_utDisk, dirLocation.directory_location.ulSector) != UTFAT_SUCCESS) { // load new sector (this will only read from physical disk when the sector is not already in local cache)
            return UTFAT_DISK_READ_ERROR;
        }
        ptrLFN_entry = (LFN_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
        ptrLFN_entry += dirLocation.ucDirectoryEntry;                    // move to the present entry
        ucEntry = ptrLFN_entry->LFN_EntryNumber;
        if (iCheckedStart == 0) {                                        // if the integrity of the first entry has not yet been confirmed
            if (ucEntry == DIR_NAME_FREE) {                              // analysing a deleted LFN
                fnDebugMsg("Deleted LFN\r\n");
                if ((ptrLFN_entry->LFN_Attribute & DIR_ATTR_MASK) != DIR_ATTR_LONG_NAME) {
                    fnDebugMsg("End\r\n");
                    break;
                }
                if (ucEntryCheck != 0) {
                    goto _show_data;
                }
            }
            else if (ucEntry & 0x40) {                                   // the first LFN entry must always start with this bit set
                ucEntry &= ~0x40;
                fnDebugMsg("First object from ");
                fnDebugDec(ucEntry, WITH_CR_LF);
                iCheckedStart = 1;
            }
            else {
                fnDebugMsg("First object error\r\n");                    // unexpected start - check the data content that follows!
                ucEntryCheck = 1;                                        // quit after displaying content
                goto _show_data;
            }
            ucEntryCheck = ucEntry;                                      // the number of LFN entries (will be 0xe5 for deleted entry)
            ucSFN_alias_CS = ptrLFN_entry->LFN_Checksum;                 // save the checksum for checking
        }
        else {
            if (--ucEntryCheck != ucEntry) {
                fnDebugMsg("LFN object error\r\n");                      // unexpected LFN entry counter - check the data content that follows!
                ucEntryCheck = 0;
            }
        }
_show_data:
        ptrData = (unsigned char *)ptrLFN_entry;
        fnDebugMsg("Data =");
        for (j = 0; j < 32; j++) {                                       // display the entries raw data
            fnDebugHex(*ptrData++, (WITH_LEADIN | WITH_SPACE | sizeof(unsigned char)));
        }
        fnDebugMsg("\r\n");
        if ((ptrLFN_entry->LFN_Attribute & DIR_ATTR_MASK) != DIR_ATTR_LONG_NAME) { // verify that the entry has LFN attribute
            fnDebugMsg("Invalid LFN attribute!!");
        }
        else {
            if ((ptrLFN_entry->LFN_Zero0 != 0) || (ptrLFN_entry->LFN_Zero1 != 0) || (ptrLFN_entry->LFN_Zero2 != 0)) { // perform some sanity checking of content
                fnDebugMsg("LFN content zero error - ");
                if (ptrLFN_entry->LFN_Zero0 != 0) {
                    fnDebugMsg("0");
                }   
                if (ptrLFN_entry->LFN_Zero1 != 0) {
                    fnDebugMsg("1");
                }
                if (ptrLFN_entry->LFN_Zero2 != 0) {
                    fnDebugMsg("2");
                }
                fnDebugMsg(" not zero!!\r\n");
            }
            if (ucSFN_alias_CS =! ptrLFN_entry->LFN_Checksum) {
                fnDebugMsg("SNF alias CS mis-match!!\r\n");
            }
        }
        if (ucEntryCheck == 1) {                                         // last LFN entry displayed
            break;
        }        
        if (fnNextDirectoryEntry(ptr_utDisk, &dirLocation) == UTFAT_DIRECTORY_AREA_EXHAUSTED) { // move to the next entry
            return UTFAT_DIRECTORY_AREA_EXHAUSTED;
        }
    } while (1);
    return UTFAT_SUCCESS;
}

static int fnDisplaySFN(int iFile, UTFILE *ptr_utFile, OPEN_FILE_BLOCK *ptr_openBlock)
{
    unsigned long ulCluster;
    unsigned long ulSector;
    unsigned long ulFatSector;
    DISK_LOCATION *ptrFileLocation = &ptr_utFile->private_disk_location;
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ptr_utDirObject->ucDrive];
    DIR_ENTRY_STRUCTURE_FAT32 *ptrEntry = 0;
    unsigned char *ptrData;
    int j;
    CHAR cBuf[12];

    if (fnLoadSector(ptr_utDisk, ptrFileLocation->directory_location.ulSector) != UTFAT_SUCCESS) { // load new sector (this will only read from physical disk when the sector is not already in local cache)
        return UTFAT_DISK_READ_ERROR;
    }
    ptrEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
    ptrEntry += ptrFileLocation->ucDirectoryEntry;                   // move to the present entry

    ptrData = (unsigned char *)ptrEntry;
    fnDebugMsg("Data =");
    for (j = 0; j < 32; j++) {                                       // display an entry
        fnDebugHex(*ptrData++, (WITH_LEADIN | WITH_SPACE | sizeof(unsigned char)));
    }
    fnDebugMsg("\r\nSFN name (");
    switch (ptrEntry->DIR_Attr) {
    case DIR_ATTR_READ_ONLY:
        fnDebugMsg("read-only");
        break;
    case DIR_ATTR_HIDDEN:
        fnDebugMsg("hidden");
        break;
    case DIR_ATTR_SYSTEM:
        fnDebugMsg("system");
        break;
    case DIR_ATTR_VOLUME_ID:
        fnDebugMsg("volume ID");
        break;
    case DIR_ATTR_DIRECTORY:
        fnDebugMsg("directory");
        break;
    case DIR_ATTR_ARCHIVE:
        fnDebugMsg("archive");
        break;
    default:
        fnDebugMsg("???");
        break;
    }
    fnDebugMsg(") = ");
    ptrData = (unsigned char *)ptrEntry;
    for (j = 0; j < 11; j++) {                                       // display an entry
        cBuf[j] = *ptrData++;
        if ((cBuf[j] < ' ') || (cBuf[j] >= 0x7f)) {                  // characters that can't be displayed
            cBuf[j] = '.';
        }
    }
    cBuf[11] = 0;
    fnDebugMsg(cBuf);
    #if defined UTFAT_LFN_READ
    if (ptr_utFile->lfn_file_location.directory_location.ulSector != 0) { // if a long file name
        fnDebugMsg(" Alias CS = ");
        fnDebugHex(ptr_openBlock->ucSFN_alias_checksum, (sizeof(ptr_openBlock->ucSFN_alias_checksum) | WITH_LEADIN));
    }
    #endif
    if (iFile != 0) {
        fnDebugMsg("\r\nFile length = ");
        fnDebugDec(ptr_utFile->ulFileSize, 0);
        ulCluster = ptr_utFile->private_file_location.ulCluster;
        ulSector = ptr_utFile->private_file_location.ulSector;
    }
    else {
        ulCluster = ptr_openBlock->ulCluster;
        ulSector = ptr_utDisk->ulVirtualBaseAddress + (ptr_openBlock->ulCluster * ptr_utDisk->utFAT.ucSectorsPerCluster);
    }
    ulFatSector = (ptr_utDisk->utFAT.ulFAT_start + (ulCluster >> 7));
    fnDebugMsg(" starting in sector ");
    fnDebugHex(ulSector, (WITH_LEADIN | sizeof(ulSector)));
    fnDebugMsg(" (cluster ");
    fnDebugHex(ulCluster, (WITH_LEADIN | sizeof(ulCluster)));
    fnDebugMsg(") FAT sector ");
    fnDebugHex(ulFatSector, (WITH_LEADIN | sizeof(ulFatSector)));
    fnDebugMsg(" offset ");
    fnDebugHex((ulCluster & 0x7f), (WITH_LEADIN | WITH_CR_LF | sizeof(unsigned char)));
    return UTFAT_SUCCESS;
}
#endif

#if defined UTFAT_EXPERT_FUNCTIONS
static void fnDisplayFileInfo(int iFile, const CHAR *ptrFilePath, UTFILE *ptr_utFile, OPEN_FILE_BLOCK *ptr_openBlock)
{
    if (iFile != 0) {
        fnDebugMsg("File: ");
    }
    else {
        fnDebugMsg("Directory: ");
    }
    fnDebugMsg((CHAR *)ptrFilePath);
    #if defined UTFAT_LFN_READ
    if (ptr_utFile->lfn_file_location.directory_location.ulSector == 0) { // if a short file name
        fnDebugMsg(" is SFN\r\n");
    }
    else {
        fnDebugMsg(" is LFN ");
        fnDebugMsg("\r\nStarting at entry ");
        fnDebugHex(ptr_utFile->lfn_file_location.ucDirectoryEntry, (WITH_LEADIN | sizeof(ptr_utFile->lfn_file_location.ucDirectoryEntry)));
        fnDebugMsg(" in sector ");
        fnDebugHex(ptr_utFile->lfn_file_location.directory_location.ulSector, (WITH_LEADIN | sizeof(ptr_utFile->lfn_file_location.directory_location.ulSector)));
        fnDebugMsg(" (cluster ");
        fnDebugHex(ptr_utFile->lfn_file_location.directory_location.ulCluster, (WITH_LEADIN | sizeof(ptr_utFile->lfn_file_location.directory_location.ulCluster)));
        fnDebugMsg(")\r\n");
        fnDisplayLFN(&ptr_utFile->lfn_file_location, &utDisks[ptr_utFile->ptr_utDirObject->ucDrive]); // display long file name information
    }
    #else
    fnDebugMsg(" is SFN\r\n");
    #endif
    fnDebugMsg("SNF File located at entry ");
    fnDebugHex(ptr_utFile->private_disk_location.ucDirectoryEntry, (WITH_LEADIN | sizeof(ptr_utFile->private_disk_location.ucDirectoryEntry)));
    fnDebugMsg(" in sector ");
    fnDebugHex(ptr_utFile->private_disk_location.directory_location.ulSector, (WITH_LEADIN | sizeof(ptr_utFile->private_disk_location.directory_location.ulSector)));
    fnDebugMsg(" (cluster ");
    fnDebugHex(ptr_utFile->private_disk_location.directory_location.ulCluster, (WITH_LEADIN | sizeof(ptr_utFile->private_disk_location.directory_location.ulCluster)));
    fnDebugMsg(")\r\n");
    fnDisplaySFN(iFile, ptr_utFile, ptr_openBlock);                      // display short file name information
}
#endif

// Internal function to open a file or directory object
//
static int _utOpenFile(const CHAR *ptrFilePath, UTFILE *ptr_utFile, unsigned long ulAccessMode)
{
    int iReturn = UTFAT_SUCCESS;
    OPEN_FILE_BLOCK openBlock;
    uMemset(&openBlock, 0, sizeof(openBlock));                           // initialise open file block

    if (ulAccessMode & UTFAT_OPEN_FOR_RENAME) {
        ptr_utFile->ptr_utDirObject->usDirectoryFlags |= (UTDIR_DIR_AS_FILE | UTDIR_REFERENCED | UTDIR_SET_START); // opens for renames are set to allow directories to be handled as files 
    }
    else {
        ptr_utFile->ptr_utDirObject->usDirectoryFlags |= (UTDIR_REFERENCED | UTDIR_SET_START); // the next open is a referenced open; set to start of lowest directory when file not found
    }

    iReturn = _fnHandlePath(&openBlock, ptrFilePath, ptr_utFile->ptr_utDirObject); // handle the input path string
    if (openBlock.iContinue == 0) {                                      // if the path handling had an error or completed all work
        return iReturn;                                                  // return with code
    }
#if defined UTFAT_EXPERT_FUNCTIONS
    if (ulAccessMode & UTFAT_DISPLAY_INFO) {
        openBlock.usDirFlags |= UTDIR_DIR_AS_FILE;                       // handle directories as files
    }
#endif

    iReturn = _utOpenDirectory(&openBlock, ptr_utFile->ptr_utDirObject, ((ulAccessMode & UTFAT_OPEN_DELETED) != UTFAT_SUCCESS));
    if (UTFAT_PATH_IS_FILE == iReturn) {
        uMemcpy(&ptr_utFile->private_disk_location, &ptr_utFile->ptr_utDirObject->public_disk_location, sizeof(ptr_utFile->private_disk_location)); // copy the referenced directory details
        uMemcpy(&ptr_utFile->private_file_location, &ptr_utFile->ptr_utDirObject->public_file_location, sizeof(ptr_utFile->private_file_location)); // copy the referenced file start details
        uMemcpy(&ptr_utFile->public_file_location, &ptr_utFile->ptr_utDirObject->public_file_location, sizeof(ptr_utFile->public_file_location)); // copy the referenced file start details
        if (ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_Attr & DIR_ATTR_READ_ONLY) { // {10}
            ulAccessMode &= ~(UTFAT_OPEN_FOR_DELETE | UTFAT_OPEN_FOR_WRITE); // the file is read-only so remove possible delete and write modes
        }
        ptr_utFile->ulFileMode = ulAccessMode;
        ptr_utFile->ulFileSize = ((ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_FileSize[3] << 24) + (ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_FileSize[2] << 16) + (ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_FileSize[1] << 8) + ptr_utFile->ptr_utDirObject->ptrEntryStructure->DIR_FileSize[0]);
        ptr_utFile->ulFilePosition = 0;                                  // set position to start of file on open
        ptr_utFile->ucDrive = ptr_utFile->ptr_utDirObject->ucDrive;
#if defined UTFAT_LFN_READ && ((defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE) || defined UTFAT_EXPERT_FUNCTIONS)
        uMemcpy(&ptr_utFile->lfn_file_location, &openBlock.lfn_file_location, sizeof(ptr_utFile->lfn_file_location)); // save the start of a LFN entry so that its length and location is known
        ptr_utFile->ucLFN_entries = openBlock.ucLFN_entries;
#endif
#if defined UTFAT_WRITE
        if (ulAccessMode & UTFAT_TRUNCATE) {                             // the file is to be overwritten so delete its content
            fnDeleteFileContent(ptr_utFile, &utDisks[ptr_utFile->ucDrive], REUSE_CLUSTERS); // delete the content and set the file length to zero
            ptr_utFile->ulFileSize = 0;
        }
        else if (UTFAT_APPEND & ulAccessMode) {
            if ((iReturn = utSeek(ptr_utFile, 0, UTFAT_SEEK_END)) != UTFAT_SUCCESS) { // seek to the end of the file so that writes cause append
                return iReturn;
            }
        }
#endif
#if defined UTFAT_EXPERT_FUNCTIONS
        if (ulAccessMode & UTFAT_DISPLAY_INFO) {
            fnDisplayFileInfo(1, ptrFilePath, ptr_utFile, &openBlock);
        }
#endif
        return (UTFAT_PATH_IS_FILE);
    }
#if defined UTFAT_WRITE
    else if (ulAccessMode & UTFAT_CREATE) {                              // file doesn't exist so we should create it
        if (iReturn != UTFAT_FILE_NOT_FOUND) {
            return iReturn;                                              // invalid path has been entered so return error code
        }
        openBlock.ptr_utDisk = &utDisks[ptr_utFile->ptr_utDirObject->ucDrive];
        return (fnCreateFile(&openBlock, ptr_utFile, openBlock.ptrLocalDirPath, (ulAccessMode & ~_RENAME_EXISTING))); // create the file
    }
#endif
    else if (iReturn == UTFAT_SUCCESS) {                                 // a directory was matched 
        uMemcpy(&ptr_utFile->private_disk_location, &ptr_utFile->ptr_utDirObject->public_disk_location, sizeof(ptr_utFile->ptr_utDirObject->public_disk_location)); // copy the referenced directory details
        ptr_utFile->ulFileMode = (UTFAT_FILE_IS_DIR | ulAccessMode);     // mark that the entry is not a file but a directory
        ptr_utFile->ucDrive = ptr_utFile->ptr_utDirObject->ucDrive;
#if defined UTFAT_LFN_READ && ((defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE) || defined UTFAT_EXPERT_FUNCTIONS)
        uMemcpy(&ptr_utFile->lfn_file_location, &openBlock.lfn_file_location, sizeof(ptr_utFile->lfn_file_location)); // save the start of a LFN entry so that its length and location is known
        ptr_utFile->ucLFN_entries = openBlock.ucLFN_entries;
#endif
#if defined UTFAT_EXPERT_FUNCTIONS
        if (ulAccessMode & UTFAT_DISPLAY_INFO) {
            fnDisplayFileInfo(0, ptrFilePath, ptr_utFile, &openBlock);
        }
#endif
    }
    return iReturn;
}

// Open a file or directory object
//
extern int utOpenFile(const CHAR *ptrFilePath, UTFILE *ptr_utFile, UTDIRECTORY *ptr_utDirectory, unsigned long ulAccessMode)
{
    if ((ptr_utFile->ptr_utDirObject = ptr_utDirectory) == 0) {
        return UTFAT_SEARCH_INVALID;
    }    
    if ((utDisks[ptr_utFile->ptr_utDirObject->ucDrive].usDiskFlags & WRITE_PROTECTED_SD_CARD) && (ulAccessMode & (UTFAT_OPEN_FOR_RENAME | UTFAT_OPEN_FOR_WRITE | UTFAT_OPEN_FOR_DELETE))) { // {34}
        return UTFAT_DISK_WRITE_PROTECTED;
    }
#if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0             // allow SD card interface to be used without UTMANAGED_FILE_COUNT
    if (ulAccessMode & UTFAT_MANAGED_MODE) {
        int iFileHandle = 0;
        UTMANAGED_FILE *ptrManagedFile = utManagedFiles;
        while (iFileHandle < UTMANAGED_FILE_COUNT) {
            if (ptrManagedFile->managed_owner == 0) {                
                if ((utOpenFile(ptrFilePath, ptr_utFile, ptr_utDirectory, (ulAccessMode & ~UTFAT_MANAGED_MODE)) != UTFAT_PATH_IS_FILE)) { // recursive call with adjusted access mode
                    return UTFAT_FILE_NOT_FOUND;                         // file doesn't exist
                }
                if (fnFileLocked(ptr_utFile) != 0) {
                    return UTFAT_FILE_LOCKED;                            // the file is locked by another user so may not be opened
                }
                ptrManagedFile->utManagedFile = ptr_utFile;
                ptrManagedFile->managed_owner = ptr_utFile->ownerTask;   // validate the entry with the owner task
                ptr_utFile->ulFileMode = ulAccessMode;
                ptrManagedFile->managed_mode = (unsigned char)ulAccessMode;
                ptr_utFile->iFileHandle = iFileHandle;
                return UTFAT_PATH_IS_FILE;                               // open in managed mode was successful
            }
            iFileHandle++;
            ptrManagedFile++;
        }
        return MANAGED_FILE_NO_FILE_HANDLE;                              // all managed file spaces are presently occupied
    }
#endif
#if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
    ptr_utFile->ptrFileDataCache = 0;                                    // initially no cache is allocated
#endif
    return _utOpenFile(ptrFilePath, ptr_utFile, ulAccessMode);
}

#if defined UTFAT_WRITE                                                  // allow operation without write support
    #if defined UTFAT_LFN_READ && (defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE)
// If the file name being deleted this routine deletes the complete long file name part before the short file name alias (it doesn't delete file content)
//
static int fnDeleteLFN_entry(UTFILE *ptr_utFile)
{
    DISK_LOCATION FileLocation;
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
    unsigned long ulSectorToWrite = ptr_utDisk->ulPresentSector;
    LFN_ENTRY_STRUCTURE_FAT32 *ptrFileEntry;
    unsigned char ucEntryNumber;

    if (ptr_utFile->lfn_file_location.directory_location.ulSector == 0) {// the file is using a short file name and so no long file name part needs to be deleted
        return UTFAT_SUCCESS;
    }
    uMemcpy(&FileLocation, &(ptr_utFile->lfn_file_location), sizeof(DISK_LOCATION)); // make a copy of the long file name entry details
    while (1) {
        ptr_utDisk->usDiskFlags |= WRITEBACK_BUFFER_FLAG;                // mark that the modified sector content must be committed each time the sector is changed
        if (fnLoadSector(ptr_utDisk, FileLocation.directory_location.ulSector) != UTFAT_SUCCESS) { // load sector where the LFN begins
            return UTFAT_DISK_READ_ERROR;
        }
        ptrFileEntry = (LFN_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData;
        ptrFileEntry += FileLocation.ucDirectoryEntry;
        ucEntryNumber = ptrFileEntry->LFN_EntryNumber;
        ptrFileEntry->LFN_EntryNumber = DIR_NAME_FREE;                   // delete the entry
        if ((ucEntryNumber & ~0x40) <= 1) {
            break;                                                       // final LFN entry has been deleted (following SFN alias still exists)
        }
        fnNextDirectoryEntry(ptr_utDisk, &FileLocation);                 // move to next entry
    }
    // Since the renames file object will be added there is no need to commit the changes yet
    //
    return UTFAT_SUCCESS;
}
    #endif

// Rename a referenced file or directory (can result in SFN->LFN or LFN->SFN conversions or LFN with different entry lengths)
//
extern int utRenameFile(const CHAR *ptrFilePath, UTFILE *ptr_utFile)
{
    int iReturn;
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
    DISK_LOCATION file_location;
    OPEN_FILE_BLOCK openBlock;

    if (ptr_utDisk->usDiskFlags & WRITE_PROTECTED_SD_CARD) {
        return UTFAT_DISK_WRITE_PROTECTED;                               // can't rename anything on a write protected disk
    }
    if (!(ptr_utFile->ulFileMode & (UTFAT_OPEN_FOR_RENAME | UTFAT_OPEN_FOR_DELETE))) { // if the file is neither open for rename or write don't allow a rename
        return UTFAT_FILE_NOT_WRITEABLE;                                 // file access doesn't allow renaming
    }
    uMemset(&openBlock, 0, sizeof(openBlock));                           // initialise open file block

    uMemcpy(&file_location, &ptr_utFile->ptr_utDirObject->public_disk_location, sizeof(file_location)); // backup the present file location details

    iReturn = _fnHandlePath(&openBlock, ptrFilePath, ptr_utFile->ptr_utDirObject); // handle the input path string
    if (openBlock.iContinue == 0) {                                      // if the path handling had an error or completed all work
        return iReturn;                                                  // return with code
    }
    if ((iReturn = _utOpenDirectory(&openBlock, ptr_utFile->ptr_utDirObject, 0)) != UTFAT_FILE_NOT_FOUND) { // check that the new file or directory name doesn't already exist
        return iReturn;                                                  // if the file or directory is already present return code
    }
#if !defined UTFAT_LFN_WRITE                                             // don't allow renaming to LFN when LFN writing is not supported
    if (FULLY_QUALIFIED_LONG_NAME == openBlock.iQualifiedPathType) {
        return LFN_RENAME_NOT_POSSIBLE;
    }
#endif
    openBlock.ptr_utDisk = ptr_utDisk;
    uMemcpy(&openBlock.DirectoryEndLocation, &ptr_utFile->ptr_utDirObject->public_disk_location, sizeof(openBlock.DirectoryEndLocation));
    uMemcpy(&ptr_utFile->ptr_utDirObject->public_disk_location, &file_location, sizeof(file_location)); // return the file location details
    return (fnCreateFile(&openBlock, ptr_utFile, openBlock.ptrLocalDirPath, _RENAME_EXISTING)); // delete the original file/directory name and create the new one without changing file properties or content
}
#endif

// A close only has real significance when the file was opened as a managed file, or when data caching or delayed directory object write is used
//
extern int utCloseFile(UTFILE *ptr_utFile)
{
    int iReturn = UTFAT_SUCCESS;
#if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0             // allow SD card interface to be used without UTMANAGED_FILE_COUNT
    if (ptr_utFile->ulFileMode & UTFAT_MANAGED_MODE) {
        utManagedFiles[ptr_utFile->iFileHandle].managed_owner = 0;       // free the managed file entry
    }
#endif
#if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
    if (ptr_utFile->ptrFileDataCache != 0) {                             // if we are working with a data cache check whether data has to be saved to disk
        FILE_DATA_CACHE *ptrDataCache = ptr_utFile->ptrFileDataCache;
    #if defined UTFAT_WRITE
        if (ptrDataCache->ucFileCacheState & FILE_BUFFER_MODIFIED) {     // if modified data waiting to be saved
            while (utCommitSectorData(&utDisks[ptr_utFile->ucDrive], ptrDataCache->ucFileDataCache, ptrDataCache->ulFileCacheSector) == CARD_BUSY_WAIT) {} // commit the data to the disk
        }
    #endif
        ptrDataCache->ucFileCacheState &= ~(FILE_BUFFER_MODIFIED);       // free the cache for use by other files
    }
#endif
#if defined UTFAT_WRITE
    if (ptr_utFile->ulFileMode & _FILE_CHANGED) {                        // the file's content was changed while the file was open so we write the information on close
        iReturn = fnCommitFileInfo(ptr_utFile, &utDisks[ptr_utFile->ucDrive]);
    }
#endif
    uMemset(ptr_utFile, 0, sizeof(UTFILE));                              // delete the file object
    return iReturn;
}


#if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0             // allow SD card interface to be used without UTMANAGED_FILE_COUNT
    #if defined UTFAT_WRITE
// The file has changed so update all user information
//
static void fnSynchroniseManagedFile(UTFILE *ptr_utFile)
{
    int iFileHandle = 0;
    while (iFileHandle < UTMANAGED_FILE_COUNT) {                         // check through the managed file list
        if (utManagedFiles[iFileHandle].managed_owner != 0) {            // if the entry is valid
            if (uMemcmp(&ptr_utFile->private_file_location, &utManagedFiles[iFileHandle].utManagedFile->private_file_location, sizeof(FILE_LOCATION)) == 0) { // if a managed file matches this one
                utManagedFiles[iFileHandle].utManagedFile->ulFileSize = ptr_utFile->ulFileSize; // update the file size
                if (utManagedFiles[iFileHandle].utManagedFile->ulFileSize < utManagedFiles[iFileHandle].utManagedFile->ulFilePosition) { // if the file size has shrunk to before the present file location
                    utManagedFiles[iFileHandle].utManagedFile->ulFilePosition = utManagedFiles[iFileHandle].utManagedFile->ulFileSize; // set the new file location to the end of the file
                }
            }
        }
        iFileHandle++;
    }
}
        #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
static void fnSynchroniseManagedFileCache(FILE_DATA_CACHE *ptrDataCache)
{
    int iFileHandle = 0;
    while (iFileHandle < UTMANAGED_FILE_COUNT) {                         // check through the managed file list
        if (utManagedFiles[iFileHandle].managed_owner != 0) {            // if the entry is valid
            if (utManagedFiles[iFileHandle].utManagedFile->ptrFileDataCache) { // and a managed file has its own cache
                if (ptrDataCache->ulFileCacheSector == utManagedFiles[iFileHandle].utManagedFile->ptrFileDataCache->ulFileCacheSector) { // and is caching the same sector
                    if (utManagedFiles[iFileHandle].utManagedFile->ptrFileDataCache->ucFileDataCache != ptrDataCache->ucFileDataCache) {
                        uMemcpy(utManagedFiles[iFileHandle].utManagedFile->ptrFileDataCache->ucFileDataCache, ptrDataCache->ucFileDataCache, sizeof(utManagedFiles[iFileHandle].utManagedFile->ptrFileDataCache->ucFileDataCache)); // synchronise the other cache content
                    }
                }
            }
        }
        iFileHandle++;
    }
}
        #endif
    #endif
    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
static int fnGetManagedFileCache(unsigned long ulSector, unsigned char *ptrBuffer, unsigned short usAccessOffset, unsigned short usAccessLength)
{
    int iFileHandle = 0;
    while (iFileHandle < UTMANAGED_FILE_COUNT) {                         // check through the managed file list
        if (utManagedFiles[iFileHandle].managed_owner != 0) {            // if the entry is valid
            if (utManagedFiles[iFileHandle].utManagedFile->ptrFileDataCache) { // and a managed file has its own cache
                if (ulSector == utManagedFiles[iFileHandle].utManagedFile->ptrFileDataCache->ulFileCacheSector) { // and is caching the same sector being read
                    uMemcpy(ptrBuffer, &utManagedFiles[iFileHandle].utManagedFile->ptrFileDataCache->ucFileDataCache[usAccessOffset], usAccessLength); // read data form the cache
                    return 1;                                            // copy has been taken from cache so no need to read form the disk
                }
            }
        }
        iFileHandle++;
    }
    return 0;                                                            // no cache entries found
}
    #endif

// Check whether the file is locked by another user
//
static int fnFileLocked(UTFILE *ptr_utFile)
{
    int iFileHandle = 0;
    while (iFileHandle < UTMANAGED_FILE_COUNT) {
        if (utManagedFiles[iFileHandle].managed_owner != 0) {
            if (uMemcmp(&ptr_utFile->private_file_location, &utManagedFiles[iFileHandle].utManagedFile->private_file_location, sizeof(FILE_LOCATION)) == 0) {
                if (utManagedFiles[iFileHandle].managed_mode & UTFAT_PROTECTED) {
                    return 1;                                            // another user is locking this file
                }
            }
        }
        iFileHandle++;
    }
    return 0;                                                            // this file is not being locked by another user
}
#endif

#if defined UTFAT_WRITE
extern int utWriteFile(UTFILE *ptr_utFile, unsigned char *ptrBuffer, unsigned short usLength)
{
    unsigned short usWriteLength;
    unsigned short usRemainingBuffer;
    unsigned char  ucDataContent[512];                                   // temporary buffer for reading disk data to and manipulating data in
    unsigned char *ptrFileDateBuffer = ucDataContent;                    // use the temporary buffer by default
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
    FILE_DATA_CACHE *ptrDataCache;
    #endif
    ptr_utFile->usLastReadWriteLength = 0;
    if (ptr_utDisk->usDiskFlags & WRITE_PROTECTED_SD_CARD) {
        return UTFAT_DISK_WRITE_PROTECTED;
    }
    if (!(ptr_utFile->ulFileMode & UTFAT_OPEN_FOR_WRITE)) {              // only allow writes if the file is open for writing
        return UTFAT_FILE_NOT_WRITEABLE;
    }
    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
    if ((ptr_utFile->ulFileMode & UTFAT_WITH_DATA_CACHE) && (ptr_utFile->ptrFileDataCache == 0)) { // use a file data buffer if one is available in the pool
        ptr_utFile->ptrFileDataCache = fnGetDataCache();                 // try to get data cache to work with
    }
    ptrDataCache = ptr_utFile->ptrFileDataCache;
    #endif
    while (usLength != 0) {
        usRemainingBuffer = (unsigned short)(ptr_utFile->ulFilePosition % 512); // the start offset in the sector buffer to be written to
        if ((usRemainingBuffer != 0) || ((usLength < 512) && (usLength < (ptr_utFile->ulFileSize - ptr_utFile->ulFilePosition)))) { // if the complete content is not being overwritten
    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
            if (ptrDataCache != 0) {                                     // if a data cache is being used by this file
                int iReturnValue = fnHandleFileDataCache(ptr_utDisk, ptr_utFile, ptrDataCache, 0); // handle cache - load data to cache if necessary and save any cached data that may require saving
                if (iReturnValue != UTFAT_SUCCESS) {
                    return iReturnValue;                                 // access error
                }
                ptrFileDateBuffer = ptrDataCache->ucFileDataCache;       // use the data cache
            }
            else {
    #endif
                if (utReadDiskSector(ptr_utDisk, ptr_utFile->public_file_location.ulSector, ucDataContent) != UTFAT_SUCCESS) { // read data sector to the temporary buffer
                    return UTFAT_DISK_READ_ERROR;
                }
    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
            }
    #endif
        }
        else {                                                           // the complete sector content will be overwritten so there is no need to load any data from disk
    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
            if (ptrDataCache != 0) {                                     // if a data cache is being used by this file
                int iReturnValue = fnHandleFileDataCache(ptr_utDisk, ptr_utFile, ptrDataCache, 1); // handle cache - save any cached data that may require saving
                if (iReturnValue != UTFAT_SUCCESS) {
                    return iReturnValue;                                 // access error
                }
                ptrFileDateBuffer = ptrDataCache->ucFileDataCache;       // use the data cache
            }
    #endif
            uMemset(ptrFileDateBuffer, 0, sizeof(ucDataContent));        // zero the temporary data buffer so that any unmodified content at the end will be 0x00
        }
        usWriteLength = usLength;
        if (usWriteLength > (512 - usRemainingBuffer)) {                 // check whether the data will fit into the content buffer
            usWriteLength = (512 - usRemainingBuffer);                   // if it won't fit copy as much as possible
        }
        uMemcpy(&ptrFileDateBuffer[usRemainingBuffer], ptrBuffer, usWriteLength); // copy the data that is to be saved to the correct location in the data content buffer
    #if defined UTFAT_FILE_CACHE_POOL && UTFAT_FILE_CACHE_POOL > 0
        if (ptrDataCache != 0) {                                         // if working with data cache
            ptrDataCache->ucFileCacheState |= (FILE_BUFFER_MODIFIED | FILE_BUFFER_VALID); // content has been modified and so needs to be saved
            ptrDataCache->ulFileCacheSector = ptr_utFile->public_file_location.ulSector; // the sector that the cache belongs to
        #if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0
            fnSynchroniseManagedFileCache(ptrDataCache);                 // synchronise any managed files caching the data content
        #endif
        }
        else {
            while (utCommitSectorData(ptr_utDisk, ptrFileDateBuffer, ptr_utFile->public_file_location.ulSector) == CARD_BUSY_WAIT) {} // commit the data to the disk
        }
    #else
        while (utCommitSectorData(ptr_utDisk, ptrFileDateBuffer, ptr_utFile->public_file_location.ulSector) == CARD_BUSY_WAIT) {} // commit the data to the disk
    #endif
        ptrBuffer += usWriteLength;                                      // increment the data pointer to beyond the data that has been saved
        ptr_utFile->ulFilePosition += usWriteLength;                     // also move the present file pointer accordingly
        if (ptr_utFile->ulFilePosition > ptr_utFile->ulFileSize) {       // if the write has made the file grow in size
            ptr_utFile->ulFileSize = ptr_utFile->ulFilePosition;         // increase the file length accordingly
        }
        ptr_utFile->usLastReadWriteLength += usWriteLength;              // the number of bytes written
        usLength -= usWriteLength;                                       // the number of bytes that still need to be written
        if ((usRemainingBuffer + usWriteLength) >= 512) {                // if the write has just filled a sector
            int iResult = fnNextSectorCreate(ptr_utDisk, &(ptr_utFile->public_file_location), 0); // move to next sector and create a new cluster if needed
            if (iResult != UTFAT_SUCCESS) {                              // move to next sector/cluster and create additional cluster if necessary
                return iResult;
            }
        }
    }
    if (ptr_utFile->usLastReadWriteLength != 0) {                        // updata the file entry if there was a change
        if (ptr_utFile->ulFileMode & UTFAT_COMMIT_FILE_ON_CLOSE) {       // if the user wishes to make file size/time stamp changes only on file close
            ptr_utFile->ulFileMode |= _FILE_CHANGED;                     // mark that a change is still required - it takes place on file close
        }
        else {
            int iReturn = fnCommitFileInfo(ptr_utFile, ptr_utDisk);
            if (UTFAT_SUCCESS != iReturn) {
                return iReturn;
            }
        }
    }
    #if defined UTMANAGED_FILE_COUNT && UTMANAGED_FILE_COUNT > 0         // allow SD card interface to be used without UTMANAGED_FILE_COUNT
    fnSynchroniseManagedFile(ptr_utFile);
    #endif
    return UTFAT_SUCCESS;
}
#endif

// Change the directory location with reference to its present position
//
extern int utChangeDirectory(const CHAR *ptrDirPath, UTDIRECTORY *ptrDirObject)
{
    unsigned short usPathTerminator = ptrDirObject->usRelativePathLocation; // original path location
    ptrDirObject->usDirectoryFlags |= UTDIR_ALLOW_MODIFY_PATH;           // allow the open to modify the path if necessary
    switch (utOpenDirectory(ptrDirPath, ptrDirObject)) {                 // try to locate the new directory
    case UTFAT_PATH_IS_ROOT_REF:
        ptrDirObject->usRelativePathLocation = usPathTerminator = 3;     // reset the root string
        ptrDirPath++;                                                    // jump the root reference and fall through to set new path
    case UTFAT_SUCCESS:
        if (ptrDirObject->ptrDirectoryPath != 0) {
            if (*ptrDirPath != 0) {
                if (usPathTerminator > 3) {
                    ptrDirObject->ptrDirectoryPath[usPathTerminator] = '\\';
                }
                if (usPathTerminator != 3) {
                    usPathTerminator++;
                }
                ptrDirObject->usRelativePathLocation = (uStrcpy(&ptrDirObject->ptrDirectoryPath[usPathTerminator], ptrDirPath) - ptrDirObject->ptrDirectoryPath);
            }
            else {
                ptrDirObject->usRelativePathLocation = usPathTerminator;
                if (usPathTerminator == 2) {
                    usPathTerminator = 3;
                }
                ptrDirObject->ptrDirectoryPath[usPathTerminator] = 0;
            }
        }
        break;
    case UTFAT_PATH_IS_ROOT:                                             // change was to the root
        ptrDirObject->usRelativePathLocation = 3;                        // reset the root string
        ptrDirObject->ptrDirectoryPath[3] = 0;
        break;
    case UTFAT_DISK_NOT_READY:
        return UTFAT_DISK_NOT_READY;
    case UTFAT_DISK_READ_ERROR:
        return UTFAT_DISK_READ_ERROR;
    case UTFAT_SUCCESS_PATH_MODIFIED:                                    // successful and no further action required since it has been performed
        break;
    default:
         return UTFAT_PATH_NOT_FOUND;
    }
    uMemcpy(&ptrDirObject->private_disk_location, &ptrDirObject->public_disk_location, sizeof(ptrDirObject->private_disk_location)); // commit the new directory location
    return UTFAT_SUCCESS;
}

// Move through sectors and cluster chain until the end sector is reached
//
static int fnRunThroughClusters(UTDISK *ptr_utDisk, FILE_LOCATION *prtFileLocation, unsigned long ulStartPosition, unsigned long ulEndPosition)
{
    int iResult = UTFAT_SUCCESS;
    unsigned short usBytesPerSector = ptr_utDisk->utFAT.usBytesPerSector;
    ulStartPosition &= ~(usBytesPerSector - 1);                          // round down to start of present sector
    ulEndPosition &= ~(usBytesPerSector - 1);                            // round down to start of end sector
    while (ulStartPosition < ulEndPosition) {
        ulStartPosition += usBytesPerSector;
        if ((iResult = fnNextSectorCreate(ptr_utDisk, prtFileLocation, 1)) != UTFAT_SUCCESS) { // if the end of the file is reached, which is at the end of a cluster, a new cluster will be created if write is supported
            break;
        }
    }
    return iResult;
}

extern int utSeek(UTFILE *ptr_utFile, unsigned long ulPosition, int iSeekType)
{
    int iResult = UTFAT_SUCCESS;
    unsigned long ulNewPosition = 0;                                     // default to start of file
    UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
    switch (iSeekType) {
    case UTFAT_SEEK_SET:                                                 // relative to the start of the file - always forwards
        ulNewPosition = ulPosition;
        break;
    case UTFAT_SEEK_CUR:                                                 // relative to current position
        if ((signed long)ulPosition < 0) {                               // relative backwards
            if ((unsigned long)(-(signed long)ulPosition) > ptr_utFile->ulFilePosition) {
                break;                                                   // move to beginning of file
            }
        }
        else {                                                           // relative forwards
            if (ulPosition > (ptr_utFile->ulFileSize - ptr_utFile->ulFilePosition)) {
                ulPosition = (ptr_utFile->ulFileSize - ptr_utFile->ulFilePosition);
            }
        }
        ulNewPosition = ptr_utFile->ulFilePosition + ulPosition;
        break;
    case UTFAT_SEEK_END:                                                 // relative to the end of the file (always backwards)
        if (ulPosition < ptr_utFile->ulFileSize) {
            ulNewPosition = (ptr_utFile->ulFileSize - ulPosition);
        }
        break;
    }
    if (ulNewPosition != ptr_utFile->ulFilePosition) {                   // update the cluster pointer
        if (ulNewPosition < ptr_utFile->ulFilePosition) {                // move back
            uMemcpy(&ptr_utFile->public_file_location, &ptr_utFile->private_file_location, sizeof(ptr_utFile->private_file_location)); // rewind to start and search from there
            ptr_utFile->ulFilePosition = 0;
        }
        else if (ulNewPosition > ptr_utFile->ulFileSize) {               // limit seek to end of file
            ulNewPosition = ptr_utFile->ulFileSize;                      
        }
        iResult = fnRunThroughClusters(ptr_utDisk, &ptr_utFile->public_file_location, ptr_utFile->ulFilePosition, ulNewPosition);
        if (iResult == UTFAT_SUCCESS) {                                  // if the cluster location could be successfully set
            ptr_utFile->ulFilePosition = ulNewPosition;                  // set the file position to the new position
        }
    }
    return iResult;
}

#if defined UTFAT_WRITE                                                  // support writing as well as reading
    #if defined NAND_FLASH_FAT
static int utCommitSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    switch (iMemoryOperation[DISK_D] & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation[DISK_D] |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            if (fnCheckBlankPage(ulSectorNumber) != 0) {
                if (fnOverWriteSector(ulSectorNumber, ptr_utDisk->ptrSectorData) != UTFAT_SUCCESS) {
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            else {
                if (fnWriteNANDsector(ulSectorNumber, 0, ptr_utDisk->ptrSectorData, 512) != 0) {
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            ptr_utDisk->ulPresentSector = ulSector;                      // update the sector which is presently valid
            iMemoryOperation[DISK_D] &= ~_WRITING_MEMORY;                // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

static int utCommitSectorData(UTDISK *ptr_utDisk, void *ptrBuffer, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    switch (iMemoryOperation[DISK_D] & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation[DISK_D] |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            if (fnCheckBlankPage(ulSectorNumber) != 0) {
                if (fnOverWriteSector(ulSectorNumber, ptrBuffer) != UTFAT_SUCCESS) {
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            else {
                if (fnWriteNANDsector(ulSector, 0, ptrBuffer, 512) != UTFAT_SUCCESS) {
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            iMemoryOperation[DISK_D] &= ~_WRITING_MEMORY;                // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Delete the specified sector by writing data content of 0x00
//
static int utDeleteSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    switch (iMemoryOperation[DISK_D] & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        iMemoryOperation[DISK_D] |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            unsigned long ulTemp[512/sizeof(unsigned long)];             // temporary long-word aligned buffer
    #if defined LONG_UMEMSET
            uMemset_long(ulTemp, 0x00, sizeof(ulTemp));                  // zero buffer content for delete
    #else
            uMemset(ulTemp, 0x00, sizeof(ulTemp));
    #endif
            if (fnWriteNANDsector(ulSector, 0, (unsigned char *)ulTemp, 512) != 0) {
                fnMemoryDebugMsg("Write error\r\n");
                return UTFAT_DISK_WRITE_ERROR;
            } 
            iMemoryOperation[DISK_D] &= ~_WRITING_MEMORY;                // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}
    #else
static int utCommitSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation[DISK_D] & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation[DISK_D] |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            unsigned char ucResult;
            if ((iActionResult = fnSendSD_command(fnCreateCommand(WRITE_BLOCK_CMD24, ulSector), &ucResult, 0)) != 0) {
                return iActionResult;
            }
            if (ucResult == 0) {
                if (fnPutSector(ptr_utDisk->ptrSectorData, 0) != UTFAT_SUCCESS) { // start writing buffer to single sector
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
                if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
                    ulSector /= 512;                                     // convert back to sector number from byte offset
                }
                ptr_utDisk->ulPresentSector = ulSector;                  // update the sector which is presently valid
            }
            SET_SD_CS_HIGH();
            iMemoryOperation[DISK_D] &= ~_WRITING_MEMORY;                // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}


static int utCommitSectorData(UTDISK *ptr_utDisk, void *ptrBuffer, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation[DISK_D] & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation[DISK_D] |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            unsigned char ucResult;
    #if defined UTFAT_MULTIPLE_BLOCK_WRITE
            if (ulBlockWriteLength != 0) {                               // if multiple block writes are to be performed
                if (!(ulBlockWriteLength & 0x80000000)) {                // if block write command has not yet been sent
                    if ((iActionResult = fnSendSD_command(fnCreateCommand(WRITE_MULTIPLE_BLOCK_CMD25, ulSector), &ucResult, (unsigned char *)&ulBlockWriteLength)) != 0) { // start multiple block write
                        return iActionResult;
                    }
                    ulMultiBlockAddress = ulSector;
                    ulBlockWriteLength |= 0x80000000;                    // mark that the command has been sent
                }
                else {
                /*  if (!(ulBlockWriteLength & ~0x80000000)) {           // if multiple write is being aborted
                        ulBlockWriteLength = 0;
                        while (fnSendSD_command(fnCreateCommand(STOP_TRANSMISSION_CMD12, 0), &ucResult, 0) == CARD_BUSY_WAIT) {}; // terminate present multiple block mode
                        if ((iActionResult = fnSendSD_command(fnCreateCommand(WRITE_BLOCK_CMD24, ulSector), &ucResult, 0)) != 0) { // single block write command
                            return iActionResult;
                        }
                    }*/
                    ucResult = 0;                                        // since we are in a multiple write state we can continue writing data
                }
            }
            else {
                if ((iActionResult = fnSendSD_command(fnCreateCommand(WRITE_BLOCK_CMD24, ulSector), &ucResult, 0)) != 0) { // single block write command
                    return iActionResult;
                }
            }
            if (ucResult == 0) {
                if (fnPutSector((unsigned char *)ptrBuffer, (ulBlockWriteLength != 0)) != UTFAT_SUCCESS) { // start writing buffer to single sector
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
                if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
                    ulMultiBlockAddress += 512;                         // monitor the multiple write sector
                }
                else {
                    ulMultiBlockAddress++;
                }
            }
    #else
            if ((iActionResult = fnSendSD_command(fnCreateCommand(WRITE_BLOCK_CMD24, ulSector), &ucResult, 0)) != 0) { // single block write command
                return iActionResult;
            }
            if (ucResult == 0) {
                if (fnPutSector((unsigned char *)ptrBuffer, 0) != UTFAT_SUCCESS) { // start writing buffer to single sector
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
    #endif
    #if defined UTFAT_MULTIPLE_BLOCK_WRITE
            if (ulBlockWriteLength != 0) {                               // in multiple block write
                ulBlockWriteLength--;                                    // block has been written
                if ((ulBlockWriteLength & ~(0x80000000)) == 0) {         // final block has been written
                    ulBlockWriteLength = 0;                              // planned multiple block write has completed
                    fnSendSD_command(fnCreateCommand(STOP_TRANSMISSION_CMD12, 0), &ucResult, 0); // terminate multiple block mode
                }
            }
    #endif
            SET_SD_CS_HIGH();
            iMemoryOperation[DISK_D] &= ~_WRITING_MEMORY;                // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}

// Delete the specified sector by writing data content of 0x00
//
static int utDeleteSector(UTDISK *ptr_utDisk, unsigned long ulSectorNumber)
{
    static unsigned long ulSector;
    static int iActionResult;
    switch (iMemoryOperation[DISK_D] & _WRITING_MEMORY) {
    case _IDLE_MEMORY:
        if (!(ptr_utDisk->usDiskFlags & HIGH_CAPACITY_SD_CARD)) {
            ulSectorNumber *= 512;                                       // convert the sector number to byte address
        }
        SET_SD_CS_LOW();
        iMemoryOperation[DISK_D] |= _WRITING_MEMORY;
        ulSector = ulSectorNumber;
    case _WRITING_MEMORY:
        {
            unsigned char ucResult;
            if ((iActionResult = fnSendSD_command(fnCreateCommand(WRITE_BLOCK_CMD24, ulSector), &ucResult, 0)) != 0) {
                return iActionResult;
            }
            if (ucResult == 0) {
                unsigned long ulTemp[512/sizeof(unsigned long)];         // temporary long-word aligned buffer
    #if defined LONG_UMEMSET
                uMemset_long(ulTemp, 0x00, sizeof(ulTemp));              // zero buffer content for delete
    #else
                uMemset(ulTemp, 0x00, sizeof(ulTemp));
    #endif
                if (fnPutSector((unsigned char *)ulTemp, 0) != UTFAT_SUCCESS) { // write sector content to 0x00
                    fnMemoryDebugMsg("Write error\r\n");
                    return UTFAT_DISK_WRITE_ERROR;
                }
            }
            SET_SD_CS_HIGH();
            iMemoryOperation[DISK_D] &= ~_WRITING_MEMORY;                // write operation has completed
        }
        break;
    }
    return UTFAT_SUCCESS;
}
    #endif

static int fnDeleteClusterChain(unsigned long ulClusterStart, unsigned char ucDrive, int iDestroyClusters)
{
    UTDISK *ptr_utDisk = &utDisks[ucDrive];
    unsigned long ulCluster = (((ulClusterStart - ptr_utDisk->ulLogicalBaseAddress)/ptr_utDisk->utFAT.ucSectorsPerCluster) + ptr_utDisk->ulDirectoryBase);
    unsigned long ulClusterSector;
    unsigned long ulNextCluster, ulNextClusterSector;
    unsigned long ulRemovedClusters = 0;
    unsigned char ucClusterEntry;
    unsigned long ulSectorContent[512/sizeof(signed long)];              // temporary long-word aligned buffer
    #if defined UTFAT16
    unsigned long ulClusterMask;
    if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulCluster >> 8)); // section where the FAT responsible for this cluster resides
        ulCluster -= (32 - 1);                                           // compensate for fixed 16k boot sector
        ucClusterEntry = (unsigned char)ulCluster;
        ulClusterMask = FAT16_CLUSTER_MASK;
    }
    else {
        ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulCluster >> 7)); // section where the FAT responsible for this cluster resides
        ucClusterEntry = (unsigned char)(ulCluster & 0x7f);
        ulClusterMask = CLUSTER_MASK;
    }
    #else
    ulClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulCluster >> 7)); // section where the FAT responsible for this cluster resides
    ucClusterEntry = (unsigned char)(ulCluster & 0x7f);
    #endif
    if ((utReadDiskSector(ptr_utDisk, ulClusterSector, ulSectorContent)) != UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
        return UTFAT_DISK_READ_ERROR;
    }
    while (1) {
    #if defined UTFAT16
        if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
            ulNextCluster = LITTLE_LONG_WORD(ulSectorContent[ucClusterEntry/2]);
            if (ucClusterEntry & 1) {
                ulNextCluster &= LITTLE_LONG_WORD(0xffff0000);
                ulNextCluster >>= 16;
            }
            else {
                ulNextCluster &= LITTLE_LONG_WORD(0x0000ffff);
            }
        }
        else {
            ulNextCluster = LITTLE_LONG_WORD(ulSectorContent[ucClusterEntry]); // read the next entry
        }
        if (((ulNextCluster & ulClusterMask) == ulClusterMask) || (ulNextCluster <= ptr_utDisk->ulDirectoryBase)) // check whether the end of the cluster chain has been reached (if the next cluster is 0, 1 or 2 it is ignored since it must be corrupt - 0, 1 are not used and 2 is always the start of the root directory! [FAT16 is 0..1])
    #else
        ulNextCluster = LITTLE_LONG_WORD(ulSectorContent[ucClusterEntry]); // read the next entry
        if (((ulNextCluster & CLUSTER_MASK) == CLUSTER_MASK) || (ulNextCluster <= ptr_utDisk->ulDirectoryBase)) // check whether the end of the cluster chain has been reached (if the next cluster is 0, 1 or 2 it is ignored since it must be corrupt - 0, 1 are not used and 2 is always the start of the root directory!)
    #endif
        {
            int iResult = UTFAT_SUCCESS;
            unsigned char ucFatCopies = 0;
            if (iDestroyClusters == REUSE_CLUSTERS) {                    // if the file's cluster is to be reused and only occupied one cluster, don't delete it
                break;
            }
            else {
    #if defined UTFAT16
                if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                    if (ucClusterEntry & 1) {
                        ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0xffff0000);
                    }
                    else {
                        ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0x0000ffff);
                    }
                }
                else {
                    ulSectorContent[ucClusterEntry] = 0;                 // delete the previous entry
                }
    #else
                ulSectorContent[ucClusterEntry] = 0;                     // delete the previous entry
    #endif
                if (ptr_utDisk->usDiskFlags & FSINFO_VALID) {
                    ptr_utDisk->utFileInfo.ulFreeClusterCount += ++ulRemovedClusters;
                    ptr_utDisk->usDiskFlags |= WRITEBACK_INFO_FLAG;      // mark that the info block information has changed
                }
            }
            while (ucFatCopies < ptr_utDisk->utFAT.ucNumberOfFATs) {
                while ((iResult = utCommitSectorData(ptr_utDisk, ulSectorContent, (ulClusterSector + (ucFatCopies * ptr_utDisk->utFAT.ulFatSize)))) == CARD_BUSY_WAIT) {}
                ucFatCopies++;
            }
            return iResult;
        }
        if (iDestroyClusters == REUSE_CLUSTERS) {                        // if the file's cluster is to be reused
    #if defined UTFAT16
            if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                if (ucClusterEntry & 1) {
                    ulSectorContent[ucClusterEntry/2] = ulSectorContent[ucClusterEntry/2] & ~LITTLE_LONG_WORD(0xffff0000);
                    ulSectorContent[ucClusterEntry/2] = (ulSectorContent[ucClusterEntry/2] | LITTLE_LONG_WORD(FAT16_CLUSTER_MASK << 16)); // mark last cluster in extension
                }
                else {
                    ulSectorContent[ucClusterEntry/2] = ulSectorContent[ucClusterEntry/2] & ~LITTLE_LONG_WORD(0x0000ffff);
                    ulSectorContent[ucClusterEntry/2] = (ulSectorContent[ucClusterEntry/2] | LITTLE_LONG_WORD(FAT16_CLUSTER_MASK)); // mark last cluster in extension
                }
            }
            else {
                ulSectorContent[ucClusterEntry] = CLUSTER_MASK;          // mark the end of the cluster chain
            }
    #else
            ulSectorContent[ucClusterEntry] = CLUSTER_MASK;              // mark the end of the cluster chain
    #endif
            iDestroyClusters = SIMPLE_DELETE;                            // from here on, destroy rest of chain
        }
        else {
    #if defined UTFAT16
            if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
                if (ucClusterEntry & 1) {
                    ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0xffff0000);
                }
                else {
                    ulSectorContent[ucClusterEntry/2] &= ~LITTLE_LONG_WORD(0x0000ffff);
                }
            }
            else {
                ulSectorContent[ucClusterEntry] = 0;                     // delete the previous entry
            }
    #else
            ulSectorContent[ucClusterEntry] = 0;                         // delete the previous entry
    #endif
            ulRemovedClusters++;
        }
    #if defined UTFAT16
        if (ptr_utDisk->usDiskFlags & DISK_FORMAT_FAT16) {
            ulNextClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulNextCluster >> 8));
            ucClusterEntry = (unsigned char)(ulNextCluster);
        }
        else {
            ulNextClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulNextCluster >> 7));
            ucClusterEntry = (unsigned char)(ulNextCluster & 0x7f);
        }
    #else
        ulNextClusterSector = (ptr_utDisk->utFAT.ulFAT_start + (ulNextCluster >> 7));
        ucClusterEntry = (unsigned char)(ulNextCluster & 0x7f);
    #endif
        if (ulNextClusterSector != ulClusterSector) {                    // moving to a new sector so update the FAT
            unsigned char ucFatCopies = 0;
            while (ucFatCopies < ptr_utDisk->utFAT.ucNumberOfFATs) {
                while (utCommitSectorData(ptr_utDisk, ulSectorContent, (ulClusterSector + (ucFatCopies * ptr_utDisk->utFAT.ulFatSize))) == CARD_BUSY_WAIT) {}
                ucFatCopies++;
            }
            while (utCommitSectorData(ptr_utDisk, ulSectorContent, ulClusterSector) == CARD_BUSY_WAIT) {}
            ulClusterSector = ulNextClusterSector;
            if ((utReadDiskSector(ptr_utDisk, ulClusterSector, ulSectorContent)) != UTFAT_SUCCESS) { // read a FAT sector containing the cluster information
                return UTFAT_DISK_READ_ERROR;
            }
        }
    }
    return UTFAT_SUCCESS;
}

static int fnDeleteFileContent(UTFILE *ptr_utFile, UTDISK *ptr_utDisk, int iDestroyClusters)
{
    int iResult;
    if (iDestroyClusters != SIMPLE_DELETE) {                             // the file is to be re-used so set length to zero
        fnSetFileInformation(ptr_utFile->ptr_utDirObject->ptrEntryStructure, 0); // set the file size to zero
    }
    #if defined UTFAT_LFN_READ && (defined UTFAT_LFN_DELETE || defined UTFAT_LFN_WRITE) // if long file names are in use, delete also possible long file name    
    if (iDestroyClusters != REUSE_CLUSTERS) {
        if ((iResult = fnDeleteLFN_entry(ptr_utFile)) != UTFAT_SUCCESS) {// delete the long file name as long as the file is not being truncated for reuse
            return iResult;                                              // return error code
        }
    }
    #endif
    #if defined UTFAT_SAFE_DELETE
    if (iDestroyClusters == SAFE_DELETE) {
        uMemset(ptr_utFile->ptr_utDirObject->ptrEntryStructure, 0, sizeof(DIR_ENTRY_STRUCTURE_FAT32)); // completely remove the directory entry content
    }
    #endif
    while (utCommitSector(ptr_utDisk, ptr_utDisk->ulPresentSector) == CARD_BUSY_WAIT) {} // commit the file entry change
    ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;
    if (ptr_utFile->private_file_location.ulSector >= ptr_utDisk->ulLogicalBaseAddress) { // don't delete cluster chain if there is none allocated
        iResult = fnDeleteClusterChain(ptr_utFile->private_file_location.ulSector, ptr_utDisk->ucDriveNumber, iDestroyClusters); // free up all clusters belonging to the file content
        if (iResult < 0) {
            return iResult;                                              // return error cause
        }
    }
    return fnCommitInfoChanges(ptr_utDisk);                              // update the info sector accordingly if it has changed
}

static int _utDeleteFile(const CHAR *ptrFilePath, UTDIRECTORY *ptrDirObject, int iSafeDelete)
{
    int iResult;
    UTFILE utFile;
    UTDISK *ptr_utDisk = &utDisks[ptrDirObject->ucDrive];
    utFile.ptr_utDirObject = ptrDirObject;
    ptrDirObject->usDirectoryFlags |= (UTDIR_TEST_REL_PATH);
    iResult = _utOpenFile(ptrFilePath, &utFile, UTFAT_OPEN_FOR_DELETE);
    if (iResult == UTFAT_SUCCESS) {                                      // directory has been located
        DISK_LOCATION *ptrDirContent = &utFile.private_disk_location;
        DIR_ENTRY_STRUCTURE_FAT32 *ptrDirEntry;
        uMemcpy(&utFile.private_file_location, &utFile.private_disk_location.directory_location, sizeof(utFile.private_disk_location.directory_location));
        do {
            if (fnLoadSector(ptr_utDisk, utFile.private_disk_location.directory_location.ulSector) != UTFAT_SUCCESS) {
                return UTFAT_DISK_READ_ERROR;
            }
            ptrDirEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
            ptrDirEntry += ptrDirContent->ucDirectoryEntry;              // move to the present entry
            if (ptrDirEntry->DIR_Name[0] == 0) {
                break;                                                   // directory is empty so allow delete
            }
            else if (ptrDirEntry->DIR_Name[0] != DIR_NAME_FREE) {
                return UTFAT_DIR_NOT_EMPTY;
            }
            if (fnNextDirectoryEntry(ptr_utDisk, ptrDirContent) == UTFAT_DIRECTORY_AREA_EXHAUSTED) { // move to the next entry
                return UTFAT_DIRECTORY_AREA_EXHAUSTED;
            }
        } while (1);
        ptrDirObject->usDirectoryFlags |= (UTDIR_TEST_REL_PATH | UTDIR_DIR_AS_FILE); // handle a directory as a file so that it can also be deleted if found
        iResult = _utOpenFile(ptrFilePath, &utFile, UTFAT_OPEN_FOR_DELETE);
        if (iResult == UTFAT_SUCCESS) {
            utFile.ptr_utDirObject->ptrEntryStructure->DIR_Name[0] = DIR_NAME_FREE; // delete the (short) file name
            return (fnDeleteFileContent(&utFile, ptr_utDisk, iSafeDelete));
        }
    }
    else if (iResult == UTFAT_PATH_IS_FILE) {                            // if file has been located
        if (utFile.ptr_utDirObject->ptrEntryStructure->DIR_Attr & DIR_ATTR_READ_ONLY) {
            return UTFAT_FILE_NOT_WRITEABLE;                             // the file is read-only so can not be deleted
        }
    #if defined UTFAT_SAFE_DELETE
        if (iSafeDelete == SAFE_DELETE) {                                // destroy the file content
            unsigned long ulLength = utFile.ulFileSize;                  // the content size
            unsigned short usWriteLength = 512;
            unsigned long ulEmptyBuffer[512/sizeof(unsigned long)];
            uMemset(ulEmptyBuffer, 0, sizeof(ulEmptyBuffer));            // empty buffer content
            utSeek(&utFile, 0, UTFAT_SEEK_SET);                          // ensure that the file pointer is set to the start fo the file
            while (ulLength != 0) {
                if (ulLength < 512) {
                    usWriteLength = (unsigned short)ulLength;
                }
                utWriteFile(&utFile, (unsigned char *)ulEmptyBuffer, usWriteLength);
                ulLength -= usWriteLength;
            }
        }
    #endif
        utFile.ptr_utDirObject->ptrEntryStructure->DIR_Name[0] = DIR_NAME_FREE; // delete the (short) file name
        return (fnDeleteFileContent(&utFile, ptr_utDisk, iSafeDelete));
    }
    return iResult;
}

extern int utDeleteFile(const CHAR *ptrFilePath, UTDIRECTORY *ptrDirObject)
{
    return _utDeleteFile(ptrFilePath, ptrDirObject, SIMPLE_DELETE);
}

    #if defined UTFAT_SAFE_DELETE
extern int utSafeDeleteFile(const CHAR *ptrFilePath, UTDIRECTORY *ptrDirObject)
{
    return _utDeleteFile(ptrFilePath, ptrDirObject, SAFE_DELETE);
}
    #endif

// Make a new directory - don't allow if a directory of the same name already exists
//
extern int utMakeDirectory(const CHAR *ptrDirPath, UTDIRECTORY *ptrDirObject)
{
    UTDISK *ptr_utDisk = &utDisks[ptrDirObject->ucDrive];
    OPEN_FILE_BLOCK openBlock;
    int iReturn;

    uMemset(&openBlock, 0, sizeof(openBlock));                           // initialise open file block

    if (ptr_utDisk->usDiskFlags & WRITE_PROTECTED_SD_CARD) {
        return UTFAT_DISK_WRITE_PROTECTED;
    }
    ptrDirObject->usDirectoryFlags |= UTDIR_SET_START;                   // set to start of lowest directory when file not found

    openBlock.iContinue = 0;
    iReturn = _fnHandlePath(&openBlock, ptrDirPath, ptrDirObject);       // handle the input path string
    if (openBlock.iContinue == 0) {                                      // if the path handling had an error or completed all work
        return iReturn;                                                  // return with code
    }

    if ((iReturn = _utOpenDirectory(&openBlock, ptrDirObject, 0)) != UTFAT_FILE_NOT_FOUND) { // check that the directory doesn't already exist
        if (UTFAT_SUCCESS == iReturn) {
            return UTFAT_DIRECTORY_EXISTS_ALREADY;                       // directory already exists
        }
        return iReturn;                                                  // another error
    }
    openBlock.ptr_utDisk = ptr_utDisk;
    uMemcpy(&openBlock.DirectoryEndLocation, &ptrDirObject->public_disk_location, sizeof(openBlock.DirectoryEndLocation));
    return (fnCreateFile(&openBlock, 0, openBlock.ptrLocalDirPath, 0));  // create a new directory
}

/// Reformat a disk that is presently formatted
//
static int utReFormat(const unsigned char ucDrive, const CHAR *cVolumeLabel, unsigned char ucFlags) // static and pass flags
{
    if (utDisks[ucDrive].usDiskFlags & WRITE_PROTECTED_SD_CARD) {
        return UTFAT_DISK_WRITE_PROTECTED;
    }
    iMemoryOperation[DISK_D] = 0;
    utDisks[ucDrive].usDiskFlags &= (HIGH_CAPACITY_SD_CARD);             // remove all flags apart from high capacity information
    utDisks[ucDrive].usDiskFlags |= DISK_UNFORMATTED;                    // consider unformatted from this point on
    #if defined UTFAT16
    if (ucFlags & UTFAT_FORMAT_16) {
        utDisks[ucDrive].usDiskFlags |= DISK_FORMAT_FAT16;               // FAT16 to be formatted rather than FAT32
        iMemoryState[DISK_D] = STATE_FORMATTING_DISK_2;                  // no extended record added to FAT16
    }
    else {
        iMemoryState[DISK_D] = STATE_FORMATTING_DISK_1;                  // start with an extended record when formatting to FAT32
    }
    #else
    iMemoryState[DISK_D] = STATE_FORMATTING_DISK_1;
    #endif
    if (ucFlags & UTFAT_FULL_FORMAT) {
        utDisks[ucDrive].usDiskFlags |= DISK_FORMAT_FULL;                // delete all data content as well as just FAT content
    }
    uMemcpy(utDisks[ucDrive].cVolumeLabel, cVolumeLabel, 11);
    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);                      // schedule the task to start formatting
    return UTFAT_SUCCESS;
}

// Format a disk that is presently not unformatted
//
extern int utFormat(const unsigned char ucDrive, const CHAR *cVolumeLabel, unsigned char ucFlags)
{
    if ((!(ucFlags & UTFAT_REFORMAT)) && (iMemoryState[DISK_D] != DISK_NOT_FORMATTED)) {
        return UTFAT_DISK_NOT_READY;
    }
    return utReFormat(ucDrive, cVolumeLabel, ucFlags);
}

// Truncate the file to the present file pointer location
// this usually makes the file smaller and cleans up no longer used clusters
//
extern int utTruncateFile(UTFILE *ptr_utFile)
{
    if (ptr_utFile->ulFileSize > ptr_utFile->ulFilePosition) {           // if the file pointer is at the end of the present file there is nothing to do
        if (!(ptr_utFile->ulFileMode & UTFAT_OPEN_FOR_WRITE)) {          // only allow truncation if the file is open for writing
            return UTFAT_FILE_NOT_WRITEABLE;
        }
        else {
            UTDISK *ptr_utDisk = &utDisks[ptr_utFile->ucDrive];
            DIR_ENTRY_STRUCTURE_FAT32 *ptrFileEntry = (DIR_ENTRY_STRUCTURE_FAT32 *)ptr_utDisk->ptrSectorData; // the directory entry in the sector buffer
            ptrFileEntry += ptr_utFile->private_disk_location.ucDirectoryEntry; // move to the file entry
            if (fnLoadSector(ptr_utDisk, ptr_utFile->private_disk_location.directory_location.ulSector) != UTFAT_SUCCESS) { // ensure that the directory sector is loaded
                return UTFAT_DISK_READ_ERROR;
            }
            // We set the new file size to be equal to the present file pointer location
            //
            ptr_utFile->ulFileSize = ptr_utFile->ulFilePosition;
            fnSetFileInformation(ptrFileEntry, ptr_utFile->ulFileSize);

            while (utCommitSector(ptr_utDisk, ptr_utFile->private_disk_location.directory_location.ulSector) == CARD_BUSY_WAIT) {} // force writeback to finalise the operation
            ptr_utDisk->usDiskFlags &= ~WRITEBACK_BUFFER_FLAG;           // the disk is up to date with the buffer


            // We need to free all clusters no longer needed by content after the new end of the file
            //
            if (ptr_utFile->public_file_location.ulSector >= ptr_utDisk->ulLogicalBaseAddress) { // don't delete cluster chain if there is none allocated
                return (fnDeleteClusterChain(ptr_utFile->public_file_location.ulSector, ptr_utDisk->ucDriveNumber, REUSE_CLUSTERS)); // free up all clusters belonging to the file content
            }
        }
    }
    return UTFAT_SUCCESS;
}
#endif


// This routine is called to count the free clusters and update the disk information
//
extern int utFreeClusters(unsigned char ucDisk, UTASK_TASK owner_task)
{
    if (cluster_task[DISK_D] != 0) {                                     // cluster count already in progress
        return MISSING_USER_TASK_REFERENCE;                              // the user must pass a task reference
    }
    ulActiveFreeClusterCount[DISK_D] = 0;
    ulClusterSectorCheck[DISK_D] = (utDisks[ucDisk].utFAT.ulFAT_start + (utDisks[ucDisk].utFAT.ulFatSize - 1));
    iMemoryOperation[DISK_D] |= _COUNTING_CLUSTERS;
    cluster_task[DISK_D] = owner_task;
    uTaskerStateChange(OWN_TASK, UTASKER_ACTIVATE);                      // start cluster counting operation
    return UTFAT_SUCCESS;
}

// Routine to check for a certain file extension (type of file)
//
extern int uMatchFileExtension(UTFILEINFO *ptrFileInfo, const CHAR *ptrExtension)
{
    int i = 0;
    CHAR *ptrName = ptrFileInfo->cFileName;
    while (i++ < MAX_UTFAT_FILE_NAME) {
        if (*ptrName++ == '.') {
            int iMatch = uStrEquiv(ptrName, ptrExtension);
            if (iMatch > 0) {
                if ((*(ptrName + iMatch) == 0) && (*(ptrExtension + iMatch) == 0)) {
                    return 0;                                            // successful match
                }
            }
            break;                                                       // match failed
        }
    }
    return -1;                                                           // file type doesn't match
}

// Low level access routines
//
extern int fnReadSector(unsigned char ucDisk, unsigned char *ptrBuffer, unsigned long ulSectorNumber)
{
    if (ptrBuffer == 0) {                                                // if a zero pointer is given force a load but don't copy to a buffer
        return (fnLoadSector(&utDisks[ucDisk], ulSectorNumber));
    }
    return (utReadDiskSector(&utDisks[ucDisk], ulSectorNumber, ptrBuffer));
}

#if defined UTFAT_WRITE                                                  // allow operation without write support

#if defined UTFAT_MULTIPLE_BLOCK_WRITE
// Prepare the write to sectors if there are capabilities to improve the following write speed
//
extern int fnPrepareBlockWrite(unsigned char ucDisk, unsigned long ulWriteBlocks, int iPreErase)
{
    if (ulBlockWriteLength == 0) {                                       // if a multi block write is not already in operations
    #if defined UTFAT_PRE_ERASE
        if ((iPreErase != 0) && (ulWriteBlocks > 1)) {                   // ignore if single block is to written or no pre-erase is requested
            unsigned char ucResult;
            int iActionResult;
            SET_SD_CS_LOW();
            while ((iActionResult = fnSendSD_command(ucAPP_CMD_CMD55, &ucResult, 0)) == CARD_BUSY_WAIT) {}
            while ((iActionResult = fnSendSD_command(fnCreateCommand(PRE_ERASE_BLOCKS_CMD23, ulWriteBlocks), &ucResult, 0)) == CARD_BUSY_WAIT) {}
            SET_SD_CS_HIGH();
        }
    #endif
        ulBlockWriteLength = ulWriteBlocks;                              // set the number of blocks to be written
    }
    else {                                                               // test abort
        //ulBlockWriteLength = 0x80000000;
    }
    return UTFAT_SUCCESS;
}
#endif

extern int fnWriteSector(unsigned char ucDisk, unsigned char *ptrBuffer, unsigned long ulSectorNumber)
{
    int iResult;
    UTDISK *ptr_utDisk = &utDisks[ucDisk];
    while ((iResult = utCommitSectorData(ptr_utDisk, ptrBuffer, ulSectorNumber)) == CARD_BUSY_WAIT) {}
    return iResult;
}
#endif

extern const UTDISK *fnGetDiskInfo(unsigned char ucDisk)
{
    return (&utDisks[ucDisk]);
}

#if defined HTTP_ROOT || defined FTP_ROOT
extern int utServer(UTDIRECTORY *ptr_utDirectory, unsigned long ulServerType)
{
    if (ptr_utDirectory == 0) {                                          // change setting
        usServerStates &= ~(unsigned short)(ulServerType >> 16);         // disable
        usServerResets |= ((unsigned short)ulServerType & ~usServerStates); // the servers that have just been enabled can be reset once
        usServerStates |= (unsigned short)(ulServerType);                // enable
    }
    else {
        if (utDisks[ptr_utDirectory->ucDrive].usDiskFlags & (DISK_NOT_PRESENT | DISK_TYPE_NOT_SUPPORTED | DISK_UNFORMATTED)) { // unusable disk states
            usServerResets = 0;
            ptr_utDirectory->usDirectoryFlags &= (UTDIR_ALLOCATED);
            return UTFAT_DISK_NOT_READY;
        }
        if (!(usServerStates & ulServerType)) {                          // the server type has no access rights so return an error
            return UTFAT_DISK_NOT_READY;
        }
        if (usServerResets & ulServerType) {                             // if a server has been reset its root has to be confirmed
            ptr_utDirectory->usDirectoryFlags &= (UTDIR_ALLOCATED);
            usServerResets &= ~ulServerType;                             // only once
        }
        if (!(ptr_utDirectory->usDirectoryFlags & UTDIR_VALID)) {        // if the disk is not known to be valid try to locate the root directory
            const CHAR *ptrRoot;
            if (ulServerType & UTFAT_HTTP_SERVER) {
    #if defined HTTP_ROOT
                ptrRoot = HTTP_ROOT;
    #endif
            }
            else if (ulServerType & UTFAT_FTP_SERVER) {
    #if defined FTP_ROOT
                ptrRoot = FTP_ROOT;
    #endif
            }
            else {
                return UTFAT_DISK_NOT_READY;                             // unknown server
            }
            if (utOpenDirectory(ptrRoot, ptr_utDirectory) < 0) {         // open the root directory of disk D on connection (if no disk or no directory this fails and the HTTP server falls back to other file systems)
                return UTFAT_DISK_NOT_READY;
            }
        }
    }
    return UTFAT_SUCCESS;
}
#endif

#if defined SDCARD_DETECT_INPUT_INTERRUPT
// Interrupt call-back on change in SD-card presence
//
static void sdcard_detection_change(void)
{
    fnInterruptMessage(OWN_TASK, E_SDCARD_DETECTION_CHANGE);             // send interrupt event to task so that it can respond accordingly
}

// SD-card detection interrupt configuration
//
static void fnPrepareDetectInterrupt(void)
{
    INTERRUPT_SETUP interrupt_setup;                                     // interrupt configuration parameters
    interrupt_setup.int_type = PORT_INTERRUPT;                           // identifier when configuring
    interrupt_setup.int_handler = sdcard_detection_change;               // handling function
    #if defined _HW_AVR32                                                // AVR32 specific
    interrupt_setup.int_port = SDCARD_DETECT_PORT;                       // the port used
    interrupt_setup.int_port_bits = SDCARD_DETECT_PIN;                   // the input connected
    interrupt_setup.int_priority = PRIORITY_GPIO;                        // port interrupt priority
    interrupt_setup.int_port_sense = (IRQ_BOTH_EDGES | IRQ_ENABLE_GLITCH_FILER); // interrupt on both edges with active glitch filter
    #elif defined _KINETIS || defined _LPC23XX || defined _LPC17XX       // {54} KINETIS {57} LPC17xx and LPC2xxx
    interrupt_setup.int_port = SDCARD_DETECT_PORT;                       // the port used
    interrupt_setup.int_port_bits = SDCARD_DETECT_PIN;                   // the input connected
    interrupt_setup.int_priority = PRIORITY_SDCARD_DETECT_PORT_INT;      // port interrupt priority
    interrupt_setup.int_port_sense = (IRQ_BOTH_EDGES | PULLUP_ON);       // interrupt on both edges
    #elif defined _M5223X                                                // {55} Coldfire V2
    interrupt_setup.int_port_bit = SDCARD_DETECT_PIN;                    // the IRQ input connected
    interrupt_setup.int_priority = PRIORITY_SDCARD_DETECT_PORT_INT;      // port interrupt priority
    interrupt_setup.int_port_sense = (IRQ_BOTH_EDGES);                   // interrupt on both edges
    #elif defined _STM32                                                 // {56} SMT32
    interrupt_setup.int_port = SDCARD_DETECT_PORT;                       // the port used
    interrupt_setup.int_port_bit = SDCARD_DETECT_PIN;                    // the IRQ input connected
    interrupt_setup.int_priority = PRIORITY_SDCARD_DETECT_PORT_INT;      // port interrupt priority
    if (SDCARD_DETECTION()) {
        interrupt_setup.int_port_sense = (IRQ_RISING_EDGE);              // interrupt on rising edge to detect card being removed
    }
    else {
        interrupt_setup.int_port_sense = (IRQ_FALLING_EDGE);             // interrupt on falling edge to detect card being inserted
    }
    #endif
    fnConfigureInterrupt((void *)&interrupt_setup);                      // configure test interrupt
}
#endif

#endif
